' Constructs the UI manager.  Context is an object consisting of references
' to any data you wish all nodes have access to.
function constructUI(app as Object) as Object
    ui = {
        app: app,

        screenStack: constructScreenStack(app),

        fonts: {},
        colors: {},
        arrows: {},

        fontRegistry    : CreateObject("roFontRegistry"),
        addFont         : addUIFont,
        setDefaultFont  : setDefaultUIFont,

        ' Methods for calculating menu dimensions.
        calculateMenuColumns    : calculateMenuColumns,

        resolveColor            : resolveUIColor,
        resolveFont             : resolveUIFont,

        ' Methods for constructing UI nodes.
        constructBaseUINode     : constructBaseUINode,
        constructBaseMenuNode   : constructBaseMenuNode,
        constructString         : constructStringUI,
        constructFileEntry      : constructFileEntryUI,
        constructSelector       : constructSelectorUI,
        constructRange          : constructRangeUI,
        constructAction         : constructActionUI,
        constructTopLevelMenu   : constructTopLevelMenuUI,
        constructVMenu          : constructVMenuUI,
        constructSMenu          : constructSMenuUI,

        ' Methods to generate layout items (which are processed by
        ' Screen).
        layoutRectangle : layoutRectangle,
        layoutText      : layoutText,
        layoutImage     : layoutImage,
        layoutBox       : layoutBox
    }

    return ui
end function

sub addUIFont(name as String, size as Integer, bold=false as Boolean)
    if invalid <> m.fonts[name] then
        print "Warning: Overriding existing font name: " + name
    end if

    m.fonts[name] = m.fontRegistry.getDefaultFont(size, bold, false)
end sub

sub setDefaultUIFont(name as string)
    m.fonts["default"] = m.fonts[name]
end sub

' This function calculates the widths of each column by iterating through all possible submenus, tracking their
' depth and then calculating the width of the label.
function calculateMenuColumns(root as Object) as Object
    if root.children.count() > 15 then
        maxRows = 15
    else if root.children.count() < 5 then
        maxRows = 5
    else
        maxRows = root.children.count()
    end if

    dimensions = {
        totalWidth: 600,
        maxRows: maxRows,
        x: 10
        y: 10
    }
    
    dimensions.columnLeftOffsets = []
    dimensions.columnLeftOffsets[0] = dimensions.x + 40
    dimensions.columnLeftOffsets[1] = dimensions.columnLeftOffsets[0] + 430

    dimensions.rowHeight = m.fonts["default"].GetOneLineHeight() + 4
    dimensions.totalHeight = dimensions.rowHeight * dimensions.maxRows

    return dimensions
end function

' Base UI node, allowing common properties to be inherited among all
' UI components.
function constructBaseUINode(params as Object) as Object
    o = {
        type: "Base",
        ui: m,
        app: m.app,
        shouldShowValue: shouldUIShowValue,
        calculateWidth: calculateBaseUIWidth
    }

    return RokuMfgMerge(o, params)
end function

function calculateBaseUIWidth(font as Object, basewidth as Integer) as Integer
    return 0
end function

' We also have a base menu node, which extends the base node with
' the capacity add items with parentage.
function constructBaseMenuNode(params as Object) as Object
    o = m.constructBaseUINode({
        type: "BaseMenu",
        addItem: addMenuItem,
        getSelected: getSelectedMenuItem,
        update: updateSelectedMenuItem,
        exit: exitMenu,
        selectedChild: Invalid,
        children: []
    })

    return RokuMfgMerge(o, params)
end function

sub addMenuItem(item as Object) as Object
    item.parent = m
    m.children.push(item)
end sub

function getSelectedMenuItem() as Dynamic
    if invalid = m.selectedChild then
        return invalid
    end if

    return m.children[m.selectedChild]
end function

function updateSelectedMenuItem(change as Integer)
    m.selectedChild = RokuMfgModulo(m.selectedChild + change, m.children.count())
end function

sub exitMenu()
    m.selectedChild = invalid
end sub

function shouldUIShowValue(ctx as Object, params as Object) as Boolean
    if m.displayMode <> "Never" then
        if params.selected OR (params.activePath and "SelectedOnly" <> m.displayMode) then        
            return true
        end if
    end if

    return false
end function

function resolveUIColor(color as Dynamic, default as String) as Dynamic
    typename = type(color)

    if typename.instr("String") <> -1 then
        colorname = color
        color = m.colors[color]

        if invalid = color then
            print "Unknown color: " + colorname + ", resorting to default: " + default
        end if

        return color
    else if typename.instr("Int") <> -1 then
        return color
    end if

    return m.colors[default]
end function

function resolveUIFont(font as Dynamic, default as String) as Dynamic
    typename = type(font)
    if typename.instr("String") <> -1 then
        fontname = font
        font = m.fonts[font]

        if invalid = font then
            print "Unknown font: " + fontname + ", resorting to default: " + default
        end if

        return font
    else if "roFont" = typename then
        return font
    end if

    return m.fonts[default]
end function

function layoutRectangle(params as Object) as Object
    defaults = {
        type: "rect",
        x: 0, y: 0, w: 0, h: 0
    }

    params.color = m.resolveColor(params.color, "gray")
    rect = RokuMfgMerge(defaults, params)
    return rect
end function

function layoutText(params as Object) as Object
    defaults = {
        type: "text",
        x: 0, y: 0, w: 0, h: 0
    }

    if -1 = type(params.text).instr("String") then
        print "ERROR: layoutText params missing 'text' attribute!"
        stop
    end if

    params.color = m.resolveColor(params.color, "white")
    params.font = m.resolveFont(params.font, "default")
    text = RokuMfgMerge(defaults, params)
    text.w = RokuMfgMax(text.w, params.font.getOneLineWidth(params.text, m.app.screen.width))
    text.h = RokuMfgMax(text.h, params.font.getOneLineHeight())

    return text
end function

function layoutImage(params as Object) as Object
    defaults = {
        type: "image",
        source: invalid,
        x: 0, y: 0, w: invalid, h: invalid,
        scaleX: 1.0,
        scaleY: 1.0
    }

    image = RokuMfgMerge(defaults, params)
    sourceType = type(image.source)

    if -1 <> sourceType.instr("String") or "roPath" = sourceType then
        image.source = createObject("roBitmap", image.source)
    else if "roBitmap" <> type(image.source) then
        print "ERROR: layoutImage expected either filepath or roBitmap object!"
        stop
    end if

    ' If w and/or h are specified, then we will calculate the scale factor for drawing.
    if invalid <> image.w then
        image.scaleX =  image.w / image.source.getWidth()
    end if

    image.w = image.source.getWidth() * image.scaleX

    if invalid <> image.h then
        image.scaleY = image.h / image.source.getHeight()
    end if

    image.h = image.source.getHeight() * image.scaleY

    return image
end function