REM ******************************************************
REM Copyright Roku 2011,2012,2013.
REM All Rights Reserved
REM ******************************************************

REM This file has no dependencies on other common files.
REM
REM Functions in this file:
REM     GetDeviceVersion
REM     GetDeviceVersionAA
REM     CheckDeviceVersion
REM     GetDeviceCaptionsMode
REM     GetDeviceESN
REM     IsHD
REM

'******************************************************
'Get our device version as single string
'******************************************************
Function GetDeviceVersion()
    if m.softwareVersion = invalid OR m.softwareVersion = "" then
        m.softwareVersion = CreateObject("roDeviceInfo").GetVersion()
    end if
    return m.softwareVersion
End Function

'******************************************************
'Get our device version as parsed associative array
'******************************************************
Function GetDeviceVersionAA()
    if m.softwareVersionAA = invalid then
        version = GetDeviceVersion()
        m.softwareVersionAA = {
            FullVersion: version
            Major: version.Mid(2, 1).ToInt()
            Minor: version.Mid(4, 2).ToInt()
            Build: version.Mid(7, 5).ToInt()
        }
    end if
    return m.softwareVersionAA
End Function

'******************************************************
'Compare current device firmware version vs version supplied with parameters
'
'Function returns
'    1 - if current version is greater than supplied
'    0 - if current version is equal to supplied
'   -1 - if current version is smaller than supplied
'******************************************************
Function CheckDeviceVersion(major as Dynamic, minor = invalid as Dynamic, build = invalid as Dynamic) as Integer
    version = GetDeviceVersionAA()
    if version.Major > major then
        return 1
    else if version.Major < major then
        return -1
    else
        if minor <> invalid then
            if version.Minor > minor then
                return 1
            else if version.Minor < minor then
                return -1
            else
                if build <> invalid then
                    if version.Build > build then
                        return 1
                    else if version.Build < build then
                        return -1
                    else
                        return 0
                    end if
                else
                    return 0
                end if
            end if
        else
            return 0
        end if
    end if
End Function

'******************************************************
'Get our device closed captions mode
'(only for fimware 5.3 or greater, returns invalid otherwise)
'******************************************************
Function GetDeviceCaptionsMode() as Dynamic
    if CheckDeviceVersion(5, 3) >= 0 then
        if m.deviceCaptionsMode = invalid then
            deviceInfo = CreateObject("roDeviceInfo")
            m.deviceCaptionsMode = deviceInfo.GetCaptionsMode()
        end if
        return m.deviceCaptionsMode
    else
        return invalid
    end if
End Function

'******************************************************
'Get our serial number
'******************************************************
Function GetDeviceESN()
    if m.serialNumber = invalid OR m.serialNumber = "" then
        m.serialNumber = CreateObject("roDeviceInfo").GetDeviceUniqueId()
    end if
    return m.serialNumber
End Function


'******************************************************
'Get our hardware model number
'******************************************************
Function GetHardwareModel()
    if m.hardwareModel = invalid OR m.hModel = "" then
        m.hardwareModel = CreateObject("roDeviceInfo").GetModel()
    end if
    return m.hardwareModel
End Function


'******************************************************
'Determine if the UI is displayed in SD or HD mode
'******************************************************
Function IsHD()
    di = CreateObject("roDeviceInfo")
    if di.GetDisplayMode() = "720p" then return true
    return false
End Function

'******************************************************
' @returns The maximum between a and b. Both inputs
'          should be the same type for this comparison
'          to work.
'******************************************************
Function max(a As Dynamic, b As Dynamic) As Dynamic
    if b > a then return b
    return a
End Function

'******************************************************
' @return The MD5 hash of the specified text
'******************************************************
Function getMD5Hash(text As String) As String
    digest = CreateObject("roEVPDigest")
    digest.Setup("md5")
    ba = CreateObject("roByteArray")
    ba.FromAsciiString(text)
    digest.Update(ba)
    return LCase(digest.Final())
End Function


'******************************************************
' @return The time in seconds since the epoch (1/1 1970).
'******************************************************
Function nowSecs() As Integer
    now = m.globalClock
    if now=invalid
        now = CreateObject("roDateTime")
        m.globalClock = now
    end if
    now.mark()
    return now.asSeconds()
End Function

'******************************************************
' Convert AA to JSON string
'******************************************************
Function AAToJSON(aa As Object) As String
    return AAToJSONHelper(aa, 0)
End Function

Function AAToJSONHelper(aa As Object, indent As Integer) As String
    result = ""
    if (aa <> invalid)
        result = result + chr(10)
        for index = 1 to indent step 1
            result = result + chr(9)    ' tab
        end for

        result = result + "{"
        leadingComma = false
        for each e in aa
            if (leadingComma)
                result = result + "," + chr(10)
                for index = 1 to indent step 1
                    result = result + chr(9)    ' tab
                end for
            else
                leadingComma = true
            end if

            REM - chr(34) = "
            result = result + chr(34) + e + chr(34) + ":"
            x = aa[e]
            if (x = invalid)
	        result = result + "null"
            else if (type(x) = "roAssociativeArray")
                result = result + AAToJSONHelper(x, indent + 1)
            else if (isint(x))
                result = result + itostr(x)
            else if (isfloat(x))
                result = result + Str(x).Trim()
            else if (isstr(x))
                result = result + chr(34) + x + chr(34)
            else if (type(x) = "roArray")
                result = result + "["
                leadingArrayComma = false
                for each item in x
                    if (leadingArrayComma)
                        result = result + "," + chr(10)
                        for index = 1 to indent step 1
                            result = result + chr(9)    ' tab
                        end for
                    else
                        leadingArrayComma = true
                    end if
                    result = result + AAToJSONHelper(item, indent + 1)
                end for
                result = result + "]"
            else if (type(x) = "roBoolean")
                if (x)
                    result = result + "true"
                else
                    result = result + "false"
                end if
            else
                result = result + "invalid type"
            end if
        next
        result = result + "}"
    end if

    return result
End Function

Function JSONEscape(s As String) As String
    result = ""
    length = Len(s)
    for index = 1 to length step 1
        oneChar = Mid(s, index, 1)
        oneCharCode = Asc(oneChar)
        if (oneCharCode < 32)
            REM - control character
            if (oneCharCode < 16)
                result = result + "\u000" + HexChar(oneCharCode)
            else
                result = result + "\u001" + HexChar(oneCharCode - 16)
            end if
        else if (oneCharCode = 34)
            REM - double quote
            result = result + "\" + oneChar
        else if (oneCharCode = 92)
            REM - slash
            result = result + "\\"
        else
            result = result + oneChar
        end if
    end for

    return result
End Function

Function HexChar(code As Integer) As String
    if (code < 10)
        return Chr(48 + code)
    else
        return Chr(65 + (code - 10))
    end if
End Function

function copyAA(source as Object) as Object
    dest = {}
    for each key in source
              dest.AddReplace(key, source.Lookup(key))
    next
    return dest
end function

'**********************************************************************
'  Checks if array contains given element value
'  Returns: index if element exist, -1 if no such element
'********************************************************************** 
Function IndexInArray(arr as Dynamic, elem as Dynamic) As Integer
    if arr <> invalid AND type(arr) = "roArray" then
        for i = 0 to arr.count() - 1
            if arr[i] = elem then
                return i
            end if
        end for
    end if
    return -1
End Function


