' Screen handles UI rendering.

function InitScreenModule(app as Object)

    ' Pick the best display size for the screen.
    devInfo = createObject("roDeviceInfo")
    display = invalid

    for each res in devInfo.GetSupportedGraphicsResolutions()
        if invalid = display or display.width < res.width then
            display = res
        end if
    end for

    if invalid = display then
        print "ERROR: No supported display resolutions found!"
        stop
    end if

    screen = createObject("roScreen", true, display.width, display.height)

    o = {
        app: app,
        impl: screen,
        port: createObject("roMessagePort"),

        ' Dimensions, used by layout-generating code.'
        width: screen.getWidth(),
        height: screen.getHeight(),
        visibleTop: 30,
        visibleLeft: 30,

        ' Layouts for each layer get placed in these arrays.
        mainItems : createLayer("main"),
        midItems  : createLayer("mid"),
        menuItems : createLayer("menu"),
        osdItems  : createLayer("osd"),
        frontItems: createLayer("front"),


        osds: {},

        createBarGraph: createBarGraph,

        clear: clearScreen,
        fill: fillScreen,
        swapBuffers: swapScreenBuffers,
        drawBase: drawBaseLayer,
        drawItem: drawItem,
        clearItems: clearScreenItems,

        addOSD: addOSD,
        prepareOSDs: prepareOSDs,
        updateOSDTimers: updateOSDTimers
    }

    o.impl.setPort(o.port)

    ' Calculate visible screen dimensions.
    o.visibleWidth  = o.width - (2 * o.visibleLeft)
    o.visibleHeight = o.height - (2 * o.visibleTop)
    o.visibleRight  = o.visibleLeft + o.visibleWidth
    o.visibleBottom = o.visibleTop + o.visibleHeight

    ' Since AAs can't enforce order without weirdness, we'll
    ' use a separate array.
    o.layers = [o.mainItems, o.midItems, o.menuItems, o.osdItems, o.frontItems]

    o.impl.setAlphaEnable(false)

    app.screen = o
end function

function createLayer(name as String) as Object
    o = {
        name: name,
        enable: true,
        items: [],
        append: appendToLayer,
        push: addItemToLayer,
        clear: clearLayer
    }
    return o
end function

sub appendToLayer(items as Object)
    m.items.append(items)
end sub
sub addItemToLayer(item as Object) 
    m.items.push(item)
end sub
sub clearLayer()
    m.items.clear()
end sub

sub clearScreen()
    m.fill("transparent")
end sub

'TODO: Move this to UI
sub fillScreen(color)
    if "String" = type(color) or "roString" = type(color) then
        colorname = color
        color = m.app.ui.colors[color]

        if invalid = color then
            print "rect: Unknown color: " + colorname + ", defaulting to transparent"
        end if
    end if

    if invalid = color then
        color = m.colors.gray
    end if

    m.impl.clear(color)
end sub

sub drawBaseLayer()
   'Set opaque background
    m.impl.Clear(&h00000000)

    for each layer in m.layers
        if layer.enable then
            for each item in layer.items
                m.drawItem(item, {x: 0, y: 0})
            end for
        end if
    end for

    m.swapBuffers()
end sub

sub swapScreenBuffers()
    m.impl.swapBuffers()
end sub

sub drawItem(item as Object, ctx as Object)
    if item.type = "image" then
        m.impl.setAlphaEnable(true)
        m.impl.drawScaledObject(item.x + ctx.x, item.y + ctx.y, item.scaleX, item.scaleY, item.source)
        m.impl.setAlphaEnable(false)
    else if "rect" = item.type then
        m.impl.drawRect(item.x + ctx.x, item.y + ctx.y, item.w, item.h, item.color)
    else if "text" = item.type then
        m.impl.drawText(item.text, item.x + ctx.x, item.y + ctx.y, item.color, item.font)
    else if "box" = item.type then
        m.impl.drawRect(item.x + ctx.x - item.padding, item.y + ctx.y - item.padding, item.getWidth() + (item.padding * 2), item.getHeight() + (item.padding * 2), item.color)
        for each subitem in item.content
            m.drawItem(subitem, {x: ctx.x + item.x, y: ctx.y + item.y})
        end for
    end if
end sub

sub clearScreenItems()
    m.mainItems.clear()
    m.midItems.clear()
    m.menuItems.clear()
    m.osdItems.clear()
    m.frontItems.clear()
end sub

function addOSD(ms as Integer, osd as Object)
    defaults = {
        name: invalid,
        timeout: 3000,
        display: {
            type: "text", 
            info: {
                text: "OSD Example",
                textAttrs: {
                    color: "white"
                },
                targetRect: {x: 200, y: 200}
            }
        }
        screen: m,
        delete: deleteOSD,
        onDelete: invalid
    }


    o = RokuMfgDeepMerge(defaults, osd)

    if invalid <> o.timeout then
        o.timeout = o.timeout + ms
    end if

    ' Generate an anonymous name in case one isn't specified.
    if invalid = o.name then
        i = 0

        while true
            name = "anonymous" + RokuMfgStrCast(i)
            if invalid = m.osds[name] then
                o.name = name
                exit while
            end if
        end while
    end if

    m.osds[o.name] = o
    return o
end function

sub prepareOSDs()
    m.osdItems.clear()

    for each key in m.osds
        osd = m.osds[key]

        if invalid <> osd then
            if "roAssociativeArray" = type(osd.display) then
                m.osdItems.push(osd.display)
            else if "roArray" = type(osd.display) then
                m.osdItems.append(osd.display)
            end if
        end if
    end for
end sub

' Walk through every OSD 
function updateOSDTimers(ms as Integer, nextTimeout) 
    for each key in m.osds
        osd = m.osds[key]

        if invalid <> osd.timeout then
            osd.timeout = osd.timeout - ms

            if 0 >= osd.timeout then
                m.osds[key].delete()
                osd = invalid
            else
                if nextTimeout = 0 or nextTimeout > osd.timeout then
                    nextTimeout = osd.timeout
                end if
            end if
        end if
    end for

    return nextTimeout
end function

sub deleteOSD()
    if invalid <> m.onDelete then
        m.onDelete()
    end if

    m.screen.osds.delete(m.name)
end sub

function createBarGraph(params as Object) as Object
    defaults = {
        label: "Bar Graph",
        width: 500,
        height: 100,
        background: CreateObject("roBitmap", "pkg:/images/OSD-back2.jpg"),

        data: {
            value: 50,
            min: 0,
            max: 100,
            show: true
        },

        bar: {
            width: 0,
            height: 0,
            color: "red",
            background: "darkGray"
        }
    }

    defaults.x = (m.width - defaults.width) / 2
    defaults.y = (m.height - defaults.height - 50)
    defaults.bar.width = defaults.width - 40
    defaults.bar.height = defaults.height / 3
    defaults.bar.x = defaults.x + 20
    defaults.bar.y = defaults.y + 10

    bar = RokuMfgDeepMerge(defaults, params)

    value = RokuMfgMax(RokuMfgMin(bar.data.value, bar.data.max), bar.data.min)
    percent = Int(bar.bar.width * ((value - bar.data.min) / (bar.data.max - bar.data.min)))
    text = bar.label

    if bar.data.show then
        text = text + ": " + RokuMfgStrCast(value)
    end if

    textWidth = m.app.ui.fonts["default"].GetOneLineWidth(text, bar.width)
    textX = bar.x + ((bar.width - textWidth) / 2)
    textY = bar.bar.y + bar.bar.height + 10

    output = []

    ' Create background of osd'
    output.push(m.app.ui.layoutImage({
        type: "image",
        source: bar.background,
        x: bar.x,
        y: bar.y,
        w: bar.width,
        h: bar.height
    }))

    ' Draw the bar background
    output.push(m.app.ui.layoutRectangle({
        color: bar.bar.background
        x: bar.bar.x,
        y: bar.bar.y,
        w: bar.bar.width,
        h: bar.bar.height
    }))

    ' Draw the bar foreground
    output.push(m.app.ui.layoutRectangle({
        color: bar.bar.color,
        x: bar.bar.x,
        y: bar.bar.y,
        w: percent,
        h: bar.bar.height
    }))

    ' Draw the label
    output.push(m.app.ui.layoutText({
        type: "text",
        text: text,
        x: textX,
        y: textY
    }))

    return output
end function