Library "Roku_MFG.brs"

function getApp()
    globals = getGlobalAA()

    if invalid = globals.app then
        globals.app = initApplication()
    end if

    return globals.app
end function

function initApplication() as object
    o = {
        runlevel: getGlobalAA().runlevel,
        mfg: RokuMfg(),
        tv: RokuMfg().legacy,
        timer: createObject("roTimespan"),
        rawMessagePort: createObject("roMessagePort"),
        keyFilter: RokuMfgKeyFilterModule(),
        run: runApplication,
        wait: waitForApplicationEvent,

        ' Wrappers for screenStack to make it less tedious.'
        registerScreen: registerApplicationScreen,
        currentScreen: currentApplicationScreen,
        pushScreen: pushApplicationScreen,
        exitScreen: exitApplicationScreen,
        setScreen: setApplicationScreen,

        screenHandler: applicationScreenHandler,
        setInternalPattern: applicationSetInternalPattern

        ' tables to store menu categories and their respective fields
        tables: {}
    }

    o.tables["PQMenuItems"] = []

    o.tv.setMessagePort(o.rawMessagePort)
    o.tv.enableRawKeys()
    o.tv.enableKeyReleaseEvents(1)
    RokuMfgCheckLegacyResponse("savePQOnExit", o.tv.SavePQOnExit(1))

    ' Initialize base modules first.
    initScreenModule(o)
    initDataModule(o)
    o.usb = RokuMfgUSBModule({callback: handleUSBEvent})
    initUpdateModule(o)
    initUIModule(o)
    initKeycodes(o)
    initVideoModule(o)
    initTunerModule(o)

    ' Initialize the serial module and install serial module key filter.
    initSerialModule(o)

    if "Sigma SX7" = o.data.getValue("ChipType") then
        RokuMfgCheckLegacyResponse("forceGammaWBOnGraphics(1)", o.tv.forceGammaWBOnGraphics(1))
    end if

    o.keyFilter.install(applicationKeyFilter, o)

    getGlobalAA().app = o

    return o
end function

' The main application run loop manages two things:
'   1. Passing key events to the key filter
'   2. Updating OSD timeouts (where applicable).
function runApplication()
    m.timer.mark()
    msg = invalid
    timeout = 0

    m.setScreen("NoOverlay")
    m.pushScreen("Main")

    'Playback test video in background for PQ tuning
    m.video.showProgress(true, false)
    m.video.playFile(m.usb, {
        file_path: "videos/test_video.mp4",
        volume: "pkg:",
        loop: true
    })

    while true
       if type(msg) = "roManufacturingEvent" then
            info = m.mfg.parseEvent(msg)
            if info.type = "rawkeyevent" then
                if "false" = m.data.getValue("Loading") then
                    event = info.data
                    event.rawKeyValue = m.keycodes.simplifyNavigation(event.rawKeyValue)
                    m.keyFilter.handle(event)
                else
                    print "Keypresses blocked for loading"
                end if
            else if info.type = "pqapply" then
                m.data.pq.handle(info.data)
            else if info.type = "drmodechanged" then
                if "service" = m.runlevel then
                    m.data.get("HDRState").set(info.data.drmode)
                end if
            else
                print "Unhandled roManufacturingEvent: " + info.type
            end if
        else if type(msg) = "roVideoPlayerEvent" then
            m.video.handleEvent(msg)
        else
            m.usb.check()
        end if

        ' Check for serial input and update the command buffer.
        ' Only check if we have a valid impl
        if invalid <> m.serial and invalid <> m.serial.impl then
            m.serial.check()
        end if

        ' Updates the remaining timers on all OSDs and returns
        ' the number of milliseconds to wait for the next timeout.
        timeout = m.screen.updateOSDTimers(m.timer.totalMilliseconds(), 0)
        m.timer.mark()

        m.currentScreen().prepare()
        m.screen.prepareOSDs()
        m.screen.drawBase()

        timeout = m.screen.updateOSDTimers(m.timer.totalMilliseconds(), timeout)
        m.timer.mark()

        msg = m.wait(40)
    end while
end function

function waitForApplicationEvent(timeout as Integer) as Dynamic
    ev = m.mfg.checkEvent()
    if ev = invalid then
      ev = wait(timeout, m.rawMessagePort)
    end if
    return ev
end function

' Keys that should be handled at any time (provided a filter isn't actively stopping them from reaching this one)
' should go here.'
function applicationKeyFilter(context as Object, keycode as Integer)
    key = context.keycodes.lookupName(keycode)
    print "applicationKeyFilter: " + key

    ' Roku Home key always exits
    if "ROKU_HOME" = key then
        END
    else if "ROKU_MUTE" = key then
        context.data.get("Mute").update(1)
    else if context.keycodes.isVolumeKey(keycode) then
        volume = context.tv.getAudioVolume()
        if RokuMfgCheckLegacyResponse("getAudioVolume", volume) then
            ' Change always gets +/- 1
            change = context.keycodes.isVolumeKey(keycode)
            if "CHF" = Left(key, 3) then
                ' The factory remote adjusts the volume by 15
                change = 15 * change
            end if

            change = volume.value + change
            context.data.get("Volume").set(change)
            context.screen.addOSD(context.timer.totalMilliseconds(), {
                name: "VolumeBar",
                timeout: 3000,
                display: context.screen.createBarGraph({label: "Volume", data: {value: change, min: 0, max: 100}})
            })
        end if
    else if "ROKU_SPLAT" = key then
        if "NoOverlay" <> context.currentScreen().name then
            hideMenu()
        end if
    else if "CH+" = Right(key, 3) then
        context.tv.incrementChannel()
        context.mfg.legacy.getChannelInfoByIndex(-1)
    else if "CH-" = Right(key, 3) then
        context.tv.decrementChannel()
        context.mfg.legacy.getChannelInfoByIndex(-1)
    else if ("0" <= Right(key, 1) and "9" >= Right(key, 1)) then
        context.keyFilter.install(context.tuner.keyFilter, context.tuner)
        return context.tuner.keyFilter(context.tuner, keycode)
    else
        return keycode
    end if

    return invalid
end function

sub applicationScreenHandler(screen as String)
    current = m.currentScreen().name
    if "NoOverlay" <> current then
        m.exitScreen()
    end if

    if screen <> current then
        m.pushScreen(screen)
    end if
end sub

sub registerApplicationScreen(screen as Object)
    m.ui.screenStack.registerScreen(screen)
end sub

function currentApplicationScreen()
    screen = m.ui.screenStack.getCurrent()
    if invalid = screen
        return {name: "<NONE>"}
    end if

    return screen
end function

sub setApplicationScreen(screen as Dynamic)
    m.ui.screenStack.set(screen)
    print "Screen set to: " m.currentScreen().name
end sub

sub pushApplicationScreen(screen as Dynamic)
    m.ui.screenStack.push(screen)
    print "Current screen is now: " + m.currentScreen().name
end sub

sub exitApplicationScreen()
    m.ui.screenStack.pop()
    print "Current screen is now: " + m.currentScreen().name
end sub

function  applicationSetInternalPattern(enable as Boolean, params={} as Object) as Boolean
    ' Defaults parameters
    o = {
        useRGB:   false,
        useIRE:   false,
        useColor: false,
        red:      0,
        green:    0,
        blue:     0,
        ire:      0,
        color:    "red"
    }

    o = RokuMfgMerge(o, params)

    app = getApp()
    layer = app.data.get("TestPatternLayer")
    if false = enable or layer.changed then
        if layer.changed then
            print "Clearing patterns because layer changed"
        end if

        ' Disable the pattern for both video and graphics plane
        prev = layer.getValue()
        layer.set("Graphics")
        RokuMfgCheckLegacyResponse("setTestPatternRGBColor", app.tv.setTestPatternRGBColor({
            red: 0,
            green: 0,
            blue: 0,
            enable: 0
        }))

        layer.set("Video")
        RokuMfgCheckLegacyResponse("setTestPatternColor", app.tv.setTestPatternColor("", 0))
        layer.set(prev)
        layer.changed = false

        if false = enable then
            return true
        end if
    end if

    print "Test pattern applied to: " + layer.getValue()
    if true = o.useRGB then
        ret = app.tv.setTestPatternRGBColor({
            red: o.red,
            green: o.green,
            blue: o.blue,
            enable: 1
        })
        check_function = "setTestPatternRGBColor"
    else if true = o.useIRE then
        ret = app.tv.setTestPatternIRE(1, o.ire)
        check_function = "setTestPatternIRE"
    else if true = o.useColor then
        ret = app.tv.setTestPatternColor(o.color, 1)
        check_function = "setTestPatternColor"
    else
        print "applicationSetInternalPattern: No pattern is drawn"
        return false
    end if

    if RokuMfgCheckLegacyResponse(check_function, ret) then
        return true
    else
        return false
    end if
end function