'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
'////
'//// FILE
'////    LogoRain.brs
'////    Copyright (c) 2018, 2019 Roku, Inc.  All rights reserved.
'////
'//// APPLICATION
'////     Main application BrightScript file for "Logo Rain" screensaver.
'////
'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


Sub RunLogoRain( params As Object )

' DESCRIPTION
'     Contains start-up code and the main animation loop for the Logo Rain screensaver.
'
'     This function is defined primarily so a more complex screensaver (like City Stroll: Movie
'     Magic) can embed a call to this lighter-weight animation in the case of the screensaver 
'     being invoked while video playback is paused. 
'
' PARAMETERS
'     params - a list of parameters passed by the caller. (Currently not used.)
'
' RETURNS
'     Void.

    g = GetGlobalAA()

    '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
    ''''                               Testing Related Options                                 ''''
    '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  
    ' Optionally force a particular vendor/model for testing vendor specific logo/animation  
    ' attributes. NOTE: MAKE SURE THESE ARE EMPTY STRINGS BEFORE RELEASE OUT INTO THE WILD!   
    '
    g.forceVendor = ""
    g.forceModel  = ""
 
    ' Optionally force a particular UI resolution for testing SD/HD/FHD logo sizing logic.
    ' attributes. NOTE: MAKE SURE THIS IS AN EMPTY STRING BEFORE RELEASE OUT INTO THE WILD!   
    '
    g.forceUIRes  = ""
   
    ' Flag to indicate whether screensaver was started in testing mode or not. Primarily 
    ' used to suppress logging messages regarding Easter-Egg sound playback.
    '
    g.startedFromMain = params.DoesExist( "startedFromMain" )
    
    
    '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
    ''''                   Setup Required Items for Later Sections to Work                     ''''
    '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
    
    ' Create some useful default colors to use as fallbacks in this screen-saver.
    g.whiteColor     = &hC8C8C8FF
    g.blackColor     = &h010101FF
    g.rokuPurple     = &hA94AF0FF 'Official Roku Purple: &h662D91FF (102,45,145) x 1.66
    g.rokuDarkPurple = &h100815FF 'Dark Background verson of official Roku Purple
   
    ' Create a device information object that we can use to customize the behavior of this
    ' screensaver based on the vendor / platform.
    '
    g.devInfo = CreateObject( "roDeviceInfo" )

    ' Determine whether the vendor for this device needs its logo forced to the default
    ' logo. This is an emergency fall-back for when the logo included in the custom 
    ' package is incorrect for some reason.
    ' 
    g.forceDefaultLogo = ShouldForceDefaultLogo( params.defaultLogoVendors )
    
    ' Animations are tuned for 1080p displays with 60Hz refresh rate. The actual display
    ' for this device may be running at another rate. So calculate an animation speed
    ' multiplier to keep the animation rate looking the same regardless of refresh rate.
    '
    CalcAnimationRateInfo()
 
     
    '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
    ''''                     Create JSON Configurable Animation Attributes                     ''''
    '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
    
    ' NOTE: Values here may be overridden below by calls to LoadAnimAttr(), which reads optional 
    ' xxxx_attr.json files specifying fallback and vendor specific attribute values.
    
    ' Default to creating a double-buffered main screen. Using double-buffering will prevent 
    ' visible tearing as the logo animates. NOTE: If the system has insufficient memory for
    ' double-buffering, the code below will fall back to single buffered display.
    '
    g.doubleBufferScreen = true

    ' How many frames should be drawn before a garbage collection is performed. If set to zero,
    ' no per-frame-count garbage collection is performed.
    '
    g.garbageCollectFrameCount = 0
    
    ' Set the default maximum number of logos that can appear on-screen at any given time.
    g.maxLogos = 15
   
    ' By default display a slow white falling logo.
    g.slowWhiteLogo = false

    ' By default let logos draw at fractional pixel positions in Y only. Doing so prevents
    ' vertical edges in the logo from flickering at certain sizes. Allowing fractional Y
    ' positions produces smoother looking animation, but may cause some flickering on
    ' horizontal edges in the logo depending on logo size. 
    '
    ' Also note that flickering may be less apparent on tinted logos than on white logos, since
    ' tinted logos tend to be darker than pure white ones Thus, it may make sense to enable or 
    ' disable fractional logo positioning on a per-brand basis depending on a brand's logo tint.
    '
    g.fracLogoXPos = false
    g.fracLogoYPos = false
    
    ' Default to allowing logos to fall partially off the screen. For some low-count logos
    ' this can be set to ensure that the logo is always entirely visible.
    '
    g.fullyVisibleLogos = false
    
    ' Set the default background color to use.
    g.bkgdColor = g.rokuDarkPurple

    ' Set the default logo tint colors to use.
    g.logoColor1 = g.rokuPurple
    g.logoColor2 = 0
    g.logoColor3 = 0

    ' Set the default minimum and maximum brightness tint values for falling logos.
    g.logoMinBrightness = 0.20
    g.logoMaxBrightness = 0.80

    ' Set the default brightness gamma curve exponent. Note that values > 1 will favor 
    ' brightness values at the low end of the range, while values > 1 will favor 
    ' brightness values at the high end of the range.
    '
    g.logoBrightnessGamma = 0.9   

    ' Set the default size range permitted for falling logos, and select FHD by default.
    g.logoMinWidthSD  = 11
    g.logoMaxWidthSD  = 128
    g.logoMinWidthHD  = 22
    g.logoMaxWidthHD  = 256
    g.logoMinWidthFHD = 33
    g.logoMaxWidthFHD = 384
    g.logoSizeInUse   = "FHD"

    ' Set the default size gamma curve exponent. Note that values > 1 will favor logos at 
    ' the low end of the range, while values > 1 will favor logos at the high end of the
    ' range.
    '
    g.logoSizeGamma = 1.5

    ' Set the default speed range permitted for falling logos.
    g.logoMinSpeed = 0.5
    g.logoMaxSpeed = 2.0

    ' Set the default speed gamma curve exponent. Note that values > 1 will favor speeds at 
    ' the low end of the range, while values > 1 will favor speeds at the high end of the
    ' range.
    '
    g.logoSpeedGamma = 1.0
     
    g.logoMinRotRate    = 0
    g.logoMaxRotRate    = 0
    g.logoRotGamma      = 1.0
    g.logoRotDistScalar = 0.0
    g.logoRotMuteScalar = 0.8
 
    ' Easter-Egg sound to be played randomly but infrequently. If g.soundFile is set to 
    ' "" then no sound will be played ever.
    '
    g.soundFile = "" ' To enable by default, replace with: "pkg:/sounds/Tone1.wav"
    
    ' Set up various Easter-Egg sound playback parameters.
    g.soundMinVolume   = 10     ' Minimum sound volume based on logo size.
    g.soundMaxVolume   = 25     ' Maximum sound volume based on logo size.
    g.soundVolumeGamma = 0.75   ' Sound volume gamma curve exponent.
    g.soundMinPeriod   = 180*60 ' Minimum number of seconds between sound play.
    g.soundMaxPeriod   = 300*60 ' Maximum number of seconds between sound play.
    g.soundLikelyPct   = 0.42   ' Percent likelyhood of sound playing at each period.
    
    ' Actual sound play object and flags indicating if the sound is playing or not, and
    ' whether sound play should be forced regardless of time (for testing sound on-demand.)
    '
    g.sound            = invalid
    g.isSoundPlaying   = false
    g.forcePlaySound   = false
     
     
    '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
    ''''                         Get JSON Specified Animation Attributes                       ''''
    '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''

    ' List of animation attribute names we support in JSON files and in config-service overrides.
    g.attrNames = [ 
        "doubleBufferScreen",
        "garbageCollectFrameCount",
        "maxLogos",
        "slowWhiteLogo",
        "fullyVisibleLogos",
        "fracLogoXPos",
        "fracLogoYPos",
        "bkgdColor",
        "logoColor1",
        "logoColor2",
        "logoColor3",
        "logoMinBrightness",
        "logoMaxBrightness",
        "logoBrightnessGamma",
        "logoMinWidthSD",
        "logoMaxWidthSD",
        "logoMinWidthHD",
        "logoMaxWidthHD",
        "logoMinWidthFHD",
        "logoMaxWidthFHD",
        "logoSizeGamma",
        "logoMinSpeed",
        "logoMaxSpeed",
        "logoSpeedGamma",
        "soundFile",
        "soundMinVolume",
        "soundMaxVolume",
        "soundVolumeGamma",
        "soundMinPeriod",
        "soundMaxPeriod",
        "soundLikelyPct"
    ]
    
    ' Flag array to indicate which animation attributes have been read from JSON files or the 
    ' config-service for logging to console. Initially "" means attributes are unchanged. But
    ' if override values are read from JSON files or config-service overrides, vendor overrides
    ' will be marked with a '*', model overrides with a '+', and config-service overrides with 
    ' '!', '!*' or '!+'.
    '
    g.attrChanged = {} 
    for each attrName in g.attrNames
        g.attrChanged[attrName] = ""
    end for
    
    ' Read any JSON animation attribute files that may exist to tune the animation. Note that
    ' we start with fallback attributes first, followed by vendor overrides. This way each more 
    ' specific attribute file builds on the previous one.
    '
    LoadAnimAttrs( "default" )
    LoadAnimAttrs( "vendor" ) 

    ' Finally, fold in any config-service overrides that may have been set on the fly. 
    LoadCfgSvcAnimAttrs() 
 
    ' Summarize the final animation attributes for debugging purposes.
    SummarizeAnimAttrs()
    
 
    '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
    ''''                            Create and Init Main UI Objects                            ''''
    '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
    ' Start with no valid screen object.
    g.screen = invalid
    
    ' If the user wants to force the UI to FHD resolution, try to create a (possibly
    ' double-buffered) screen with FHD dimensions.
    '
    if g.forceUIRes = "FHD" 
          
        g.screen = CreateObject( "roScreen", g.doubleBufferScreen, 1920, 1080 )    
        if g.screen <> invalid then print "UI Resolution:          Forced FHD"
    
    ' If the user wants to force the UI to HD resolution, try to create a (possibly
    ' double-buffered) screen with HD dimensions.
    '
    else if g.forceUIRes = "HD"
    
        g.screen = CreateObject( "roScreen", g.doubleBufferScreen, 1280, 720 )  
        if g.screen <> invalid then print "UI Resolution:          Forced HD"
    
    end if
    
    ' If no particular UI resolution was forced, or the forced resolution failed, create a
    ' (possibly double-buffered) screen at the native UI resolution for the current display.
    '  
    if g.screen = invalid
    
        g.screen = CreateObject( "roScreen", g.doubleBufferScreen )   
        if g.screen <> invalid then print "UI Resolution:          Native for Display"
 
    end if  
    
    ' If any of the previous screen resolutions could not be created for some reason, fall
    ' back to creating a single buffered screen at the native UI resolution for the current
    ' display.
    '
    if g.screen = invalid 
        print "UI Resolution:       Single-Buffered Native Fallback"
        g.screen = CreateObject( "roScreen", false )
        g.doubleBufferScreen = false
    end if
    
    ' Enable alpha blending so that see-through parts of logos will actually be see-through.
    g.screen.SetAlphaEnable( true )

    ' Save the screen dimensions for later use.
    g.screenWidth  = g.screen.GetWidth()
    g.screenHeight = g.screen.GetHeight()

    ' Based on the UI screen dimensions, select whether we're going to use HD or FHD size
    ' limits logos that fall down the screen.
    '
    g.logoSizeRange = "HD"
    if g.screenHeight > 720 then g.logoSizeRange = "FHD"
    
    print ""
    print "Screen Dimensions:      " + g.screenWidth.ToStr() + " x " + g.screenHeight.ToStr()
    print "Double-Buffered:        " + g.doubleBufferScreen.ToStr()
    print "Logo Size Range:        " + g.logoSizeRange
    print ""
    
    ' Establish a message port for the screen so that we can receive messages when the user
    ' presses a button on the remote.
    '
    g.screenMsgPort = CreateObject( "roMessagePort" )
    g.screen.SetMessagePort( g.screenMsgPort )
     
    ' If an easter-egg sound file was specified, load it up.
    if g.soundFile <> ""
        g.sound = CreateObject( "roAudioResource", g.soundFile )
    end if
        
        
    '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
    ''''                              Set-Up Animation Control Data                            ''''
    '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''

    ' Load the vendor specific logo for this device (if any.) If no vendor specific 
    ' logo exists, the generic "Roku" logo will be used instead.
    '
    'LoadLogoBitmap( "images/brand_images" )
    LoadLogoBitmap()

    ' Create the array to hold instance info for each dropping Roku Logo, and set 
    ' the maximum number of dropping logos we'll all at one time on the screen.
    '
    g.logoArray = []

    'Establish a number of "lanes" down which to drop logos to prevent clumping.
    g.laneArray   = [ 0, 2, 4, 1, 3 ]
    g.laneCount   = g.laneArray.Count()
    g.currLaneIdx = 0
    g.laneGutter  = 50
    g.laneWidth   = (g.screenWidth + 2*g.laneGutter) / g.laneCount
    
    ' Used to temporarily limit the max logo size to something less than the slow
    ' dropping white logo if it's present on screen.
    '
    g.logoScalarCeiling = 1.00

    ' Set the first logo to drop immediately, and define a delay time to use between
    ' subsequent new logo drops.
    '
    g.nextLogoDropTime  = UpTime(0)
    g.nextLogoDropDelay = 1.5

    ' Set the time at which to play the initial easter-egg sound.
    soundPeriodRange = g.SoundMaxPeriod - g.soundMinPeriod
    g.nextPlaySoundTime = UpTime(0) + Rnd(0)*soundPeriodRange + g.soundMinPeriod
     
    ' Initally, the array will be sorted each time a new dropping logo is added. 
    ' After that, we'll only need to re-sort the array when a logo falls off the 
    ' bottom of the screen and is recycled.
    '
    mustSort = false


    '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
    ''''                                  Main Animation Loop                                  ''''
    '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''

    frameCount = 0
    
    ' Animate falling Roku Logos until the user presses a button on the remote or a keyboard key.
    while true

        ' Determine the current system time.
        frameStartTime = UpTime(0)

        ' Drop a new logo (if any.)
        DropNewLogo( frameStartTime )
        
        ' At this point we're ready to draw. So start by clearing the screen (or back buffer in
        ' a double-buffered situation) to the background color.
        '
        g.screen.Clear( g.bkgdColor )

        ' Next, for each logo instance in the array of active logos...
        for each logo in g.logoArray

            ' Draw the logo to the display buffer.
            DrawLogoInstance( frameStartTime, logo )
                        
            ' After drawing it, update the position of the current logo instance. If the logo 
            ' fell off the bottom of the screen and got recycled, flag that the array must be 
            ' re-sorted to reestablish the correct drawing order.
            '
            if UpdateLogoInstance( logo ) then mustSort = true

        end for

        ' Once all the logos have been drawn, swap buffers to update the display. Note that in
        ' non-double-buffered situations, this will do nothing.
        '
        g.screen.SwapBuffers()

        ' Once the display has been updated, if the logo array needs to be re-sorted...
        if mustSort

            ' Every now an then, convert a recycled logo to a slow moving white logo (if enabled.)
            MakeWhiteLogo()
 
            ' Re-sort the array to reestablish the correct drawing order for the logos, and
            ' clear the re-sort flag.
            '
            g.logoArray.SortBy( "width" )
            mustSort = false
 
        end if

        ' See if a remote button or keyboard key was pressed. 
        '
        ' NOTE: If this screensaver is auto-launched as an actual screensaver, any keypress 
        '       made by the user is intercepted by the firmware and causes the screensaver 
        '       to exit. Consequently, this code will be executed only when the screensaver
        '       is compiled as a channel store app for testing purposes.
        '
        msg = g.screenMsgPort.GetMessage()
        if Type(msg) = "roUniversalControlEvent"
            if msg.IsPress()
            
                ' If the info (*) button was pressed, force a sound play. For all other key 
                ' or remote control button presses, exit the screensaver.
                '
                if msg.GetKey() = 10
                    g.forcePlaySound = true
                else
                    exit while
                end if
        
            end if
        end if
    
        ' Play the "easter-egg" sound if one is defined and it's the proper time.
        PlaySoundIfNeeded( frameStartTime )

        ' If enabled, run the garbage collector every fixed number of frames. We do this to keep 
        ' the animation smooth by preventing garbage from piling up and causing long duration 
        ' collection periods. (Note that this is helpful preventing animation hiccups in single-
        ' buffered situations. You won't notice much of a change in double-buffered situations.)
        '
        frameCount++
        if g.garbageCollectFrameCount > 0 
            if frameCount Mod g.garbageCollectFrameCount = 0 
                RunGarbageCollector()
            end if
        end if
        
        ' If we're running in single-buffered mode...
        if not g.doubleBufferScreen
 
            ' Once all the work for the current frame is done, figure out how long it took to 
            ' complete. If the time taken was less than one display refresh period, sleep for the 
            ' remainder of the refresh period. Doing so makes the redraw rate for single-buffered
            ' situations match the double-buffered update rate established by g.screen.SwapBuffers().
            '       
            elapsedFrameTime = UpTime(0) - frameStartTime
            if elapsedFrameTime < g.refreshPeriod
                Sleep( Int(1000*(g.refreshPeriod - elapsedFrameTime)) )
            end if 
             
        end if
        
    end while 'true

End Sub ' RunScreenSaver( params As Object )


'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub DropNewLogo( currTime as Float )

' DESCRIPTION
'     Creates a new falling logo instance above the top of the screen.
'
'     A new logo is dropped whenever the current system time reaches the next drop time. 
'     This continues until the maximum number of permitted on-screen logos has been 
'     reached. After that, logo instances are recycled when the fall past the bottom of 
'     display.
'
' PARAMETERS
'     currTime - The current system time.
'     
' RETURNS
'     Void.

    g = GetGlobalAA()

    ' If we've reached the maximum number of falling logos, don't drop any more.
    if g.logoArray.Count() >= g.maxLogos then return
    
    ' If we've reached the appropriate time to start the next logo drop...
    if currTime >= g.nextLogoDropTime

        ' Create a new falling logo instance at a random position on the screen with a
        ' random size (which in turn implies depth and tint.)
        '
        g.logoArray.Push( CreateLogoInstance() )

        ' Re-sort the logo array so that falling logos will be drawn in the correct back
        ' to front order based on the logo size.
        '
        g.logoArray.SortBy( "width" )

        ' Pick the next time to drop a new logo.
        g.nextLogoDropTime = currTime + g.nextLogoDropDelay*Rnd(0) + 1.5

     end if 'currTime > nextDropTime
    
End Sub ' DropNewLogo( dropTime as Float )


'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub DrawLogoInstance( currTime as Float, logo as Object )

' DESCRIPTION
'     Draw the specified logo instance to the screen buffer.
'
' PARAMETERS
'     currTime - The current system time (used to step through logo squeeze animation when 
'                Easter-Egg sound playback is enabled.)
'                 
'         logo - The logo instance to draw.
'     
' RETURNS
'     Void.

    g = GetGlobalAA()

    ' Start by assuming we'll be applying the plain logo size and tint.
    logoZoom = logo.zoom
    tint     = logo.tint
    
    ' If the logo is visibly squeezing when the sound is being played...
    if logo.DoesExist("maxSoundSqueeze")
    
        ' And the sound has finished playing...
        if currTime > logo.soundStartTime + logo.soundDuration
        
            ' Delete the sound fields from the associated logo instance.
            logo.Delete( "soundDuration" )
            logo.Delete( "soundStartTime" )
            logo.Delete( "maxSoundSqueeze" )
            
            ' And flag that sound is nolonger playing.
            g.isSoundPlaying = false
            
        ' The sound has not finished playing, so we need to coninute the squeeze animation
        ' for the logo instance making the sound.
        '
        else
            ' Determine where in the squeeze animation we currently are.
            squeezePct = (currTime - logo.soundStartTime) / logo.soundDuration
            
            ' If we're in the first half of the sound playback, we should be squeezing
            ' the logo instance more and more.
            '
            if squeezePct <= 0.5
                
                ' Change the squeeze percent to represent how far along we are in the 
                ' squeezing down portion of the animation, and adjust the logo's natural
                ' size zoom by the squeeze amount.
                '
                squeezePct = squeezePct / 0.5
                logoZoom = logoZoom - squeezePct*logo.maxSoundSqueeze*logoZoom
           
            ' Otherwise we're in the second half of sound playback and we should be 
            ' unsqueezing the logo back to it's regular zoom size.
            '
            else
                ' Change the squeeze percent to represent how far along we are in the 
                ' unsqueezing portion of the animation, and adjust the logo's natural
                ' size zoom by the squeeze amount.
                '
                squeezePct = (1.0 - squeezePct) / 0.5
                logoZoom = logoZoom - squeezePct*logoZoom*logo.maxSoundSqueeze
            
            endif

            ' Brighten the logo's tint a little bit to make the squeezing logo a little 
            ' more obvious on-screen.
            '
            tint = AdjustTint( tint, 1.0 + 0.25*squeezePct, 1.0 )

        end if
    
    end if
    
    ' If the logo needs to be drawn with simulated rotation...
    if logo.rotSpeed <> 0.0 or logo.DoesExist( "maxSoundSqueeze" )
        
        ' Determine the screen width of the logo.
        logoScreenWidth  = logoZoom*g.logoBitmapWidth
        logoScreenHeight = logoZoom*g.logoBitmapHeight
        
        ' Determine the x-scaling to apply to the logo due to simulated rotation.
        logoRotScalar = Cos( 0.01745329*logo.rotAngle )
        
        ' Combine the rotation x-scaling with the zoom x-scaling to determine the 
        ' total x-scaling to apply to the logo bitmap.
        '
        logoXScalar = logoRotScalar * logoZoom 
        
        ' Determine the on-screen position at which to draw the "rotated" logo so
        ' that it appears tp spin around it's center.
        '
        logoX = logo.x + 0.5*( logo.width  - logoRotScalar*logoScreenWidth )
        logoY = logo.y + 0.5*( logo.height - logoScreenHeight )
        
        ' Round up logo X position if fractional X positioning is disabled.
        if not g.fracLogoXPos then logoX = Int(logoX+0.5)
        
        ' Round up logo Y position if fractional Y positioning is disabled.
        if not g.fracLogoYPos then logoY = Int(logoY+0.5)
        
        ' Draw the specified logo instance.
        g.screen.DrawScaledObject( logoX, logoY, logoxScalar, logoZoom, g.logoBitmap, tint )
    
    ' Otherwise, save some compute time by eliminating rotation calculations and just 
    ' draw the bitmap at the proper position, scale, and tint.
    '
    else
    
        ' Round up logo X position if fractional X positioning is disabled.
        logoX = logo.x
        if not g.fracLogoXPos
            logoX = Int(logoX+0.5)
        end if 
       
        ' Round up logo Y position if fractional Y positioning is disabled.
        logoY = logo.y
        if not g.fracLogoYPos
            logoY = Int(logoY+0.5)
        end if 

        ' And finally, draw the specified logo instance.
        g.screen.DrawScaledObject( logoX, logoY, logoZoom, logoZoom, g.logoBitmap, logo.tint )          

    end if

End Sub ' DrawLogoInstance( logo as Object )


'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub PlaySoundIfNeeded( currTime as Float )

' DESCRIPTION
'     Play the Easter-Egg sound if enabled and the time is right.
'
' PARAMETERS
'     currTime - The current system time.
'     
' RETURNS
'     Void.

    g = GetGlobalAA()

    ' If the sound object wasn't created (Easter-Egg sound playback is disabled) or 
    ' the sound is already playing, there's nothing to do.
    ' 
    if g.sound = invalid or g.isSoundPlaying then return

    ' If it's time to play the next sound or the user wants to force sound playback...
    if currTime >= g.nextPlaySoundTime or g.forcePlaySound

        ' Assume we won't need to suppress sound playback.
        suppressPlayback = false

        ' Then see what the display properties are for this device.    
        displayProps = g.devInfo.GetDisplayProperties()

        ' If this is a headless device, suppress sound playback. We do this to prevent 
        ' Roku accessories (i.e., Roku wireless speakers) from playing sounds on their
        ' own in addition to the Roku device controlling the screen.
        '
        if displayProps.headless 
            suppressPlayback = true
            if g.startedFromMain
                print "Easter-Egg sound playback suppressed on headless device"
            end if
        end if

        ' If the display reports back a zero width, we take that to mean that the 
        ' display is turned off. In that case, also suppress sound playback. We do 
        ' this to prevent Roku devices built into sound bars or paired with wireless 
        ' speakers from playing Easter-Egg sounds all night when the customer has 
        ' turned the display off and gone to bed.
        ' 
        if displayProps.width = 0
            suppressPlayback = true
            if g.startedFromMain
                print "Easter-Egg sound playback suppressed because display is off"
            end if
        end if
    
        ' If we're not suppressing playback, and the random likelyhood to play the
        ' sound hit or the user wants to force sound playback, it's time to trigger
        ' the Easter-Egg sound.
        '
        if not suppressPlayback and ( Rnd(0) < g.soundLikelyPct or g.forcePlaySound )
 
            ' For forced sound playback, always associate the sound with the largest (and
            ' loudest) on-screen logo. For on-demand playback, randomly associate the 
            ' sound playback and volume with a logo instance that falls in the range of
            ' the largest third of the logo instances.
            '
            if g.forcePlaySound
                logoIdx = g.logoArray.Count()-1
            else
                logoIdx = g.logoArray.Count() - Rnd( Int(g.logoArray.Count()/3+1) )
            end if
            
            ' It is possible that the selected logo is not easily visible (has yet to fall
            ' down from above the screen or its random horiziontal position has put it 
            ' really far to the left or right.) If this is the case, we'll want to search
            ' for a different logo that is well within the screen boundaries.
            while true
                
                ' First get the logo we randomly selected.
                logo = g.logoArray[logoIdx]
           
                ' If the logo is completely on screen, then we can stop searching for a
                ' more noticable logo.
                '
                if logo.x > 20 and logo.y > 20
                    if logo.x+logo.width < g.screenWidth and logo.y+logo.height < g.screenHeight
                        exit while
                    end if
                end if
            
                ' If we got here, the logo is partially or wholly off-screen. In this case
                ' try the next larger logo instead.
                '
                logoIdx = logoIdx + 1
                
                ' If there is no next larger logo, then give up and just use the largest
                ' logo, whether it's sufficiently on screen or not.
                ''
                if logoIdx >= g.logoArray.Count()
                    logoIdx = g.logoArray.Count()-1
                    exit while
                end if
                
            end while
            
            ' Set the sound start time as now, set how much to squeeze the noisy logo, and
            ' approximately how long the squeeze animation should last. 
            ' 
            logo.soundStartTime  = currTime
            logo.maxSoundSqueeze = 0.07
            logo.soundDuration   = 0.25

            ' Set the playback volume for the sound based on the permitted volume range 
            ' and the relative size of the logo.
            '
            volumeRange  = g.soundMaxVolume - g.soundMinVolume
            volumeScalar = logo.scalar ^ g.soundVolumeGamma
            volumeLevel  = Int(volumeScalar*volumeRange + g.soundMinVolume)
            
            ' And make some noise.
            g.isSoundPlaying = true
            g.sound.Trigger( volumeLevel )
             
        end if

        ' Once we've kicked off the sound playback (or suppressed it), clear the forced
        ' playback flag so that playout doesn't auto-repeat.
        '
        g.forcePlaySound = false
        
        ' Determine when to next attempt to auto-play the Easter-Egg sound.
        soundPeriodRange = g.SoundMaxPeriod - g.soundMinPeriod
        g.nextPlaySoundTime = currTime + Rnd(0)*soundPeriodRange + g.soundMinPeriod
    
    end if
    
End Sub


'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub CreateLogoInstance() as Object

' DESCRIPTION
'     Create an new instance of a falling logo.
'
' PARAMETERS
'     None.
'
' RETURNS
'     A falling logo object instance.
'

    g = GetGlobalAA()

    ' Generate a random 'size' scalar for the logo.
    logoScalar = RandomLogoScalar()
    
    ' From that scalar, generate the actual on-screen size, tint, and fall speed for 
    ' the logo instance.
    '
    logoSize  = LogoSizeFromScalar( logoScalar )
    logoTint  = LogoTintFromScalar( logoScalar )
    logoSpeed = LogoSpeedFromScalar( logoScalar )

    ' To keep logo instances from clumping together (completely random distribution 
    ' can be lumpy), determine what drop 'lane' to place the new logo instance in.
    '
    lane = g.laneArray[g.currLaneIdx]
        
    ' Allow the placement of the logo to run outside the lane on either side a bit, but only
    ' if the logo is smaller than the lane width.
    '
    laneExtra = 0.25*g.laneWidth
    if logoSize.width > g.laneWidth then laneExtra = 0.05*g.laneWidth
    
    logoOffset = 0.5*laneExtra
    
    ' Total allowable room in which the logo can be placed is the difference between the
    ' width of the lane and the width of the scaled logo plus any extra space we allotted.
    ' 
    wiggleRoom = g.laneWidth + laneExtra - logoSize.width
    
    ' The actual x position of the logo is then the starting position of the drop lane
    ' (offset by half the screen padding and the lane specific offset) plus the calculated
    ' wiggle room.
    ' 
    startX = Int( lane*g.laneWidth - g.laneGutter - logoOffset + Rnd(0)*wiggleRoom  )
    
    ' If we need to ensure that logos entirely on-screen do so.
    if g.fullyVisibleLogos
        if startX < 20
            startX = 20
        else if startX + logoSize.width > g.screenWidth - 20
            startX = g.screenWidth - 20 - logoSize.width
        end if    
    end if
    
    ' The actual y start position of the logo is a random value between just off the top 
    ' of the screen and an additional half logo height above the screen. This randomizes
    ' the start of the drop a little to not exactly match the disappearance of a logo off
    ' the bottom of the screen.
    '
    startY = Int( 0 - logoSize.height - logoSize.height*Rnd(0)/2 )

    ' Before returning, update the lane to use for the next time around.
    g.currLaneIdx = ( g.currLaneIdx + 1 ) mod g.laneCount
    
    rotSpeed = 0
    rotAngle = 0
    if logoScalar < 0.8
       rotAngle = 60.0*Rnd(0)  
       rotSpeed = (1.0-logoScalar)^2
    end if 
    
    ' Finally, return the logo instance information to the caller.
    return {
               x:        startX,
               y:        startY,
               scalar:   logoScalar,
               width:    logoSize.width,
               height:   logoSize.height,
               zoom:     logoSize.zoom,
               tint:     logoTint,
               speed:    logoSpeed,
               rotAngle: 0,
               rotSpeed: 0
           }

End Sub ' CreateLogoInstance() as Object


'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub UpdateLogoInstance( logo as Object ) as Boolean

' DESCRIPTION
'     Update the position of a logo instance on the screen, and if it falls off the bottom, 
'     recycle it back into a new random logo at the top of the screen.
'
' PARAMETERS
'     logo - The logo instance to update / recycle.
'
' RETURNS
'     An updated falling logo instance.

    g = GetGlobalAA()

    ' Update the vertical position of the falling logo.
    logo.y += logo.speed
    
    if logo.rotSpeed <> 0.0 
        logo.rotAngle = logo.rotAngle + logo.rotSpeed
    
        if logo.rotAngle > 180.0
           logo.rotAngle = 180.0
           logo.rotSpeed = -Abs(logo.rotSpeed)
        else if logo.rotAngle < 0
           logo.rotAngle = 0.0
           logo.rotSpeed = Abs(logo.rotSpeed)
        end if 
    end if
    
    ' If the new position fell off the bottom of the screen, recycle the logo instance. This
    ' consists of generating a new random logo size, a new horizontal starting position on the
    ' screen, and resetting the logo's tint and fall speed based on the size.
    '
    if logo.y > g.screenHeight

        ' If the logo instance being recycled was the white logo instance, then reset the 
        ' new logo ceiling to be any size in the range. 
        '
        if logo.tint = g.whiteColor
            g.logoScalarCeiling = 1.0
        end if
        
        ' Create a temporary new logo instance.
        newLogo = CreateLogoInstance()

        ' Copy the new logo's info to the one being recycled.
        logo.x        = newLogo.x
        logo.y        = newLogo.y
        logo.scalar   = newLogo.scalar
        logo.width    = newLogo.width
        logo.height   = newLogo.height
        logo.zoom     = newLogo.zoom
        logo.tint     = newLogo.tint
        logo.speed    = newLogo.speed
        logo.rotAngle = newLogo.rotAngle
        logo.rotSpeed = newLogo.rotSpeed

        ' Force the temporary logo instance to get freed now that we're done with it.
        newLogo = invalid

        ' Flag the caller that the logo was recycled and the logo array will need to be
        ' re-sorted to get the recycled logo to draw in the correct order.
        '
        return true

    end if 'logo.y > g.screenHeight

    ' If we got to this point, the falling logo's position was updated but it wasn't recycled.
    ' So flag the caller that the falling logo array will not need to be re-sorted.
    '
    return false

End Sub


'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub LoadLogoBitmap()

' DESCRIPTION
'     Loads a model/vendor/default logo PNG file for display.
'
'     If no vendor specific logo file is found or the g.forceDefaultLogo flag is set,
'     then the default "Roku" logo is used.
'
' PARAMETERS
'     None.
'
' RETURNS
'     Void.

    g = GetGlobalAA()
    
    ' Start with no logo bitmap.
    g.logoBitmap = invalid
    logoFilePath = ""
     
    ' If we're not being forced to use the default logo...
    if not g.forceDefaultLogo
    
        ' Try to get the stacked logo built into the custom package.
        logoFilePath = "brand:/Screensaver-BrandLogo-Stacked"
        g.logoBitmap = CreateObject( "roBitmap", logoFilePath )
        
        ' If there is no stacked logo, try the flat logo instead.
        if g.logoBitmap = invalid

           logoFilePath = "brand:/Screensaver-BrandLogo-Flat"
           g.logoBitmap = CreateObject( "roBitmap", logoFilePath )

        end if
        
    end if
    
    ' If we got to this point and no logo was loaded, load the default logo.
    if g.logoBitmap = invalid
    
        ' Assume we're going to use the HD version of the default logo.
        logoSizeStr = "HD"
        
        ' If FHD mode is active, then choose the FHD version of the default logo.
        if g.logoSizeInUse = "FHD"
            logoSizeStr = "FHD"
        end if

        ' Construct the path to the default logo file and load it up.
        logoFilePath = "pkg:/images/DefaultLogo_" + logoSizeStr + ".png"
        g.logoBitmap = CreateObject( "roBitmap", logoFilePath )
        
    end if
    
    if g.logoBitmap = invalid
        print "Unable to load logo bitmap from file [";logoFilePath;"]"
    else
        print "Loaded logo bitmap from file [";logoFilePath;"]"
    end if
    print ""

    ' And once we have a logo bitmap, save its native width and height for quick reference
    ' in other parts of the code.
    '
    g.logoBitmapWidth  = g.logoBitmap.GetWidth()
    g.logoBitmapHeight = g.logoBitmap.GetHeight()
     
End Sub ' LoadLogoBitmap()


'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub LoadAnimAttrs( mode as String )

' DESCRIPTION
'     Loads a default, vendor-specific, or model-specific animation attributes from 
'     a corresponding JSON file.
'
'     Calling this function for "fallback" first, and then "vendor" allows sparse but
'     more specific attributes to overlay more generic ones.
'
' PARAMETERS
'     mode - The type of attributes to read. Should be set to "fallback", or "vendor".
'                     
' RETURNS
'     Void.

    g = GetGlobalAA()

    ' Assume we will not find a logo_attr.json file to read.
    animAttrDir  = "pkg:/images"
    animAttrFile = "Default_Config.json"
    animAttrFlag = ""
    
    ' If the caller is asking to read the default animation attributes then do so.
    if mode = "default" 
 
        print "Reading default animation attributes from ["+ animAttrDir +"/" + animAttrFile + "]"

    ' At this point the caller is not asking for default animation attributes, so if 
    ' instead they want to check for vendor attributes...
    '
    else if mode = "vendor"
    
        ' Vendor specific attributes will be in the custom package area. So set up to 
        ' read the attributes file from there.
        '
        animAttrDir = "brand:"
        animAttrFile = "Screensaver-LogoRain-Config"
        animAttrFlag = " *"
        
        print "Reading vendor-specific animation attributes from [" + animAttrDir + "/" + animAttrFile + "]"              

    ' Anything else specified for mode is unrecognized, so exit early.
    else 
       return
    end if

    ' Try to read the specified animation attributes file.
    animAttrStr = ""
    animAttrStr = ReadAsciiFile( animAttrDir + "/" + animAttrFile )

    ' If the animation attribute file was successfully read, parse it.
    if Len( animAttrStr ) > 0
        ParseAnimAttrs( animAttrStr, animAttrFlag )
    end if
     
End Sub ' LoadAnimAttrs()


'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub LoadCfgSvcAnimAttrs()

' DESCRIPTION
'     Reads and applies any animation attribute overrides that may be registered
'     with the config-service.
'
' PARAMETERS
'     None.
'
' RETURNS
'     Void.

    print ""

    g = GetGlobalAA()
    
    ' Get a configuration service object.
    cfg = CreateObject( "roConfigService" )
    
    'Define the base key path for accessing config-service overrides.
    baseCfgSvcKeyPath = "fw.safemode-screensaver"
    
    ' See if any default animation overrides have been set.
    defaultKeyStr  = baseCfgSvcKeyPath + ".defaults"
    defaultAttrStr = cfg.GetString( defaultKeyStr, "" )
    
    ' If so, parse and apply them.
    anyChanges = false
    if Len( defaultAttrStr ) > 0
        anyChanges = ParseAnimAttrs( defaultAttrStr, " !" )    
    end if

    ' Log whether any default animation attributes were changed or not.
    if anyChanges
        print "Applied config-service animation attribute overrides from [";defaultKeyStr;"]"  
    else
        print "No config-service animation attribute overrides found for [";defaultKeyStr;"]"  
    end if 
    
    'Determine the name of the vendor for this device.
    vendorName = GetVendorName()
   
    ' See if any vendor-specific overrides have been set.
    vendorKeyStr  = baseCfgSvcKeyPath + "." + vendorName
    vendorAttrStr = cfg.GetString( vendorKeyStr, "" )
    
    ' If so, parse and apply them.
    anyChanges = false
    if Len( vendorAttrStr ) > 0 
        anyChanges = ParseAnimAttrs( vendorAttrStr, " !*" )
    end if

    ' Log whether any vendor-specific animation attributes were changed or not.
    if anyChanges
        print "Applied config-service animation attribute overrides from [";vendorKeyStr;"]"  
    else
        print "No config-service animation attribute overrides found for [";vendorKeyStr;"]"  
    end if 
    
    ' Determine the model name for this device.
    modelName = g.devInfo.GetModel()
    
    'See if any model-specific overrides have been set.
    modelKeyStr  = baseCfgSvcKeyPath + "." + vendorName + "." + modelName
    modelAttrStr = cfg.GetString( modelKeyStr, "" )
    
    'If so, parse and apply them.
    anyChanges = false
    if Len( modelAttrStr ) > 0 
        anyChanges = ParseAnimAttrs( modelAttrStr, " !+" )
    end if

    ' Log whether any model-specific animation attributes were changed or not.
    if anyChanges
        print "Applied config-service animation attribute overrides from [";modelKeyStr;"]"  
    else
        print "No config-service animation attribute overrides found for [";modelKeyStr;"]"  
    end if 

End Sub ' LoadCfgSvcAnimAttrs()


'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub ParseAnimAttrs( animAttrStr as String, animAttrFlag as String ) as Boolean

    g = GetGlobalAA()
    
    ' Parse the JSON text into named members with values.
    animAttrs = ParseJSON( animAttrStr )

    ' Assume that none of the attributes actually got changed.
    animAttrChanged = false

    ' If the JSON was parsed successfully...
    if animAttrs <> invalid

        ' Step through all the supported attribute names...
        for each attrName in g.attrNames
        
            ' And if the attribute was found in the JSON string...
            if animAttrs[attrName] <> invalid
               
               ' And the attribute is a color string, parse the color string into an integer.
               ' For all other attributes, keep their original type.
               '
               if attrName.InStr( 0, "Color" ) >= 0
                   g[attrName] = Val( animAttrs[attrName], 0 )
               else
                   g[attrName] = animAttrs[attrName]
               end if
               
               ' Update the attribute changed flag with passed in flag character(s).
               g.attrChanged[attrName] = animAttrFlag
               
               ' Flag that one or more attributes were changed by the JSON string.
               animAttrChanged = true
               
            end if
            
        end for
    
    end if ' animAttrs <> invalid

    ' Tell the caller whether any of the attributes were changed or not.
    return animAttrChanged
    
End Sub ' ParseAnimAttrs( animAttrStr as String, animAttrFlag as String ) as Boolean


'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub SummarizeAnimAttrs()
    
    g = GetGlobalAA()
    
    print ""
    print "Double Buffer Screen:   " + g.doubleBufferScreen.ToStr()       + g.attrChanged.doubleBufferScreen
    
    if g.garbageCollectFrameCount = 0
        print "Garbage Collect Count:  " + "auto" + g.attrChanged.garbageCollectFrameCount
    else
        print "Garbage Collect Count:  " + g.garbageCollectFrameCount.ToStr() + g.attrChanged.garbageCollectFrameCount
    end if
    
    print "Max Logo Count:         " + g.maxLogos.ToStr()                 + g.attrChanged.maxLogos
    print "Slow White Logo:        " + g.slowWhiteLogo.ToStr()            + g.attrChanged.slowWhiteLogo

    print ""
    print "Fractional Logo X Pos:  " + g.fracLogoXPos.ToStr()   + g.attrChanged.fracLogoXPos
    print "Fractional Logo Y Pos:  " + g.fracLogoYPos.ToStr()   + g.attrChanged.fracLogoYPos

    print ""
    print "Fully Visible Logos:    " + g.fullyVisibleLogos.ToStr()   + g.attrChanged.fullyVisibleLogos
    
    print ""
    print "Bkgd Color:             0x%08X".Format(g.bkgdColor)  + g.attrChanged.bkgdColor
    print "Logo Color 1:           0x%08X".Format(g.logoColor1) + g.attrChanged.logoColor1
    print "Logo Color 2:           0x%08X".Format(g.logoColor2) + g.attrChanged.logoColor2
    print "Logo Color 3:           0x%08X".Format(g.logoColor3) + g.attrChanged.logoColor3
    
    print ""
    print "Logo Min Brightness:    " + g.logoMinBrightness.ToStr()   + g.attrChanged.logoMinBrightness
    print "Logo Max Brightness:    " + g.logoMaxBrightness.ToStr()   + g.attrChanged.logoMaxBrightness
    print "Logo Brightness Gamma:  " + g.logoBrightnessGamma.ToStr() + g.attrChanged.logoBrightnessGamma
    
    print ""   
    print "Logo Min Width:         " + g["logoMinWidth"+g.logoSizeInUse].ToStr() + g.attrChanged["logoMinWidth"+g.logoSizeInUse]
    print "Logo Max Width:         " + g["logoMaxWidth"+g.logoSizeInUse].ToStr() + g.attrChanged["logoMaxWidth"+g.logoSizeInUse]
    print "Logo Size Gamma:        " + g.logoSizeGamma.ToStr() + g.attrChanged.logoSizeGamma
    
    print ""
    print "Logo Min Speed:         " + g.logoMinSpeed.ToStr()   + g.attrChanged.logoMinSpeed
    print "Logo Max Speed:         " + g.logoMaxSpeed.ToStr()   + g.attrChanged.logoMaxSpeed
    print "Logo Speed Gamma:       " + g.logoSpeedGamma.ToStr() + g.attrChanged.logoSpeedGamma

    SummarizeSoundAttrs()
    
    print ""
    print "Note: Above values followed by"
    print "    ! = Config-Service Override."
    print "    * = Vendor-Specific Override."
    print "    + = Model-Specific Override."
    print ""

End Sub ' SummarizeAnimAttrs()


'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub SummarizeSoundAttrs()

    g = GetGlobalAA()
    
    ' If the screensaver was started for debug purposes as an app, summarize the 
    ' Easter-Egg sound file playback likelyhood and other attributes.
    '
    if g.startedFromMain 
    
        print ""
    
        ' If there is no sound object, the Easter-Egg sound is disabled. So reflect that in the summary.
        if g.sound = invalid

            print "Easter-Egg:             Disabled" + g.attrChanged.soundFile

        ' Otherwise, detail the parameters for Easter-Egg playback.
        else

            print "Easter-Egg Likelyhood:   " + Int(g.soundLikelyPct*100).ToStr() + "%" + g.attrChanged.soundLikelyPct

            soundPeriodStr = ""
            soundHours     = Int( g.soundMinPeriod/3600 )
            soundMinutes   = Int( g.soundMinPeriod/60 ) - 60*soundHours

            if soundHours <> 0 
                soundPeriodStr = soundHours.ToStr() + " hour"
                if soundHours <> 1 then soundPeriodStr = soundPeriodStr + "s"
            endif
            if soundMinutes <> 0
                if soundHours <> 0 then soundPeriodStr = soundPeriodStr + ", "
                soundPeriodStr = soundPeriodStr + soundMinutes.ToStr() + " minute"
                if soundMinutes <> 1 then soundPeriodStr = soundPeriodStr + "s"
            endif 
            print "Easter-Egg Min Interval: " + soundPeriodStr + g.attrChanged.soundMinPeriod

            soundPeriodStr = ""
            soundHours     = Int( g.soundMaxPeriod/3600 )
            soundMinutes   = Int( g.soundMaxPeriod/60 ) - 60*soundHours

            if soundHours <> 0 
                soundPeriodStr = soundHours.ToStr() + " hour"
                if soundHours <> 1 then soundPeriodStr = soundPeriodStr + "s"
            endif
            if soundMinutes <> 0
                if soundHours <> 0 then soundPeriodStr = soundPeriodStr + ", "
                soundPeriodStr = soundPeriodStr + soundMinutes.ToStr() + " minute"
                if soundMinutes <> 1 then soundPeriodStr = soundPeriodStr + "s"
            endif 
            print "Easter-Egg Max Interval: " + soundPeriodStr + g.attrChanged.soundMaxPeriod 

            print "Easter-Egg Sound File:   " + g.soundFile + g.attrChanged.soundFile

            print "Easter-Egg Min Volume:   " + g.soundMinVolume.ToStr() + "%" + g.attrChanged.soundMinVolume 
            print "Easter-Egg Max Volume:   " + g.soundMaxVolume.ToStr() + "%" + g.attrChanged.soundMaxVolume

            print "Easter-Egg Volume Gamma: " + g.soundVolumeGamma.ToStr() + g.attrChanged.soundVolumeGamma

        end if ' g.sound = invalid 

    end if ' g.startedFromMain 

End Sub ' SummarizeSoundAttrs


'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub ShouldForceDefaultLogo( defaultLogoVendors as String ) as Boolean

    g = GetGlobalAA()
      
    ' Determine the name of the vendor for this device.  
    vendorName = GetVendorName()
    
    ' If this vendor is not in the list of vendors that must use the default logo, tell the caller.
    if Instr( 1, defaultLogoVendors, vendorName ) = 0 then return false
    
    ' Otherwise, tell the caller that the default logo should be used.
    return true
    
End Sub ' GetVendorName() as String


'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub GetVendorName() as String

    g = GetGlobalAA()
      
    ' Get the vendor USB name for the current device.
    details = g.devInfo.GetModelDetails()
    vendorName = details.vendorUSBName
   
    ' If the vendor USB name is valid, return that.
    if vendorName <> Invalid
        return vendorName
    end if
     
    ' Otherwise fall back to the vendor name. 
    vendorName = details.vendorName

    ' If the vendor name is valid, return that.
    if vendorName <> Invalid
        return vendorName
    end if

    ' If neither vendor name is valid for some reason, fall back to using Roku.
    return "Roku"

End Sub ' GetVendorName() as String


'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub RandomLogoScalar() as Float

' DESCRIPTION
'     Randomly select a scalar value for a falling logo instance. This scalar will be used as
'     a base attribute from which 
'
' PARAMETERS
'     None.
'
' RETURNS
'     The randomly selected scalar to use for a falling logo instance.
'
' NOTES
'     The g.logoScalarCeiling is set 1.0 when no falling logo instance is a slow-falling white
'     logo. If a falling logo instance is a slow-falling white logo, g.logoScalarCeiling will
'     be a smaller value to ensure that all other logos are smaller and not detract from the 
'     brightness, size, and color of the slow-falling logo.
'

    g = GetGlobalAA()
    return g.logoScalarCeiling * Rnd(0)

End Sub


'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub LogoSizeFromScalar( logoScalar as Float ) as Object

' DESCRIPTION
'     Construct a logo size based on the scalar value passed.
'
' PARAMETERS
'     logoScalar - The scalar value for a logo instance from which to derive the 
'                  logo size. Typically, this value is generated by a previous
'                  call to RandomLogoScalar() above.
'
' RETURNS

    g = GetGlobalAA()

    ' A logo instance's size will be limited to a value range specified by fallback or
    ' vendor/model specific attributes. First, determine what the range endpoints are.
    '
    widthRange = g["logoMaxWidth"+g.logoSizeInUse] - g["logoMinWidth"+g.logoSizeInUse]

    ' Next, determine the scaled width of the logo based on the range and a gamma value.
    ' The gamma value is used to bias logo size to favor larger sizes (gamma < 1), 
    ' smaller sizes (gamma > 1), or no bias at all (gamma = 1.)
    '
    scaledWidth = widthRange * (logoScalar ^ g.logoSizeGamma) + g["logoMinWidth"+g.logoSizeInUse]

    ' For faster drawing in the main animation loop, precalculate a zoom factor based
    ' on the ratio of the scaled logo vs the logo bitmaps native size.
    '
    scaleFactor = scaledWidth / g.logoBitmapWidth

    ' Based on the scale factor we just determned, generate the scaled logo height.
    scaledHeight = g.logoBitmapHeight * scaleFactor
    
    ' And return the sizing info back to the caller.
    return {
               width:  scaledWidth,
               height: scaledHeight,
               zoom:   scaleFactor
           }

End Sub ' LogoSizeFromScalar( logoScalar as Float ) as Object


'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub LogoSpeedFromScalar( logoScalar as Float ) as Float

' DESCRIPTION
'     Determine the fall speed for the logo scalar passed. Logos that are smaller (and
'     thus further back) will fall at a slower rate than larger ones near the front.
'
' PARAMETERS
'     logoScalar - The scalar value for a logo instance from which to derive a logo 
'                  instance's fall speed. Typically, this value is generated by a
'                  previous call to RandomLogoScalar() above.
'
' RETURNS
'     A floating point value by which to increment the y position each time the falling logo is drawn.
'
' NOTES
'     Fall speed was experimentally determined based on size range.

    g = GetGlobalAA()

    speedRange = g.logoMaxSpeed - g.logoMinSpeed
    speed = speedRange * ( logoScalar ^ g.logoSpeedGamma ) + g.logoMinSpeed

    return g.AnimSpeedScalar * speed

End Sub 'LogoSpeedFromScalar( logoScalar as Float ) as Float


'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub LogoTintFromScalar( logoScalar as Float ) as Integer

' DESCRIPTION
'     Determine the tint for a falling Roku logo.
'
'     The basic tint for a logo instance is determined by the logo color attribute.
'     Then tint is then darkened based on the scalar value so that smaller logo 
'     instances will be darker, making them look like they are receeding into the
'     distance.
'
' PARAMETERS
'     logoScalar - The scalar value for a logo instance from which to derive a logo 
'                  instance's tint. Typically, this value is generated by a
'                  previous call to RandomLogoScalar() above.
'
' RETURNS
'     An RGBA tint to use for a falling logo instance.
'
' NOTES
'     Default tint darkening was determined experimentally based on size scalar. 
'     Also, a small amount of tint randomization was added to make each logo 
'     instance a little more distinct. 

    g = GetGlobalAA()

    ' Determine how many logo colors we can choose from. Note that 0x00000000 is 
    ' for a color is considered "not specified." If logoColor3 is not specified,
    ' then all logos will be either logoColor1 or logoColor2. Similarly, if
    ' logoColor2 is not defined, all logos will be logoColor1.
    '
    nColors = 3
    if g.logoColor3 = 0 then nColors = 2
    if g.logoColor2 = 0 then nColors = 1
    
    ' Randomly select one of the available colors and get its RGB components.
    tintColorInt = g[ "logoColor" + Rnd(nColors).ToStr() ]
    tintColor    = ColorComponentsFrom( tintColorInt )
    
    ' Get more conveniently named RGB components for repeated use below.
    logoR = tintColor.r
    logoG = tintColor.g
    logoB = tintColor.b

    ' Generate a bit of noise to add to the tint so that every logo instance is
    ' just a little bit different from the others. Doing this makes the whole
    ' experience a little less flat.
    '
    noiseR = 4*Rnd(0) - 1
    noiseG = 2*Rnd(0) - 1
    noiseB = 4*Rnd(0) - 1

    ' Add the noise to the tint, and darken the tint based on the size of the 
    ' logo instance. The gamma value is used to bias logo darkening to favor 
    ' brighter logos (gamma < 1), darker logos (gamma > 1), or no bias at all 
    ' (gamma = 1.)
    '
    tintR = (logoScalar ^ g.logoBrightnessGamma) * (logoR + noiseR) / 255
    tintG = (logoScalar ^ g.logoBrightnessGamma) * (logoG + noiseG) / 255
    tintB = (logoScalar ^ g.logoBrightnessGamma) * (logoB + noiseB) / 255
    
    ' The tint values generated in the previous step should fall in the range
    ' [0.0,1.0], but the addition of the noise may have thrown one or more 
    ' components over 1.0. If that happened, rescale the tint components to 
    ' all be back in range.
    '
    maxRGB = BiggestRGB( [tintR,tintG,tintB] )
    if maxRGB > 1
        tintR = tintR / maxRGB
        tintG = tintG / maxRGB
        tintB = tintB / maxRGB
    end if

    ' A logo instance's brightness will be limited to a value range specified 
    ' by fallback or vendor/model specific attributes. So determine what the 
    ' size of that range is.
    '
    brightRangeR = g.logoMaxBrightness*logoR - g.logoMinBrightness*logoR
    brightRangeG = g.logoMaxBrightness*logoG - g.logoMinBrightness*logoG
    brightRangeB = g.logoMaxBrightness*logoB - g.logoMinBrightness*logoB

    ' Then use the range size and lower bound to detemine the final tint 
    ' color values in the range [0,255].
    ' 
    tintR = Int( brightRangeR * tintR + g.logoMinBrightness*logoR )
    tintG = Int( brightRangeG * tintG + g.logoMinBrightness*logoG )
    tintB = Int( brightRangeB * tintB + g.logoMinBrightness*logoB )

    ' And Finally, piece together the RGB and A components and return the result 
    ' as the tint for the specified logo scalar value.
    '
    return (tintR << 24) or (tintG << 16) or (tintB << 8) or &hFF

End Sub ' LogoTintFromScalar( logoScalar as Float ) as Integer


'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub BiggestRGB( rgbArray as Object ) as Dynamic

' DESCRIPTION
'     Determine the largest color component of an RGB array.
'
'     Use primarily to normalize colors to a [0:1] or [0:255] color range.
'
' PARAMETERS
'     rgbArray - The array containing the RGB component values to compare.
'
' RETURNS
'     The largest component value in the RGB array.

    first = true
    maxValue = 0

    ' Step through the RGB components...
    for each value in rgbArray

        ' And if this is the first component or the largest component encountered, save it.
        if first or (maxValue < value)
            maxValue = value
            first = false
        end if

    end for

    ' Finally, return the value of the largest component to the caller.
    return maxValue

End Sub ' BiggestRGB( rgbArray as Object ) as Integer


'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub Clamp( minV as Float, v as Float, maxV as Float ) as Float

   if v < minV then v = minV
   if v > maxV then v = maxV
   return v
   
End Sub


'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub AdjustTint( tint as integer, colorPct as float, alphaPct as Float ) as Integer
    
    tintColor = ColorComponentsFrom( tint )
    
    adjR = Int( Clamp( 0, colorPct*tintColor.r, 255 ) )
    adjG = Int( Clamp( 0, colorPct*tintColor.g, 255 ) )
    adjB = Int( Clamp( 0, colorPct*tintColor.b, 255 ) )
    adjA = Int( Clamp( 0, alphaPct*tintColor.a, 255 ) )
    
    return (adjR << 24) or (adjG << 16) or (adjB << 8) or adjA

End Sub


'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub MakeWhiteLogo()

' DESCRIPTION
'     Converts a regular logo into a slow falling white logo.
'
' PARAMETERS
'     None.
'
' RETURNS
'     Void.

    g = GetGlobalAA()

    ' If the slow-falling white logo option is turned off, return early.
    if not g.slowWhiteLogo 
        return
    endif

    ' Start by assuming we won't find an existing white logo.
    whiteFound = false
    largestNewLogo = invalid

    ' Look through the logo array for a white logo, keeping track of the 
    ' largest new logo (off the top of the screen) that we find.
    '
    for each logo in g.logoArray

        ' If we found an exiting white logo, we don't want to create another,
        ' so just return early.
        '
        if logo.tint = g.whiteColor then return

        ' The current logo isn't white. So if it's a new logo (i.e., off the
        ' top of the screen) and pretty large to start with, hang on to it as
        ' a candidate for the slow-white logo instance. (NOTE: Because the 
        ' falling logo array is sorted by size, each subsequent large logo 
        ' discovered off the top of the screen is guaranteed to be larger than
        ' the previous.)
        '
        if logo.y + logo.height < 0 and logo.scalar > 0.8 
            largestNewLogo = logo
        end if

    end for

    ' At this point, no existing white logo was found. So reset the size range
    ' for new logos to the entire range.
    '
    g.logoScalarCeiling = 1.0

    ' If no logo that was found was large enough to be a slow-white logo, return early.
    if largestNewLogo = invalid then return

    ' At this point we found a large logo that could act as a slow-white logo. But we don't
    ' want to see too many of them, so only accept one in four as a slow-white logo.
    '
    if Rnd(0) < 0.25
 
        ' Set the new logo's color to white.   
        largestNewLogo.tint  =  g.whiteColor
        
        ' Make it the largest logo the range will allow.
        largestNewLogo.scalar = 1.0
        newLogoSize = LogoSizeFromScalar( largestNewLogo.scalar )
        largestNewLogo.width  = newLogoSize.width
        largestNewLogo.height = newLogoSize.height
        largestNewLogo.zoom   = newLogoSize.zoom

        ' Make it fall at 25% normal speed.
        largestNewLogo.speed = 0.25*LogoSpeedFromScalar( largestNewLogo.scalar )
 
        ' So as to not detract from the slow-white logo, make sure that none of the 
        ' new logos that may appear after it are larger than the slow-white logo.
        '
        g.logoScalarCeiling  = 0.85*largestNewLogo.scalar

        ' For aesthetic reasons, don't allow the slow-white logo to fall partially
        ' off the screen in the horizontal direction.
        '
        if largestNewLogo.x < 20
            largestNewLogo.x = 20
        else if largestNewLogo.x + largestNewLogo.width > g.screenWidth - 20
            largestNewLogo.x = g.screenWidth - largestNewLogo.width - 20 
        endif
 
    end if ' Rnd(0) < 0.25

End Sub ' MakeWhiteLogo()


'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub ColorComponentsFrom( intColor as Integer ) as Object

' DESCRIPTION
'     Dissect an integer containing RGB and A components into separate values.
'
' PARAMETERS
'     None.
'
' RETURNS
'     Void.

  return {
            r: ( intColor >> 24 ) AND &hFF,
            g: ( intColor >> 16 ) AND &hFF,
            b: ( intColor >>  8 ) AND &hFF,
            a: (    intColor    ) AND &hFF
         }

End Sub


'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub CalcAnimationRateInfo()

' DESCRIPTION
'     Drop rate for logos were requested to be "slow and soothing." Default animation values that
'     met this criteria were determined on an STB with 60 Hz update rate.
'
'     However, Roku based systems may refresh at 24, 25, 30, 50, or 60Hz. So calculate an animation
'     drop speed scalar to ensure that the same soothing look is preserved on systems with any of
'     the above refresh rates.
'
' PARAMETERS
'     None.
'
' RETURNS
'     Void.

    g = GetGlobalAA()
    
    ' Animations were intially designed on a 60 fps system.
    g.animFPS = 60.0
 
    ' Get the video mode string for the current device, which will have the form "lines(i/p)][ff]"
    ' (i.e., 720p 1080i, 1080p50, etc.)
    '    
    videoModeStr = g.devInfo.GetVideoMode() 
    
    ' Look for the 'i' in the video mode string. If we couldn't find it, look for 'p' instead.
    rateCharPos = Instr( 1, videoModeStr, "i")
    if rateCharPos = 0 then rateCharPos = Instr( 1, videoModeStr, "p" ) 
 
    videoHeightStr = Left( videoModeStr, rateCharPos-1 )
    if videoHeightStr = "480"
        g.logoSizeInUse = "SD"
    else if videoHeightStr = "720"
        g.logoSizeInUse = "HD"
    else
        g.logoSizeInUse = "FHD"
    endif
 
    ' Isolate the refresh rate from the resolution as the 'i' or 'p' and the two digits
    ' immediately after it.
    '
    refreshRateStr = Mid( videoModeStr, rateCharPos, 3 ) 
    
    ' Assume no rate will be specified (i.e., the video mode string ends with 'i' or 'p'.)
    ' in this case, 60fps is assumed.
    '
    g.refreshRate = 60.0
    
    ' If there are some digits after the 'i' or 'p' read them as an explicit refresh rate. 
    if Len( refreshRateStr ) = 3 
        g.refreshRate = Val( Mid(refreshRateStr,2,2) )
    end if
    
    ' Determine a speed scalar that will adjust the logo drop speed if the current refresh
    ' rate doesn't match the design 60fps. (i.e., a 30fps display will update half as often
    ' as a 60fps display, and so the animation speed will need to be multiplied by 2x to
    ' get the same visual appearance as on a 60fps device.)
    '
    g.animSpeedScalar = g.animFPS / g.refreshRate
    
    ' Determine how many milliseconds the refresh period lasts on this device. This value 
    ' is used to 'sleep' the animation to achieve even update rates if single-buffering 
    ' is enabled (and wait for vertical retrace doesn't occur).
    '
    g.refreshPeriod = 1 / g.refreshRate

    ' We'll output a device summary below, so get the model number and vendor name for 
    ' this device.
    '
    model      = g.devInfo.GetModel()
    details    = g.devInfo.GetModelDetails()
    vendorName = details.vendorName

    ' Log the vendor name, model number, and animation rate information to the console for
    ' diagnostic purposes when debugging this screensaver.
    '
    print ""
    print "Vendor Name:            " + vendorName
    print "Model:                  " + model
    print ""
    print "Force Default Logo:     " + g.forceDefaultLogo.ToStr()
    print ""
    print "Target Animation Rate:  " + g.animFPS.ToStr() + "fps"
    print "Video Mode:             " + videoModeStr
    print "Refresh Rate:           " + refreshRateStr + " (" + g.refreshRate.ToStr() + "Hz)"
    print "Refresh Period:         " + (g.refreshPeriod*1000).ToStr() + "ms"
    print "Animation Speed Scalar: " +  g.animSpeedScalar.ToStr() + "x"    
    print ""
    
End Sub ' CalcAnimationRateInfo()

