' ******************************************************
' Copyright Roku 2011-2017
' All Rights Reserved
' ******************************************************
'
' NWM_Bookmarks.brs
' chris@thenowhereman.com
' partially rewritten by dluck@roku.com
'
' A BrightScript class for local storage of video bookmarks
' Useful for implementing "resume" functionality when the
' associated API does not support bookmarks
'
' REQUIREMENTS
' Each video you use with this class must have a guid attribute
' on its content-meta-data whose value is a string that uniquely
' identifies the video. The video must also have a playStart
' attribute which is then persistently stored by this class.
'

function NWM_Bookmarks()
  if m.BookmarkPtr = invalid
   this = {
    depth:  21
    debug : true
    LoadBookmarks:          NWM_BOOKMARKS_LoadBookmarks
    SaveBookmarks:          NWM_BOOKMARKS_SaveBookmarks
    UpdateBookmarkForVideo: NWM_BOOKMARKS_UpdateBookmarkForVideo
    GetBookmarkForVideo:    NWM_BOOKMARKS_GetBookmarkForVideo
    ClearBookmarks:         NWM_BOOKMARKS_ClearBookmarks
    DumpBookmarks:          NWM_BOOKMARKS_DumpBookmarks
   }
   m.BookmarkPtr = this
  endif
  
  return m.BookmarkPtr
end function

sub NWM_BOOKMARKS_LoadBookmarks()
  if m.debug then print "NWM_Bookmarks: LoadBookmarks"
  m.bookmarks = []
  
  raw = ValidStr(RegRead("bookmarks", "nwm_bookmarks"))
  xml = CreateObject("roXMLElement")
  if xml.Parse(raw)
    index = 0
    for each bookmark in xml.bookmark
      bmpos = StrToI(ValidStr(bookmark@position))
      if bmpos > 0
        if index < m.depth
          m.bookmarks.Push({
            id:         ValidStr(bookmark@id)
            position:   bmpos
          })
        else
          exit for
        endif
        index = index + 1
      endif
    next
  end if
  if m.debug m.DumpBookMarks()
end sub

sub NWM_BOOKMARKS_DumpBookmarks()
  if m.bookmarks = invalid
    m.LoadBookmarks()
  end if
  print "Dumping Bookmarks count=";m.bookmarks.count();"   --------------------------------------"
  for each bookmark in m.bookmarks
    print bookmark.position;"       [";bookmark.id;"]"
  next
  print "----------------------------------------------------------------------"
end sub

sub NWM_BOOKMARKS_SaveBookmarks()
 if m.bookmarks = invalid
     print "No bookmarks to save"
 else
  if m.debug then print "NWM_Bookmarks: SaveBookmarks count=";m.bookmarks.count()

  xml = CreateObject("roXMLElement")
  xml.SetName("bookmarks")
  for each bookmark in m.bookmarks
      node = xml.AddElement("bookmark")
      node.AddAttribute("id", bookmark.id)
      node.AddAttribute("position", bookmark.position.ToStr())
  next
  
  'if m.debug then print xml.GenXML(true)
  RegWrite("bookmarks", xml.GenXML(false), "nwm_bookmarks")
 endif
end sub

sub NWM_BOOKMARKS_UpdateBookmarkForVideo(video, saveBookmark = true)
  if video.playStart = invalid or video.guid = invalid
      if m.debug then print "NWM_Bookmarks: UpdateBookmarkForVideo - no bookmark to save - exit"
      return
  else
      if m.debug then print "NWM_Bookmarks: UpdateBookmarkForVideo pos=";video.playStart;"    guid=";video.guid
  end if

  if m.bookmarks = invalid
    m.LoadBookmarks()
  end if
  
  success = false
  index = 0
  for each bookmark in m.bookmarks
    if bookmark.id = video.guid
      ' if new playStart is zero, just delete the bookmark
      if video.playStart = 0
        m.bookmarks.delete(index)
        success = true
      else
        if index = 0
          ' if this is the first entry in the list, just update the position
          bookmark.position = video.playStart
          success = true
        else
          ' if we are not updating the first position, just delete it, and insert a new one a the front of the array
          m.bookmarks.delete(index)
        endif
      endif
      exit for
    end if
    index = index + 1
  next
  if m.debug
     print"internal update pos=";video.playStart;" index=";index;" success=";success
  endif
  
  if (not success) and (video.playStart>0)
    if m.bookmarks.count() >= m.depth  ' if list is full, drop one off the end
        m.bookmarks.pop()
    endif
    ' now shove latest one at the start of the array
    m.bookmarks.unshift({
      id:         video.guid
      position:   video.playStart
    })

  end if
  
  if saveBookmark
    m.SaveBookmarks()
  end if
end sub

function NWM_BOOKMARKS_GetBookmarkForVideo(video)
  if m.debug then print "NWM_Bookmarks: GetBookmarkForVideo ";video.guid
  result = 0
  
  if m.bookmarks = invalid
    m.LoadBookmarks()
  end if

  for each bookmark in m.bookmarks
    if bookmark.id = video.guid
      result = bookmark.position
      exit for
    end if
  next
  
  return result
end function

sub NWM_BOOKMARKS_ClearBookmarks()
  if m.debug then print "NWM_Bookmarks: ClearBookmarks"

  m.bookmarks = []
  m.SaveBookmarks()
end sub
