' ********** Copyright 2019 Roku Corp. All Rights Reserved. **********

function ShowVideoDetail(contentList as Object, selectedItem as Object) as Object
    controller = GetBaseViewController("DetailsView")
    controller.Append({
        dialog: CreateObject("roSGNode", "Dialog")
        isShownFirstTime: true
        isFirstContentSet: true
        contentList: contentList
        selectedItem: selectedItem
        scene: m.top
        unShuffleContent: invalid
        overhangOptionStr: Tr("Search & Options")

        SetUpScreen: VideoDetail_SetUpScreen
        CleanUp: VideoDetail_CleanUp
        CreateButtons: VideoDetail_CreateButtons
        StartPlayback: VideoDetail_StartPlayback

        OnItemLoaded: VideoDetail_OnItemLoaded
        OnButtonSelected: VideoDetail_OnButtonSelected
        OnDetailsWasShown: VideoDetail_OnDetailsWasShown
        OnVideoWasClosed: VideoDetail_OnVideoWasClosed
        OnContentChanged : VideoDetail_OnContentChanged
        OnDetailsShuffleContent : VideoDetail_ShuffleContent
    })
    if isUSBOnly() = true
        controller.overhangOptionStr = Tr("Options")
    end if
    controller.Observe(controller.view, "content", "OnContentChanged")
    controller.Observe(controller.view, "itemLoaded", "OnItemLoaded")
    controller.Observe(controller.view, "buttonSelected", "OnButtonSelected")
    controller.Observe(controller.view, "wasShown", "OnDetailsWasShown")

    controller.SetUpScreen()
    ShowNewView(controller.view)

    return controller.view
end function

sub VideoDetail_SetUpScreen()
    m.scene.buttonBar.visible = false
    m.view.style = "rmp"
    m.view.isContentList = true
    content = CreateObject("roSGNode", "ContentNode")
    config = {
        HandlerConfigDetails: {
            name: "VideoDetailScreenCH"
            fields: {
                contentList: m.contentList.clone(true)
                selectedItem: m.selectedItem
            }
        }
    }
    content.Update(config, true)
    m.view.content = content

    mediaTypeFilters = Config().Get("mediaTypeFilters")
    currentMTFilter = getMTFilter()
    selectedMediaType = mediaTypeFilters[currentMTFilter]

    m.view.theme = {
        OverhangOptionsText: m.overhangOptionStr
    }
    if m.view.addField("shuffle", "boolean", false)
        m.Observe(m.view, "shuffle", "OnDetailsShuffleContent")
    end if
    m.SetOverhangTitle(selectedMediaType)
end sub

sub VideoDetail_OnContentChanged(detailsView as Object, content as Object)
    if content.hasField("jumpToItem")
        detailsView.jumpToItem = content.jumpToItem
        content.removeField("jumpToItem")
    end if
end sub

' will be called when view was closed
sub VideoDetail_CleanUp()
    m.isFirstContentSet = true
    m.contentList = invalid
    m.initIndex = invalid
end sub

sub VideoDetail_CreateButtons(reloadBookmarks = false as Object)
    if m.view.content <> invalid and m.view.content.GetChildCount() > 0 and m.view.itemFocused > -1
        focusedItem = m.view.content.GetChild(m.view.itemFocused)
        position = 0
        if focusedItem.tricksAllowed = true then position = getBookmarkedStartTime(focusedItem)
        if position > 0
            buttons = [{
                title: tr("Resume")
                id: "resume"
                position: position
            },{
                title: tr("Play from beginning")
                id: "play"
            }]
        else
            buttons = [{
                title: tr("Play")
                id: "play"
            }]
        end if
    else
        buttons = [{
            title: tr("Error")
        }]
    end if
    buttonsContent = CreateObject("roSGNode", "ContentNode")
    buttonsContent.Update({children: buttons}, true)
    m.view.buttons = buttonsContent
end sub

sub VideoDetail_StartPlayback(position as Integer)
    selectedItem = m.view.content.GetChild(m.view.itemFocused)
    selectedItem.Update({
        playStart: position
    })
    repeatAllContent = GetVideoRepeatSettings()
    playlistMode = GetVideoPlaylistModeSettings()
    if playlistMode = true
        videoController = ShowVideoScreen(m.view.content, selectedItem, false, repeatAllContent)
    else
        ' creating list with one child
        contentList = CreateObject("roSGNode", "ContentNode")
        ' we should clone selected item to avoid reparenting
        selectedItemCloned = selectedItem.Clone(false)
        contentList.appendChild(selectedItemCloned)
        videoController = ShowVideoScreen(contentList, selectedItemCloned, false, repeatAllContent)
    end if
    m.Observe(videoController.view, "wasClosed", "OnVideoWasClosed")
end sub

sub VideoDetail_OnItemLoaded(detailsView as Object, itemLoaded as Boolean)
    if m.isFirstContentSet and detailsView.hasField("shuffle")
        detailsView.shuffle = GetVideoShuffleSettings()
        m.isFirstContentSet = false
    end if
    m.CreateButtons()
end sub

sub VideoDetail_OnButtonSelected(detailsView as Object, buttonSelected as Integer)
    selectedButton = detailsView.buttons.GetChild(buttonSelected)

    if selectedButton.id = "play"
        m.StartPlayback(0)
    else if selectedButton.id = "resume"
        m.StartPlayback(selectedButton.position)
    end if
end sub

sub VideoDetail_OnDetailsWasShown(detailsView as Object, wasShown as Boolean)
    if m.isShownFirstTime = true
        m.isShownFirstTime = false
    else if detailsView.content <> invalid and detailsView.content.GetChildCount() > 0
        ' reload buttons after playback
        m.CreateButtons()
    end if
end sub

sub VideoDetail_OnVideoWasClosed(videoView as Object, wasClosed as Boolean)
    RSGCallback_RemoveNodeCallback(videoView, "wasClosed")
    if wasClosed
        playlistMode = GetVideoPlaylistModeSettings()
        if playlistMode = true
            ' jump to last playing item if playlistMode is "On", otherwise we have no need to do this, since
            ' we played single video
            m.view.jumpToItem = videoView.currentIndex
        end if
    end if
end sub

sub VideoDetail_ShuffleContent(detailsView as Object, shuffle as Boolean)
    content = detailsView.content
    if shuffle
        size = content.GetChildCount() - 1
        if size > 0
            lastFocusedObjectId = content.GetChild(detailsView.itemFocused).objectId
            'need copy content before shuffling
            if m.unShuffleContent = invalid
                m.unShuffleContent = content.clone(true)
            end if
            shuffledContent = CreateObject("roSGNode", "ContentNode")
            content = content.GetChildren(-1,0)
            while content.Count() > 0
                rndRange = content.Count() - 1
                index = Rnd(rndRange)
                shuffledContent.AppendChild(content[index])
                content.Delete(index)
            end while
            detailsView.content = shuffledContent
            detailsView.jumpToItem = ObjectLocationInList(lastFocusedObjectId, shuffledContent.getChildren(-1,0))
        end if   
    else
        if m.unShuffleContent <> invalid
            lastFocusedObjectId = content.GetChild(detailsView.itemFocused).objectId
            detailsView.content = m.unShuffleContent
            detailsView.jumpToItem = ObjectLocationInList(lastFocusedObjectId, m.unShuffleContent.getChildren(-1,0))
            m.unShuffleContent = invalid
        end if
    end if
end sub

function getBookmarkedStartTime(episode as Object) as Integer
    t = m.bookmarks.GetBookmarkForVideo(episode)
    ' m.bookmarks.dumpbookmarks()
    if (t = 0)
      ' was this from usb?
      ' stop using activevolume because this might be a playlist on usb with http references
      url = episode.url
      if url.left(3) = "ext"
        ' when transitioning to new release from old bookmark, lets try the old mounted place as well
        saveguid = episode.guid
        print "Trying backup bookmark guid=";saveguid;" backup=";url
        episode.guid = url         ' try the old guid
        t = m.bookmarks.GetBookmarkForVideo(episode)
        episode.guid = saveguid
      endif
    endif
    return t
end function
