' ********** Copyright 2020 Roku Corp. All Rights Reserved. **********

sub ShowSearchScreen()
    service = GetServiceByServer(m.lastMediaServer)
    controller = GetBaseViewController("SearchView")
    controller.Append({
        overhangOptionStr: Tr("Search & Options")
        activeMediaServer: m.lastMediaServer
        scene: m.top
        service: service
        searchKeys: m.lastMediaServer.fs.searchKeys
        lastBBSelectedItem: 0

        SetUpScreen: Search_SetUpScreen
        CleanUp: Search_CleanUp
        OnInputText: Search_OnInputText
        DoSearch: Search_DoSearch
        GetQuery: Search_GetQuery
        GetCurrentCapability: Search_GetCurrentCapability
        OnRowItemSelected: ContentScreen_OnItemSelected
        OnVisibleChanged: Search_OnVisibleChanged
        OnContentChanged: Search_OnContentChanged
        GetSearchKeys: Search_GetSearchKeys
        GetSearchKeyList: Search_GetSearchKeyList
        UpdateButtonBar: Search_UpdateButtonBar
        OnBBItemSelected: Search_OnBBItemSelected
    })
    if isUSBOnly() = true
        controller.overhangOptionStr = Tr("Options")
    end if
    controller.Observe(controller.view, "query", "OnInputText")
    controller.Observe(controller.view, "rowItemSelected", "OnRowItemSelected")
    controller.Observe(controller.view, "visible", "OnVisibleChanged")
    controller.Observe(controller.view, "content", "OnContentChanged")
    controller.Observe(controller.scene.buttonBar, "itemSelected", "OnBBItemSelected")
    controller.SetUpScreen()
    ShowNewView(controller.view)
end sub

sub Search_SetUpScreen()
    m.scene.buttonBar.enableFootprint = true
    m.scene.buttonBar.alignment = "top"
    m.scene.buttonBar.footprintStyle = "selection"
    m.UpdateButtonBar(m.GetSearchKeyList())

    'set theme
    m.view.theme = {
        OverhangOptionsText: m.overhangOptionStr
        OverhangOptionsAvailable: false
        OverhangShowOptions: false
    }
    m.SetOverhangTitle(Tr("Search"))
end sub

' will be called when view was closed
sub Search_CleanUp()
    m.scene.buttonBar.visible = false
    m.lastBBSelectedItem = 0
    m.activeMediaServer = invalid
    m.scene = invalid
    m.service = invalid
end sub

sub Search_OnInputText(searchView as Object, query as String)
    if m.activeMediaServer <> invalid and query.Len() > 0
        if m.searchKeys.Count() > 0
            section = "filename"
            if m.scene.buttonBar.visible = true
                section = m.GetCurrentCapability()
            else if m.searchKeys.title <> invalid
                section = "title"
            end if
            m.DoSearch(query, section)
        end if
    else
        ' clear search results
        searchView.content = CreateObject("roSGNode", "ContentNode")
    end if
end sub

sub Search_DoSearch(text as String, section as String)
    content = CreateObject("roSGNode", "ContentNode")
    if text.Len() > 0
        objID = GetSearchRootObjectID(m.activeMediaServer)

        if m.activeMediaServer.accesstype = USBAccessType()
            m.service.cancel = true    
        end if

        config = {
            searchStr: text
            handlerConfigSearch: {
                name: "ContentScreenCH"
                fields: {
                    mediaServerAA: m.activeMediaServer.GetFields()
                    objID: objID
                    service: m.service
                }
            }
        }
        content.Update(config, true)
        ' start CH to get content from service and parse it
        m.view.content = content
        ' start loading using needed service
        StartLoadingDirectory(m.service, m.activeMediaServer, objID, m.GetQuery(text, section))
    else
        m.view.content = content
    end if
end sub

function Search_GetQuery(text as String, section as String) as String
    if m.searchKeys <> invalid and m.searchKeys[section] <> invalid
        return m.searchKeys[section].val + " contains " + quotestr(text)
    end if
    return ""
end function

function Search_GetCurrentCapability()
    if m.scene.buttonBar <> invalid and m.GetSearchKeyList()[m.scene.buttonBar.itemSelected] <> invalid
        return m.GetSearchKeyList()[m.scene.buttonBar.itemSelected].title
    end if
    return "title"
end function

function Search_GetSearchKeyList() as Object
    searchKeyList = []
    for each key in m.searchKeys
        if key = "artist"
            searchKeyList.Unshift({
                title: key
            })
        else
            searchKeyList.Push({
                title: key
            })
        end if
    end for
    return searchKeyList
end function

sub Search_OnVisibleChanged(searchView as Object, visible as Boolean)
    if m.scene <> invalid and Config().Get("enableButtonBar")
        if visible
            m.scene.buttonBar.enableFootprint = true
            m.scene.buttonBar.alignment = "top"
            m.scene.buttonBar.footprintStyle = "selection"
            m.UpdateButtonBar(m.GetSearchKeyList())
            m.scene.buttonBar.jumpToItem = m.lastBBSelectedItem
            m.scene.buttonBar.visible = visible
            'need update observer for this field because when we back
            'from MediaView this field will be un-observing
            m.Observe(m.scene.buttonBar,  "itemSelected", "OnBBItemSelected")
        end if
    end if
end sub

sub Search_OnContentChanged(searchView as Object, content as Object)
    if content.GetChildCount() > 0
        rowAspectRatio = []
        for each row in content.GetChildren(-1, 0)
            if row.type = dirID() or row.type = audioID()
                rowAspectRatio.Push("square")
            else
                rowAspectRatio.Push("16x9")
            end if
        end for
        m.view.rowPosterShapes = rowAspectRatio
    end if
end sub

function Search_GetSearchKeys(caps as Object)
    searchkeys = {}
    allowedfields = {}
    ' not translatable yet, they are used for comparison
    allowedfields.artist = "artist"
    allowedfields.album = "album"
    allowedfields.genre = "genre"
    allowedfields.title = "title"
    allowedfields.filename = "filename"

    print "Capabilities:";caps
    fields = caps.tokenize(",")

    keycount% = 0
    for each field in fields
        colonpos = field.instr(0,":")
        if colonpos >= 0
            key = field.mid(colonpos+1)
            if allowedfields.doesexist(key)
                searchkeys[key] = { name: allowedfields[key], index: keycount%, val:field }
                print "Add searchkey name=";key;" val=";field
                keycount% = keycount% + 1
            end if
        end if
    next
    return searchkeys
end function

sub Search_UpdateButtonBar(buttonsList as Object)
    if buttonsList.Count() > 0
        m.scene.buttonBar.visible = true
        content = CreateObject("roSGNode", "ContentNode")
        for each buttonAA in buttonsList
            button = content.CreateChild("ContentNode")
            button.Update(buttonAA, true)
        end for
        m.scene.buttonBar.content = content
    else
        m.scene.buttonBar.visible = false
    end if
end sub

sub Search_OnBBItemSelected(buttonBar as Object, itemSelected as Integer)
    m.lastBBSelectedItem = m.scene.buttonBar.itemSelected
    if m.view.query <> ""
        m.DoSearch(m.view.query, m.GetCurrentCapability())
    end if
end sub

function GetSearchRootObjectID(volume as Object)
    objID = GetRootObjectID()

    if volume <> invalid and volume.provider <> invalid
        if volume.provider.twonky
            objID = getTwonkySearchRoot(objID)
        else if volume.provider.windows10
            objID = getWin10SearchRoot(objID)
        else if volume.provider.miniDLNA
            objID = getMiniDLNASearchRoot(objID)
        end if
    end if

    return objID
end function

function getTwonkySearchRoot(ID as string)
'Found response u:BrowseResponse
'numreturned=11 TotalMatches=11 UPDT ID:30
'found container=Album count= (67) id=0$1$12
'found container=All Tracks count= (1999) id=0$1$8
'found container=Artist count= (39) id=0$1$11
'found container=Artist Index count= (7) id=0$1$15
'found container=Artist/Album count= (39) id=0$1$16
'found container=By Folder count= (3) id=0$1$13
'found container=Composer count= (12) id=0$1$19
'found container=Genre/Album count= (15) id=0$1$17
'found container=Genre/Artist/Album count= (15) id=0$1$18
'found container=Genre/Song count= (15) id=0$1$10
'found container=Playlists count= (105) id=0$1$9
    mediatype = getMTFilter()
    if mediatype = audioID()
        ' for audio go straight to the music/all music folder
        if ID="0" or ID="0$1"
            id = "0$1$8"
        end if
    else if mediatype = videoID()
        if ID="0" or ID="0$3"
            id = "0$3$27"
        end if
    else if mediatype = pictureID()
        if ID="0" or ID="0$2"
            id = "0$2$20"
        end if
    else
        ' TODO only do this at top two levels
        if ID="0" or ID="0$1" or ID="0$3" or ID="0$2"
            id = "0$1$8"   ' default to audio only for all
        endif
    end if
    return id
end function

function getWin10SearchRoot(ID as string)
'numreturned=4 TotalMatches=4 UPDT ID:82
'found container=Music count= (10) id=1
'found container=Pictures count= (7) id=3
'found container=Playlists count= (2) id=12
'found container=Videos count= (8) id=2

'numreturned=10 TotalMatches=10 UPDT ID:82
'found container=Album Artists count= (68) id=107
'found container=Albums count= (84) id=7
'found container=All Artists count= (152) id=6
'found container=All Music count= (676) id=4
'found container=Composers count= (364) id=108
'found container=Contributing Artists count= (152) id=100
'found container=Folders count= (1) id=14
'found container=Genre count= (23) id=5
'found container=Music Playlists count= (0) id=F
'found container=Rating count= (5) id=101
'filescount= 10 count= 20 startI= 10

    mediatype = getMTFilter()
    if mediatype = audioID()
        ' for audio go straight to the music/all music folder
        if ID="0" or ID="1"
            id = "4"
        end if
    else if mediatype = videoID()
        if ID="0" or ID="2"
            id = "8"
        end if
    else if mediatype = pictureID()
        if ID="0" or ID="3"
            id = "12"
        end if
    else
        id = "4"   ' default to audio only for all
    end if
    return id
end function

function getMiniDLNASearchRoot(ID as string)
    mediatype = getMTFilter()
    if mediatype = audioID()
        ' for audio go straight to the music/all music folder
        if ID="0" or ID="1"
            id = "1$4"
        end if
    else if mediatype = videoID()
        if ID="0" or ID="2"
            id = "2$8"
        end if
    else if mediatype = pictureID()
        if ID="0" or ID="3"
            id = "3$B"
        end if
    else
        id = "1$4"   ' default to audio only for all
    end if
    return id
end function
