' ********** Copyright 2020 Roku Corp. All Rights Reserved. **********

sub PlayDir(volume as Object, objID as String)
    'save current media server to scene scope
    m.lastMediaServer = volume
    
    dirPlayer = GetBaseViewController("GridView")
    dirPlayer.Append({
        volume: volume
        objID: objID
        service: GetServiceByServer(volume)
        loadedFolders: []
        dirsToPlay: []
        recurseLvl: 0
        currItemIndex: 0
        overhangOptionStr: Tr("Search & Options")

        Run: DirPlayer_Run
        EndPlayback: PlayDir_EndPlayback
        LoadDir: DirPlayer_LoadDir
        PlayDir: DirPlayer_PlayDir
        GoOneDirBackward: DirPlayer_GoOneDirBackward
        ShowContent: DirPlayer_ShowContent
        PlayNext: DirPlayer_PlayNext
        NeedToRepeat: DirPlayer_NeedToRepeat
        OnDirLoaded: DirPlayer_OnDirLoaded
        OnPlayerViewClosed: DirPlayer_OnPlayerViewClosed
        CleanUp: PlayDir_CleanUp
        ShowLoadingFacade: DirPlayer_ShowLoadingFacade
        HideLoadingFacade: DirPlayer_HideLoadingFacade
        OnFacadeWasClosed: DirPlayer_OnFacadeWasClosed
    })
    if isUSBOnly() = true
        dirPlayer.overhangOptionStr = Tr("Options")
    end if
    dirPlayer.Observe(dirPlayer.view, "content", "OnDirLoaded")

    dirPlayer.Run()
end sub

sub PlayDir_CleanUp()
    m.service.progressEnabled = true
    m.volume = invalid
    m.objID = invalid
    m.service = invalid
    m.loadedFolders = invalid
    m.dirsToPlay = invalid
    m.content = invalid
    m.loadingFacade = invalid
end sub

sub PlayVolume(volume as Object)
    root = "0"
    PlayDir(volume, root)
end sub

sub DirPlayer_Run()
    ShowNewView(m.view)
    m.LoadDir(m.objID)
end sub

sub PlayDir_EndPlayback()
    m.HideLoadingFacade() ' make sure that facade is closed
    m.view.close = true
end sub

sub DirPlayer_OnDirLoaded(gridView as Object, content as Object)
    if content.failedLoading = false
        if content.GetChildCount() > 0
            ' the content change can be trigered more than once for same content node
            ' we should check if this is actual content for new folder
            ' to not start playing the same folder over and over
            if m.content = invalid or not m.content.IsSameNode(content)
                m.content = content 
                m.PlayDir(content)
            end if
        else
            m.GoOneDirBackward()
        end if
    else if content.failedLoading = true
        m.GoOneDirBackward()
    end if
end sub

sub DirPlayer_LoadDir(objId as String)
    m.ShowLoadingFacade()
    content = CreateObject("roSGNode", "ContentNode")
    config = {
        HandlerConfigGrid: {
            name: "ContentScreenCH"
            fields: {
                mediaServerAA: m.volume.GetFields()
                objID: objId
                service: m.service
            }
        }
    }
    content.Update(config, true)
    m.view.viewContentGroup.visible = false
    m.view.theme = {
        OverhangShowOptions: false
    }
    m.view.content = content
    StartLoadingDirectory(m.service, m.volume, objID)
end sub

sub DirPlayer_GoOneDirBackward()
    if m.recurseLvl > 0
        m.recurseLvl--
        m.PlayDir(invalid)
    else ' don't find anything to play
        ' check if we need to repeat
        if m.NeedToRepeat()
            m.LoadDir(m.objID)
        else
            m.EndPlayback()
        end if
    end if
end sub

' Descend into directory playing all it's contents
' This code plays the files by just retrieving each file info one at a time so startup is pretty quick
' But it also doesn't know the type of a file until it's played all previous files in that list.
sub DirPlayer_PlayDir(content as Object)
    print "PLAYDIR------------------------------------ objectID=";objectID;"  recurselvl=";m.recurseLvl

    if content <> invalid
        dirs = ConvertGridToListContent(content, false)
        print "call convert gridtolist"
        m.currDirContent = ConvertGridToListContent(content, true)
        if m.loadedFolders[m.recurseLvl] = invalid
            m.loadedFolders[m.recurseLvl] = m.currDirContent
        else
            m.loadedFolders[m.recurseLvl].InsertChildren(m.currDirContent.GetChildren(-1,0), 0)
        end if
    else
        dirs = m.dirsToPlay[m.recurseLvl]
    end if
    content = m.loadedFolders[m.recurseLvl]
    filecount = content.GetChildCount()
    
    if dirs.GetChildCount() > 0 and m.recurseLvl < 20 ' don't go too deep
        m.dirsToPlay[m.recurseLvl] = dirs
        m.recurseLvl++    ' if we had directories to drop down into, disable repeat for rest of content
        dir = dirs.GetChild(0)
        print "descending into ";dir.objectID
        m.LoadDir(dir.objectID)
        dirs.RemoveChild(dir)
    else if filecount > 0
        m.HideLoadingFacade()
        mixedMedia = false
        if getMTFilter() = allMediaId()
            contentType = content.GetChild(0).type
            for each item in content.GetChildren(-1,0)
                if item.type <> contentType
                    mixedMedia = true
                    exit for
                end if
            next
        endif
        if mixedMedia
            ' in mixed media, we play one item at a time so that we can play all the different types of items
            ' also replay is disabled for all.
            m.PlayNext()
        else  ' nomixed media here
            itemToPlay = content.GetChild(0)
            contentType = itemToPlay.type
            mediaList = CreateObject("roSGnode", "ContentNode")
            if contentType = videoID()
                mediaList.AppendChildren(content.GetChildren(-1,0))
            else ' img or audio
                row = mediaList.CreateChild("ContentNode")
                row.AppendChildren(content.GetChildren(-1,0))
            end if
            ' not the mixed content, no subdirectories, need to repeat slideshow 
            isRepeatSlideShow = m.recurseLvl = 0 
            m.ShowContent(contentType, mediaList, itemToPlay, isRepeatSlideShow)
            m.loadedFolders.Delete(m.recurseLvl)
        end if
        if m.recurseLvl > 0 then m.recurseLvl--
    else ' going backward
        m.GoOneDirBackward()
    end if
end sub

sub DirPlayer_ShowContent(contentType as Integer, mediaList as Object, item as Object, isRepeatSlideShow = false as Boolean)
    if contentType = videoID()
        if GetVideoShuffleSettings()
            item.Update({
                needToFind: false
            },true)
        end if
        controller = ShowVideoScreen(mediaList, item, false, false, GetVideoShuffleSettings())
    else if contentType = audioID()
        fileCount = mediaList.GetChild(0).GetChildCount()
        if GetAudioShuffleSettings()
            if fileCount > 1
                itemIndex = Rnd(fileCount - 1)
                item = mediaList.GetChild(0).GetChild(itemIndex)
                item.Update({
                    itemIndex: itemIndex
                },true)
            end if
            item.Update({
                needToFind: false
            },true)
        end if
        controller = ShowAudioPlayerScreen(mediaList, item)
    else if contentType = pictureID()
        ' isRepeatSlideShow determines if slideshow will repeat. We pass true 
        ' if there is no mixed content (only images), the directory itself is
        ' not nested (recurse level 0) and there are no nested directories in
        ' it; otherwise we pass false
        controller = ShowSlideShowScreen(mediaList, item, isRepeatSlideShow)
    end if
    controller.orig_CleanUp = controller.CleanUp
    controller.CleanUp = DirPlayer_OnPlayerViewClosed
    controller.playerController = m
end sub

sub DirPlayer_PlayNext()
    content = m.loadedFolders[m.recurseLvl]
    item = content.GetChild(0)
    if item <> invalid
        contentType = item.type
        mediaList = CreateObject("roSGnode", "ContentNode")
        if contentType = videoID()
            mediaList.AppendChild(item)
        else ' img or audio
            row = mediaList.CreateChild("ContentNode")
            row.AppendChild(item)
        end if
        if content.GetChild(0) = invalid ' played last item from this level
            m.loadedFolders.Delete(m.recurseLvl)
        end if
        m.ShowContent(contentType, mediaList, item)
    end if
end sub

sub DirPlayer_OnPlayerViewClosed()
    m.orig_CleanUp()
    PCI = m.playerController
    m.playerController = invalid
    ' handle error state, starting next item if available
    if m.view.state = "error" and m.view.content <> invalid
        nextItem = m.view.content.GetChild(m.view.currentIndex+1)
        if nextItem <> invalid ' check if have next item
            PCI.ShowContent(nextItem.type, m.view.content, nextItem)
            return
        end if
    end if

    if m.view.close = false ' user closes playback by pressing back
        PCI.EndPlayback()
    else
        if PCI.dirsToPlay[PCI.recurseLvl] <> invalid and PCI.dirsToPlay[PCI.recurseLvl].GetChildCount() > 0
            PCI.PlayDir(invalid)
        else if PCI.loadedFolders[PCI.recurseLvl] <> invalid and PCI.loadedFolders[PCI.recurseLvl].GetChildCount() > 0
            PCI.PlayNext()
        else
            PCI.recurseLvl--
            if PCI.recurseLvl < 0
                if PCI.NeedToRepeat()
                    PCI.recurseLvl = 0
                    PCI.LoadDir(PCI.objID)
                else
                    PCI.EndPlayback()
                end if
            else
                PCI.PlayDir(invalid) ' will iterate in current folder
            end if
        end if
    end if
end sub

function DirPlayer_NeedToRepeat() as Boolean
    repeating = true
    ' check individual repeatall flags
    ' and disable repeating if appropriate
    mType = getMTFilter()
    if mType = videoid()
        if GetVideoRepeatSettings() = false
            repeating = false
        endif
    else if mType = audioid()
        if GetAudioRepeatSettings() = false
            repeating = false
        endif
    else if mType = pictureID()
        if GetPictureRepeatSettings() = false
            repeating = false
        endif
    else 'all
        ' require all repeatall settings to be true
        if not (GetPictureRepeatSettings() and GetAudioRepeatSettings() and GetVideoRepeatSettings())
            repeating = false
        endif
    endif
    return repeating
end function

sub DirPlayer_ShowLoadingFacade()
    if m.loadingFacade = invalid
        m.loadingFacade = CreateObject("roSGNode", "SGDEXComponent")
        m.loadingFacade.theme = { OverhangOptionsText: m.overhangOptionStr }
        facadeComponent = m.loadingFacade.CreateChild("LoadingFacade")
        facadeComponent.bEatKeyEvents = false
        facadeComponent.busySpinner.id = "spinner"

        ShowNewView(m.loadingFacade)
        m.Observe(m.loadingFacade, "wasClosed", "OnFacadeWasClosed")
    end if
    ' disable progress indication for nested folders
    m.service.progressEnabled = (m.recurseLvl = 0)
end sub

sub DirPlayer_HideLoadingFacade()
    if m.loadingFacade <> invalid
        m.loadingFacade.close = true
        m.loadingFacade = invalid
    end if
end sub

sub DirPlayer_OnFacadeWasClosed(loadingFacade as Object, wasClosed as Boolean)
    if loadingFacade.close = false
        ' user closes facade by pressing back
        ' cancel playback, close controller view
        m.view.close = true
    end if
end sub
