' ********** Copyright 2019 Roku Corp. All Rights Reserved. **********

sub HandleVoiceRequest(params as Object)
    ' RMP-565 make sure to close active option dialog if any
    if m.top.dialog <> invalid
        m.top.dialog.close = true
        m.top.dialog = invalid
    end if

    voiceHandler = {
        isDebug: false  'set to true to enable verbose debug prints
        
        params: params
        scene: m.top
        activeVolumes: []

        Run: VoiceHandler_Run
        HandleMusicRequest: VoiceHandler_HandleMusicRequest
        HandleControlCommand: VoiceHandler_HandleControlCommand
        SearchVolume: VoiceHandler_SearchVolume
        OnSearchResults: VoiceHandler_OnSearchResults
        PostToVoiceAdapter: VoiceHandler_PostToVoiceAdapter
        GetNowPlayingString: VoiceHandler_GetNowPlayingString
        ShowLoadingFacade: VoiceHandler_ShowLoadingFacade
        CloseLoadingFacade: VoiceHandler_CloseLoadingFacade
        OnActiveVolumesChanged: VoiceHandler_OnActiveVolumesChanged
        DoCleanup: VoiceHandler_DoCleanup
        OnStartupTimeout: VoiceHandler_OnStartupTimeout
        DebugPrint: VoiceHandler_DebugPrint
    }
    voiceHandler.activeVolumes.Append(m.top.usb.activeVolumes)
    voiceHandler.activeVolumes.Append(m.top.dlna.activeVolumes)

    voiceHandler.DebugPrint("HandleVoiceRequest()", params)

    voiceHandler.Run()
end sub

sub VoiceHandler_DebugPrint(msg as Object, obj = invalid as Object)
    if m.isDebug = true
        ?
        ?"-------- DEBUG --------"
        ?msg;
        if obj = invalid
            ?""
        else
            ?obj
        end if
        ?"-----------------------"
        ?
    end if
end sub

sub VoiceHandler_DoCleanup()
    m.DebugPrint("VoiceHandler_DoCleanup()")
    
    ' empty cached volumes list
    m.activeVolumes = []
    
    ' unregister callbacks
    gm = GetGlobalAA()
    RSGCallback_RemoveNodeCallback(gm.top.usb, "activeVolumes")
    RSGCallback_RemoveNodeCallback(gm.top.dlna, "activeVolumes")
    
    ' restore grid format if needed
    if m.needToRestoreGridFormat = true
        setContentDisplayFormatSetting(0)
    end if
    
    ' close facade if any
    m.CloseLoadingFacade()
end sub

sub VoiceHandler_Run()
    m.DebugPrint("VoiceHandler_Run()")

    if m.params.json <> invalid
        m.intent = parseIntent(m.params.json)
        if m.intent.intent = "music.request"
            if not isUSBOnly()
                m.HandleMusicRequest()
            else
                m.PostToVoiceAdapter({hasError: "true"})
            end if
        else if m.intent.command = "launch"
            if m.intent.entities.action = "open" and not isUSBOnly()
                ' play something request
                ' remove all entities so we will get all audio content
                m.intent.entities = invalid
                m.HandleMusicRequest()
            else
                m.PostToVoiceAdapter({hasError: "false"})
            end if
        end if
    else if m.params.control_command <> invalid
        ' Handle control command from voice adapter.
        print "----------control_command----------"
        print m.params
        m.HandleControlCommand(m.params)
    else if m.params.command = "start_playback"
        currView = GetCurrentView()
        if currView <> invalid
            currView.control = "play"
        end if
    end if
end sub

sub VoiceHandler_HandleMusicRequest()
    m.ShowLoadingFacade()

    if m.searchResults <> invalid
        ' break if already found some results
        m.DebugPrint("!! Already found some results !!")
        m.DoCleanup()
        return
    end if
    
    ' register callbacks for updating cached active volumes
    gm = GetGlobalAA()
    RSGCallback_RegisterNodeCallback(gm.top.usb, "activeVolumes", m, "OnActiveVolumesChanged")
    RSGCallback_RegisterNodeCallback(gm.top.dlna, "activeVolumes", m, "OnActiveVolumesChanged")
    
    m.DebugPrint("ACTIVE VOLUMES COUNT: ", m.activeVolumes.Count())
    
    i = 0
    while i < m.activeVolumes.Count()
        ?"i = "i
    
        vol = m.activeVolumes[i]
        print "look at:";vol.label
        fs = vol.fs
        provider = vol.provider
        if fs <> invalid and provider <> invalid and provider.searchable = true
            preferredIntent = GetDLNAFavorite()
            if preferredIntent <> ""
                ' if user has set a preferred DLNA server, limit DLNA searches to that DLNA server
                if vol.accesstype <> USBAccessType()
                    if preferredIntent <> provider.USN
                        fs.searched = true
                    end if
                end if
            end if

            if not fs.searched
                print "Intent Checking volume ";vol.label
                fs.searched = true ' prevent it from being searched again
                m.SearchVolume(vol, m.intent)
            else
                m.activeVolumes.Delete(i)
                i--
                print "Skipped already searched volume:";vol.label
            end if
        else
            m.activeVolumes.Delete(i)
            i--
            print "Skipped not searchable volume:";vol.label
        end if
        i++
    end while
end sub

sub VoiceHandler_HandleControlCommand(info as Object)
    cmd = info.control_command
    handled = false
    print "Handling control command " ; cmd
    currentView = GetCurrentView()
    if currentView <> invalid
        if cmd = "play" then
            if currentView.HasField("control")
                if currentView.HasField("state") and currentView.state = "paused"
                    currentView.control = "resume"
                else
                    currentView.control = "play"
                end if
                handled = true
            end if
        else if cmd = "pause" then
            if currentView.HasField("control")
                currentView.control = "pause"
                handled = true
            end if
        else if cmd = "play_pause" then
            ' if we get here, nothing is actually playing, we need to initiate playing anything
            if currentView.HasField("control")
                currentView.control = "play"
                handled = true
            end if
        else if cmd = "skip" then
            viewType = currentView.Subtype()
            if viewType = "MediaView" or viewType = "SlideShowView" and currentView.content <> invalid
                index = currentView.currentIndex
                index++
                if currentView.content.GetChildCount() > index
                    currentView.jumpToItem = index
                    handled = true
                end if
            end if
        else if cmd = "forward" then
            viewType = currentView.Subtype()
            if viewType = "MediaView"
                seekValue = currentView.position + info.val.ToInt()
                currentView.seek = seekValue
            end if
        else if cmd = "rewind" then
            viewType = currentView.Subtype()
            if viewType = "MediaView"
                seekValue = currentView.position - info.val.ToInt()
                if seekValue < 0 then seekValue = 0
                currentView.seek = seekValue
            end if
        else if cmd = "stop" then
            if currentView.HasField("control")
                currentView.control = "stop"
                handled = true
            end if
        else if cmd = "startover" then
            viewType = currentView.Subtype()
            if viewType = "MediaView"
                currentView.control = "stop"
                currentView.control = "play"
                handled = true
            end if
        else
            control = "unknown"
        end if
    end if

    if handled
        m.postToVoiceAdapter({hasError:"false", control:cmd})
    else
        m.postToVoiceAdapter({hasError:"true", control:cmd})
    end if
end sub

sub VoiceHandler_SearchVolume(volume as Object, intent as Object)
    m.DebugPrint("VoiceHandler_SearchVolume()")

    service = GetServiceByServer(volume)

    voiceSearch = {
        activeMediaServer: volume
        intent: intent
        service: service
        voiceHandler: m ' save voice controller ref to collect search results

        GetSearchKeys: Search_GetSearchKeys
        OnSearchCapabilitiesChanged: VoiceSearch_OnSearchCapabilitiesChanged
        IntentSearch: VoiceSearch_IntentSearch
        DoSearch: VoiceSearch_DoSearch
    }
    if volume.searchKeys = invalid
        RSGCallback_RegisterNodeCallback(voiceSearch.service, "searchCapabilities", voiceSearch, "OnSearchCapabilitiesChanged")
        service.getSearchCapabilities = {
            mediaServer: volume
        }
    else
        m.DoSearch()
    end if
end sub

sub VoiceSearch_OnSearchCapabilitiesChanged(service as Object, searchCapabilities as String)
    RSGCallback_RemoveNodeCallback(m.service, "searchCapabilities")
    m.searchCapabilities = searchCapabilities
    m.searchKeys = m.GetSearchKeys(searchCapabilities)
    currFS = m.activeMediaServer.fs
    currFS.searchKeys = m.searchKeys
    m.activeMediaServer.fs = currFS
    m.DoSearch()
end sub

sub VoiceSearch_DoSearch()
    searchStr = m.IntentSearch(m.activeMediaServer, m.intent) ' convert intent to dlna search string
    if searchStr <> ""
        m.voiceHandler.DebugPrint(" VoiceSearch_DoSearch() :: searchStr = ",searchStr)
    
        ' RMP-539: we should switch to list format to make sure that we get results in one row
        if getContentDisplayFormatSetting() <> ListFormatID()
            m.voiceHandler.needToRestoreGridFormat = true
            setContentDisplayFormatSetting(ListFormatID())
        end if
        objID = GetSearchRootObjectID(m.activeMediaServer)
        parserTask = CreateObject("roSGNode", "ContentScreenCH")
        parserTask.Update({
            mediaServerAA: m.activeMediaServer
            objID: objID
            service: m.service
            content: CreateObject("roSGNode", "ContentNode")
        })
        parserTask.control = "run"
        RSGCallback_RegisterNodeCallback(parserTask, "content", m.voiceHandler, "OnSearchResults")
        StartLoadingDirectory(m.service, m.activeMediaServer, objID, searchStr)
    end if
end sub

sub VoiceHandler_OnSearchResults(parserTask as Object, searchResults as Object)
    RSGCallback_RemoveNodeCallback(parserTask, "content")
    for i = 0 to m.activeVolumes.Count() - 1
        ' remove searched volumes
        if m.activeVolumes[i].usn = parserTask.mediaServerAA.usn
            parserTask.mediaServerAA.fs.searched = false
            m.activeVolumes.Delete(i)
            exit for
        end if
    end for
    
    ' handle case when results can be already found on another volume
    if m.searchResults <> invalid
        m.DebugPrint("ALREADY FOUND SOME RESULTS EARLIER, EXIT")
        return
    end if

    if searchResults.failedLoading = false and searchResults.GetChildCount() = 1
        if m.searchResults = invalid
            res = searchResults.GetChild(0)
            if res <> invalid and res.GetChildCount() > 0
                ' found some results!
                m.searchResults = res
                
                ' disable further search
                m.DebugPrint("!!!!!!!!  FOUND  !!!!!!!!!!")
                m.DoCleanup()
            end if
        end if
    end if

    ' no more volumes to process?
    if m.activeVolumes.Count() = 0
        m.CloseLoadingFacade()

        if m.searchResults <> invalid
            ' closing all views till volume selection screen
            ' to open new media screen correctly
            if GetMTSelectionScreenDisabled()
                CloseAllViewsTillID("vsscreen")
            else
                CloseAllViewsTillID("mtscreen")
                ShowVolumeSelectionScreen()
            end if
            itemToPlay = m.searchResults.GetChild(0)
            ' display media playback screen
            if m.searchResults.type = videoID()
                'respect shuffle settings, same as for PlayDir
                if GetVideoShuffleSettings()
                    itemToPlay.Update({
                        needToFind: false
                    },true)
                end if
                ShowVideoScreen(m.searchResults, itemToPlay, true)
            else if m.searchResults.type = audioID()
                'respect shuffle settings, same as for PlayDir
                if GetAudioShuffleSettings()
                    itemToPlay.Update({
                        needToFind: false
                    },true)
                end if
                ShowAudioPlayerScreen(m.searchResults, itemToPlay, true)
            else if m.searchResults.type = pictureID()
                ShowSlideShowScreen(m.searchResults, itemToPlay)
            else if m.searchResults.type = dirID()
                ' play using PlayDir
                m.DebugPrint("USING PLAYDIR")
                
                cn = createobject("roSGNode","ContentNode")
                cn.Update(parsertask.mediaServerAA, true)
                
                PlayDir(cn, itemToPlay.objectId)
            end if
            m.PostToVoiceAdapter({nowPlaying: m.GetNowPlayingString(), hasError: "false"})
        else
            ' no results so far..
            msSinceLaunch = GetGlobalAA().startupTimer.TotalMilliseconds()
            msTimeout = 10000
            ' 10 seconds or more from channel launch?
            if msSinceLaunch >= msTimeout
                m.DebugPrint("....... NO RESULTS FOUND......")
            
                ' yes, disable further search, report error
                m.DoCleanup()
                m.PostToVoiceAdapter({hasError: "true"})
            else
                ' no, retry search
                m.DebugPrint("....  NO RESULTS SO FAR BUT <10s FROM LAUNCH, SO RETRY HandleMusicRequest!")

                m.HandleMusicRequest()
                
                ' spin up a timer node to check results in X seconds left till 10 since launch
                m.DebugPrint("msTimeout - msSinceLaunch = ", msTimeout - msSinceLaunch)
                
                m.startupTimoutTimer = CreateObject("roSGNode", "Timer")
                m.startupTimoutTimer.duration = (msTimeout - msSinceLaunch)/1000
                RSGCallback_RegisterNodeCallback(m.startupTimoutTimer, "fire", m, "OnStartupTimeout")
                m.startupTimoutTimer.control = "start"
            end if
        end if
    end if
end sub

sub VoiceHandler_OnStartupTimeout(a, b)
    ' still no results even after timeout?
    if m.searchResults = invalid
        m.DebugPrint("VoiceHandler_OnStartupTimeout() :: TOO BAD, NOTHING FOUND")

        ' disable further search, close facade, report error
        m.DoCleanup()
        m.PostToVoiceAdapter({hasError: "true"})
    end if
end sub

sub VoiceHandler_PostToVoiceAdapter(params as Object)
    task = CreateObject("roSGNOde", "VoiceSayTask")
    task.params = params
    task.control = "run"
end sub

function VoiceHandler_GetNowPlayingString() as String
    nowPlayingStr = ""

    if (m.intent <> invalid and m.intent.entities <> invalid)
        nowPlayingStr = FormatJSON(m.intent.entities)
    else
        nowPlayingStr = FormatJSON({music_track: "something"})
    end if

    return nowPlayingStr
end function

sub VoiceHandler_ShowLoadingFacade()
    if m.loadingFacade = invalid
        m.loadingFacade = CreateObject("roSGNode", "GridView")
        
        ' enable spinner so it could be updated with progress indicator behind the scenes 
        m.loadingFacade.showSpinner = true
        
        m.loadingFacade.theme = { OverhangShowOptions: false }
        ShowNewView(m.loadingFacade)
    end if
end sub

sub VoiceHandler_CloseLoadingFacade()
    if m.loadingFacade <> invalid
        m.loadingFacade.close = true
        m.loadingFacade = invalid
    end if
end sub

' return dlna search string corresponding to Intent
function VoiceSearch_IntentSearch(vol, intentAA) as String
    fs = vol.fs
    if fs = invalid
        return ""
    endif

    mediatype = getMTFilter()
    if mediatype = audioID()    
        searchStr = "(upnp:class derivedfrom "+ quotestr("object.item.audioItem") +")"
    else if mediatype = videoID()
        searchStr = "(upnp:class derivedfrom "+ quotestr("object.item.videoItem") +")"
    else
        searchStr = "(upnp:class derivedfrom "+ quotestr("object.item.audioItem") +")"
    endif

    entities = intentAA.entities
    if entities = invalid
        print "no entites - get all audio"
        return searchStr
    endif

    searchKeys = fs.searchKeys

    if searchkeys.count()=0
        print "No keys available for searching, return now"
        vol.provider.Searchable = false  ' prevent it from being searched again
        return ""
    endif

    IntentToDlnaMap = {}
    IntentToDlnaMap.music_artist = "artist"
    IntentToDlnaMap.artist = "artist"
    IntentToDlnaMap.music_album = "album"
    IntentToDlnaMap.music_genre = "genre"
    IntentToDlnaMap.music_track = "title"
    IntentToDlnaMap.podcast_title = "title"
    IntentToDlnaMap.tv_title = "title"
    IntentToDlnaMap.movie_title = "title"

    andStr = " and "

    for each entityVal in entities
        searchFor = entities.Lookup(entityVal)
        dlnaKey = IntentToDlnaMap.lookup(entityVal)
        if IsNullOrEmpty(searchFor) or IsNullOrEmpty(dlnaKey) or searchKeys[dlnaKey]=invalid
            print "Intent lookup failure for :";entityVal
        else
            if dlnaKey = "genre"
                ' Look for a redundant Music in genre
                q = searchFor.instr(" music")
                if q >= 0
                    searchFor = searchFor.left(q)
                endif
                ' rock is a better search target than rock&roll
                if searchFor = "rock and roll" or searchFor = "classic rock"
                    searchFor = "rock"
                endif
            endif
            searchStr = searchStr + andStr + searchKeys[dlnaKey].val + " contains " + quotestr(searchFor)
            'andstr = " and"
        endif
    next
    print "INTENT-to-DLNA SEARCH:";searchStr
    return searchStr
end function

sub VoiceHandler_OnActiveVolumesChanged(a, b)
    m.activeVolumes = []
    m.activeVolumes.Append(m.scene.usb.activeVolumes)
    m.activeVolumes.Append(m.scene.dlna.activeVolumes)
    if m.activeVolumes.Count() > 0
        m.DebugPrint("^^^ NEW VOLUMES DETECTED, RETRY HandleMusicRequest!")
        m.HandleMusicRequest()
    end if
end sub