' ********** Copyright 2019 Roku Corp. All Rights Reserved. **********

sub Show(args as Object)
    m.startupTimer = CreateObject("roTimespan")

    ShowFakeSplash()
    m.args = args
    m.localizer = CreateObject("roLocalization")
    m.bookmarks = NWM_Bookmarks()

    ' RMP-515: don't alow channel to exit on closing of last view
    m.top.componentController.allowCloseChannelOnLastView = false
    SetupGlobalTheme()

    ReadConfig()
end sub

sub Input(args as Object)
    HandleVoiceRequest(args)
end sub

sub SetupGlobalTheme()
    m.top.backgroundURI = "pkg:/images/background.jpg"
    m.top.backgroundColor = "0x140025ff"
    darkOverlay = CreateObject("roSGNode", "Rectangle")
    darkOverlay.Update({
        width: 1280
        height: 720
        color: "0x000000"
        opacity: 0.4
    })
    m.top.InsertChild(darkOverlay, 1)
    m.top.theme = {
        global: {
            textColor: "0xc0c0c0ff"
            busySpinnerColor: "0xc0c0c0ff"

            OverhangOptionsAvailable: true
            OverhangBackgroundColor: "0x00000000"
            ' OverhangTitleColor: "0xffffff"
            OverhangLogoUri: "pkg:/images/rmp_logo.png"
            OverhangleftDividerUri: "NONE" ' is this the only way to hide it?
            OverhanglogoBaselineOffset: 5

            ' backgroundImageURI: "pkg:/images/background.jpg"
            ' backgroundColor: "0x140025"

            focusRingColor: "0xc0c0c0ff"

            itemTextBackgroundColor: "0x00000080"
            buttonsFocusedColor: "0x000000" 'text of focused button
        }
        gridView: {
            itemBackgroundColor: "0x97979719"
        }
        searchView: {
            itemBackgroundColor: "0x97979733"
        }
        detailsView: {
            posterBackgroundColor: "0x97979733"
        }
        buttonBar: {
            buttonColor: "0xffffff20"
            footprintButtonColor: "0xffffff40"

            buttonTextColor: "0xffffff80"
            focusedButtonTextColor: "0x000000"
            footprintButtonTextColor: "0xffffff80"
        }
    }
end sub

sub ShowFakeSplash()
    m.top.Update({
        children: [{
            subtype: "Poster"
            id: "fakeSplash"
            height: 720
            width: 1280
            uri: "pkg:/locale/default/images/splash_usb_hd.jpg"
        }]
    })
end sub

sub CloseFakeSplash()
    fakeSplash = m.top.FindNode("fakeSplash")
    if fakeSplash <> invalid
        m.top.RemoveChild(fakeSplash)
        fakeSplash = invalid
    end if
end sub

sub ReadConfig()
    configReaderTask = m.top.CreateChild("ConfigReader")
    configReaderTask.ObserveFieldScoped("done", "OnConfigRead")
    configReaderTask.control = "run"
end sub

sub OnConfigRead(event as Object)
    readerTask = event.GetRoSGNode()
    readerTask.UnobserveFieldScoped("done")
    m.top.RemoveChild(readerTask)
    if m.args["source"] = "usb-media-auto-run"
        Config().AddSet("launchVolume", m.args["usb_media_auto_run"])
    end if
    InitBackgroundServices()
end sub

sub InitBackgroundServices()
    m.top.usb = m.top.CreateChild("USB")
    m.top.usb.ObserveFieldScoped("initialized", "OnServiceInitialized")
    m.top.dlna = m.top.CreateChild("DLNA")
    m.top.dlna.ObserveFieldScoped("initialized", "OnServiceInitialized")
    m.top.httpfs = m.top.CreateChild("HTTPFS")
    m.top.httpfs.ObserveFieldScoped("initialized", "OnServiceInitialized")
end sub

sub OnServiceInitialized()
    if m.top.usb.initialized and m.top.dlna.initialized and m.top.httpfs.initialized
        if GetMTSelectionScreenDisabled()
            ShowVolumeSelectionScreen()
        else
            ShowHomeScreen()
        end if
        m.top.SignalBeacon("AppLaunchComplete")
        CloseFakeSplash()
        if m.args <> invalid and m.args.source = "voice-adapter"
            HandleVoiceRequest(m.args)
        end if
    end if
end sub

function onKeyEvent(key as String, press as Boolean) as Boolean
    handled = false

    if press
        if key = "options"
            currView = GetCurrentView()
            if currView.id = "mtscreen" or currView.id = "vsscreen"
                ' help and settings available on media and volume selection screens
                ShowSettings()
            else if (currView.id.InStr("ContentScreen") > -1)
                ' search and options available on content and details screens
                ShowOptions()
            else if (currView.Subtype() = "DetailsView")
                ShowVideoDetailsDialog()
            else if (currView.Subtype() = "SlideShowView")
                ShowSlideShowSettings(currView)
            end if
        else if key = "play"
            currView = GetCurrentView()
            if (currView.id.InStr("ContentScreen") > -1)
                item = invalid
                if currView.Subtype() = "GridView"
                    item = GetGridItem(currView.content, currView.rowItemFocused)
                else if currView.Subtype() = "CategoryListView"
                    item = GetGridItem(currView.content, [currView.focusedCategory, currView.focusedItemInCategory])
                end if

                if item <> invalid
                    if item.type = DirId()
                        PlayDir(m.lastMediaServer, item.objectId)
                    else if item.type = VideoId()
                        item.playStart = getBookmarkedStartTime(item)
                        ShowVideoScreen(currView.content, item, false, GetVideoRepeatSettings(), GetVideoShuffleSettings())
                    else if item.type = AudioId()
                        item.playStart = ReadAudioBookmark(item)
                        ShowAudioPlayerScreen(currView.content, item)
                    else if item.type = PictureID()
                        ShowSlideShowScreen(currView.content, item)
                    end if
                end if
            else if (currView.Subtype() = "DetailsView" and currView.content.GetChildCount() > 0)
                item = currView.content.GetChild(currView.itemFocused)
                item.playStart = getBookmarkedStartTime(item)
                playlistMode = GetVideoPlaylistModeSettings()
                if playlistMode = true
                    ShowVideoScreen(currView.content, item, false, GetVideoRepeatSettings())
                else
                    ' creating list with one child
                    contentList = CreateObject("roSGNode", "ContentNode")
                    ' we should clone selected item to avoid reparenting
                    itemCloned = item.Clone(false)
                    contentList.appendChild(itemCloned)
                    ShowVideoScreen(contentList, itemCloned, false, GetVideoRepeatSettings())
                end if
            else if currView.id = "vsscreen"
                item = GetGridItem(currView.content, currView.rowItemFocused)
                if item <> invalid
                    PlayVolume(item)
                end if
            end if
        else if key = "down"
            currView = GetCurrentView()
            if (currView <> invalid and currView.Subtype() = "SlideShowView")
                ShowSlideShowSettings(currView)
            end if
        end if
    end if

    return handled
end function

' Helper function for reading audio bookmark.
function ReadAudioBookmark(episode as Object) as Integer
    result = 0
    if episode <> invalid
        isUSB = (m.lastmediaserver <> invalid and m.lastmediaserver.accessType = USBAccessType())

        ' not a FLAC format or FLAC from USB drive?
        if episode.streamformat <> "flac" or isUSB
            ' get bookmark but convert to seconds (legacy RMP stores it in milliseconds
            ' and we do the same for backward compatibility)
            result = getBookmarkedStartTime(episode) / 1000
        end if

        ' make sure to disable bookmark indication on the grid for this piece of content
        episode.Update({ hideItemDurationBar: true }, true)
    end if
    return result
end function

' Helper sub for writing audio bookmark
sub WriteAudioBookmark(episode as Object, position = 0 as Object, isDebug = false as Object)
    if episode <> invalid
        isUSB = (m.lastmediaserver <> invalid and m.lastmediaserver.accessType = USBAccessType())

        ' not a FLAC format or FLAC from USB drive?
        if episode.streamformat <> "flac" or isUSB
            ' get new playstart value
            ps = position
            if ps = invalid
                ps = 0
            end if

            ' not the same value?
            if episode.playStart <> ps
                ' update episode playStart
                episode.playStart = ps

                if isDebug = true
                    ?
                    ?"WriteAudioBookmark :: "ps", guid = "episode.guid
                    ?
                end if

                ' we should save bookmarks for audio in milliseconds
                ' for backward compatibility with legacy RMP app
                episodeAA = {
                    guid: episode.guid
                    playStart: ps * 1000
                }
                NWM_Bookmarks().UpdateBookmarkForVideo(episodeAA)
            end if
        end if
    end if
end sub

' function for getting service for work with this server
' @Params: server - media server object
' @Return value: service object (usb/dlna/https) or invalid
function GetServiceByServer(server as Object) as Object
    service = invalid
    if server <> invalid
        scene = m.top
        if server.accessType = USBAccessType()
            service = scene.usb
        else if server.accessType = DLNAAccessType()
            service = scene.dlna
        else if server.accessType = HTTPAccessType()
            service = scene.httpfs
        end if
    end if
    return service
end function

' function for getting current visible view
function GetCurrentView() as Object
    scene = m.top.GetScene()
    currView = scene
    if scene.componentController <> invalid and scene.componentController.currentView <> invalid
        currView = scene.componentController.currentView
    end if
    return currView
end function

' function for closing all views showed after specific one
' @Params: id - id of the view till which we will close views
' @Params: shouldAccessScene - flag indicates whether we should access scene to find component controller,
'       because m.top.GetScene() returns screensaver's scene in some cases
sub CloseAllViewsTillID(id as String, shouldAccessScene = true as Boolean)
    if shouldAccessScene = true
        currentView = GetCurrentView()
        while (currentView <> invalid and currentView.id <> id)
            currentView.close = true
            currentView = GetCurrentView()
        end while
    else
        componentController = m.top.componentController
        if componentController <> invalid
            while (componentController.currentView <> invalid and componentController.currentView.id <> id)
                componentController.currentView.close = true
            end while
        end if
    end if
end sub

' function for showing new already created view
' @Params: view - view to be shown
sub ShowNewView(view as Object)
    ' this will trigger job to show this screen
    m.top.componentController.CallFunc("show", {
        view: view
    })
end sub

' function for getting item from 2-level content
' @Params: content - content node with rows and items
' @Params: rowItem - an array containing the index of the row and item
' @Return value: grid item
function GetGridItem(content as Object, rowItem as Object) as Object
    gridItem = invalid

    if content <> invalid
        rowIndex = rowItem[0]
        itemIndex = rowItem[1]

        ' get specified row
        row = content.GetChild(rowIndex)
        if row <> invalid
            ' get item
            gridItem = row.GetChild(itemIndex)
        end if
    end if

    return gridItem
end function

function customSuspend()
    print "CUSTOM SUSPEND FUNCTION"
end function

function customResume()
    print "CUSTOM RESUME FUNCTION"
    task = CreateObject("roSGNOde", "ChannelMonitorTask")
    task.control = "run"

    currentView = m.top.componentController.currentView
    if currentView <> invalid and currentView.subtype() = "MediaView"
        ' save position to compare with future value
        m.resumePosition = currentView.position

        ' set up a timer to compare positions after more than 1s of delay
        m.timer = CreateObject("roSGNode", "Timer")
        m.timer.duration = 1.1
        m.timer.ObserveField("fire", "OnTimerFired")
        m.timer.control = "start"
    end if
end function

sub OnTimerFired()
    currentView = m.top.componentController.currentView
    if currentView <> invalid and currentView.subtype() = "MediaView"
        if currentView.position = m.resumePosition and currentView.state <> "paused"
            ' position didn't changed after delay => playback finished while screensaver was being shown
            ' closing MediaView
            currentView.close = true
        end if
    end if
end sub
