'*********************************************************************
'** (c) 2018 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************
' Common_Utils.brs
' Utilites not specific to Roku MFG

function RokuMfgMerge(base as Object, params as Object, replace_existing=true as Boolean) as Object
    for each key in params
        if replace_existing or not base.doesExist(key) then
            base[key] = params[key]
        end if
    end for

    return base
end function

function RokuMfgDeepMerge(base as Object, params as Object,replace_existing=true as Boolean) as Object
    for each key in params
        if RokuMfgIsAA(params[key]) and RokuMfgIsAA(base[key]) then
            base[key] = RokuMfgDeepMerge(base[key], params[key])
        else if replace_existing or not base.doesExist(key) then
            base[key] = params[key]
        end if
    end for

    return base
end function

function RokuMfgDeepCopy(input as Dynamic) as Dynamic
    if (lcase(left(type(input), 2)) = "ro") then
        ' Converting to and from a JSON string will keep the structure and
        ' generate a new reference without needing to manually iterate
        return parseJSON(formatJSON(input))
    end if

    ' Fall through case for trivial types
    return input
end function

function RokuMfgIsFilenameGood(input as String) as Boolean
    regex = createObject("roRegex", "^[a-z0-9_./]*$", "i")
    return regex.isMatch(input)
end function

function RokuMfgMin(a as Dynamic, b as Dynamic) as Dynamic
    if a < b then
        return a
    end if

    return b
end function

function RokuMfgMax(a as Dynamic, b as Dynamic) as Dynamic
    if a > b then
        return a
    end if

    return b
end function

function RokuMfgXor(a as Integer, b as Integer) as Integer
    c = 0

    for i = 15 to 0 step -1
        bita = (a >> i) and 1
        bitb = (b >> i) and 1
        c = (c << 1) + ((bita + bitb) mod 2)
    end for

    return c
end function

' Custom modulo function that handles negative values
'
' :param a: the value to mod
' :type a: integer
' :param b: the maximum possible value
' :type b: integer
'
' :value return: a MOD b, pythonic
' :type return: integer
'
function RokuMfgModulo(a as Integer, b as Integer) as Integer
    if (b < 0)
        return mod(-a, -b)
    end if
    ret = a MOD b
    if (ret < 0)
        ret = ret + b
    end if
    return ret
end function

function RokuMfgCapitalize(input as String) as String
    return ucase(left(input, 1)) + right(input, len(input) - 1)
end function

' Type utilities
function RokuMfgHasInterface(obj as Dynamic, ifc as String) as Boolean
    if "<uninitialized>" <> type(obj) and invalid <> obj and invalid <> getInterface(obj, ifc) then
        return true
    end if

    return false
end function

function RokuMfgIsString(obj as Dynamic) as Boolean
    return RokuMfgHasInterface(obj, "ifString")
end function

function RokuMfgIsInt(obj as Dynamic) as Boolean
    return RokuMfgHasInterface(obj, "ifInt")
end function

function RokuMfgIsBool(obj as Dynamic) as Boolean
    return RokuMfgHasInterface(obj, "ifBoolean")
end function

function RokuMfgIsFloat(obj as Dynamic) as Boolean
    return RokuMfgHasInterface(obj, "ifFloat")
end function

function RokuMfgIsArray(obj as Dynamic) as Boolean
    return RokuMfgHasInterface(obj, "ifArray")
end function

function RokuMfgIsAA(obj as Dynamic) as Boolean
    return RokuMfgHasInterface(obj, "ifAssociativeArray")
end function

function RokuMfgIsInvalid(obj as Dynamic) as Boolean
    if "<uninitialized>" <> type(obj) then
        return invalid = obj
    end if

    return true
end function

' Casting
function RokuMfgStrCast(input as Dynamic) as String
    if not RokuMfgIsInvalid(findMemberFunction(input, "toStr")) then
        ' String, Boolean, Integer, and Float
        return input.toStr()
    else if RokuMfgIsArray(input) then
        str = ""
        for each el in input
            str = str + RokuMfgStrCast(el) + ", "
        end for
        return "[ " + left(str, len(str) - 2) + " ]"
    else if RokuMfgIsAA(input) then
        str = ""
        for each el in input
            str = str + RokuMfgStrCast(el) + ": " + RokuMfgStrCast(input[el]) + ", "
        end for
        return "{ " + left(str, len(str) - 2) + " }"
    end if

    return ""
end function

function RokuMfgBoolCast(input as Dynamic) as Boolean
    if RokuMfgIsBool(input) then
        return input
    else if RokuMfgIsInt(input) then
        return input <> 0
    else if RokuMfgIsString(input) then
        return lcase(input) = "true"
    end if

    return false
end function

function RokuMfgIntCast(input as Dynamic) as Integer
    if not RokuMfgIsInvalid(findMemberFunction(input, "toInt")) then
        return input.toInt()
    else if RokuMfgIsInt(input) then
        return input
    else if RokuMfgIsBool(input) then
        if input then
            return 1
        else
            return 0
        end if
    end if

    return 0
end function

'
' Used to get a padded string from an object.
'
' @param i the input object need to be padded.
' @param w the width of final padded string.
' @param v the character to pad, default is "0".
' @param p the padding direction. "true" is prepending, "false" is appending.
'
' @return the padded string.
function RokuMfgPad(i as Dynamic, w as Integer, v="0" as String, p=true as Boolean) as String
    s = RokuMfgStrCast(i)

    if (len(s) >= w) then
        return s
    else
        return (string(w - len(s), v) + s)
    end if
end function

'
' Used to get the index of specified element in an array.
'
' @param arr the array object.
' @param elem the element to query.
'
' @return  the position index in the array, -1 means not found.
function RokuMfgIndex(arr as Dynamic, elem as Dynamic) As Integer
    if RokuMfgIsArray(arr) then
        for i = 0 to arr.count() - 1
            if arr[i] = elem then
                return i
            end if
        end for
    end if
    return -1
end function

'
' Used to get HEX string from a integer.
'
' @param n the input number.
' @param w the width of final out string.
'
' @return  the HEX string.
function RokuMfgHexCast(n as Integer, w as Integer) as String
    ba = createObject("roByteArray")

    while (n)
        t = n and &hff
        ba.unshift(t)
        n = n >> 8
    end while

    return RokuMfgPad(ba.toHexString(), w)
end function

'
' Used to get a formatted string from a time.
'
' @param t the seconds as Integer.
' @param hw the width of hours as Integer, default 0 means do not display hours.
' @param mw the width of minutes as Integer.
' @param sw the width of seconds as Integer.
'
' @return  the formated time string, e.g. 02:05.
function RokuMfgStrFromTime(t as Integer, hw=0 as Integer, mw=2 as Integer, sw=2 as Integer) as String
    s = createObject("roString")
    dt = createObject("roDateTime")

    dt.FromSeconds(t)

    if hw > 0 then
        s += RokuMfgPad(Int(t/3600), hw) + ":"
        s += RokuMfgPad(dt.GetMinutes(), mw) + ":"
        s += RokuMfgPad(dt.GetSeconds(), sw)
    else
        s += RokuMfgPad(Int(t/60), mw) + ":"
        s += RokuMfgPad(dt.GetSeconds(), sw)
    end if

    return s
end function

'
' Used to check the return of legacy manufacturing API is valid or not.
'
' @param apiname the string of legacy API name.
' @param ret the returned object from API calling.

' @return  true or false
function RokuMfgCheckLegacyResponse(apiname as String, ret as Dynamic) as Boolean
    if invalid = ret then
        print apiname + " failed (ret = invalid)"
        return false
    end if

    if 0 = ret.valid then
        if invalid = ret.error then
            ret.error = "Unknown (type(error) = invalid)"
        end if

        print apiname + " failed with error: " + ret.error
        return false
    end if

    return true
end function