'*********************************************************************
'** (c) 2019 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************

'
' Construct and initialize a Video module object
' Provides the following functions:
' @see .getFileName RokuMfgVideoGetCurrentFileName,
' @see .getPlaybackStatus RokuMfgVideoGetPlaybackStatus,
' @see .playFile RokuMfgVideoPlayFile,
' @see .playPlaylist RokuMfgVideoPlaylist,
' @see .playNextFile RokuMfgVideoPlayNextFile,
' @see .exitPlayback RokuMfgVideoExitPlayback,
' @see .play RokuMfgVideoPlay
' @see .stop RokuMfgVideoStop,
' @see .pause RokuMfgVideoPause,
' @see .resume RokuMfgVideoResume,
' @see .seek RokuMfgVideoSeek,
' @see .getPlaybackDuration RokuMfgVideoGetPlaybackDuration,
'
' @param params roAssociativeArray of parameters to merge into the returned object.
'     allows overriding default functionality and adding features not provided
'     by the default implementation.
'     Default is invalid.
'
' @return video a new roAssociativeArray wrapping Video functionality
function RokuMfgVideoModule(params=invalid as Object) as Object
    print "Initializing Video Module"
    video = {
        player: CreateObject("roVideoPlayer"),

        paused: false,
        playing: false,
        filename: invalid,
        playlist: invalid,

        getFileName: RokuMfgVideoGetCurrentFileName,
        getPlaybackStatus: RokuMfgVideoGetPlaybackStatus,

        playFile: RokuMfgVideoPlayFile,
        playFolder: RokuMfgVideoPlayFolder,
        playPlaylist: RokuMfgVideoPlaylist,
        playNextFile: RokuMfgVideoPlayNextFile,
        exitPlayback: RokuMfgVideoExitPlayback,

        play: RokuMfgVideoPlay,
        stop: RokuMfgVideoStop,
        pause: RokuMfgVideoPause,
        resume: RokuMfgVideoResume,
        seek: RokuMfgVideoSeek,
        getPlaybackDuration: RokuMfgVideoGetPlaybackDuration
    }

    if RokuMfgIsAA(params) then
        RokuMfgMerge(video, params)
    end if

    video.player.SetDestinationRect({x: 0, y: 0, w: 0, h: 0})
    video.player.SetPositionNotificationPeriod(1)

    'Register MFG lib's message port to player object.
    video.player.setMessagePort(RokuMfg().port)

    return video
end function

'
' Used to get the filename for the current playing video file. Should only be
' called as getFileName from the RokuMfgVideoModule() object.
'
' @return the filename for the current playing video file
function RokuMfgVideoGetCurrentFileName() as String
    return m.filename
end function

'
' Used to get the current playing/pause status for the current playing video file.
' Should only be called as getPlaybackStatus from the RokuMfgVideoModule() object.
'
' @return a roAssociativeArray which contain the playing/pause status
function RokuMfgVideoGetPlaybackStatus() as Object
    o = {
        paused: m.paused,
        playing: m.playing
    }
    return o
end function

function p_mapMimetypes(file as Object) as Object
    mimetypes = { 'map known extensions to pseudo-mime-type here
        x_3g2:  { type: "video", format: "mp4" },
        x_3gp:  { type: "video", format: "mp4" },
        x_m4v:  { type: "video", format: "mp4" },
        x_mp4:  { type: "video", format: "mp4" },
        x_mov:  { type: "video", format: "mp4" },
        x_mkv:  { type: "video", format: "mkv" },
        x_wmv:  { type: "video", format: "wmv" },
        x_ts:   { type: "video", format: "ts"  },
        x_m3u8: { type: "video", format: "hls" },
        x_m4a:  { type: "audio", format: "mp4" },
        x_mp3:  { type: "audio", format: "mp3" },
        x_wma:  { type: "audio", format: "wma" },
        x_mka:  { type: "audio", format: "mka" },
        x_aif:  { type: "audio", format: "pcm" },
        x_aiff: { type: "audio", format: "pcm" },
        x_aifc: { type: "audio", format: "pcm" },
        x_au:   { type: "audio", format: "pcm" },
        x_wav:  { type: "audio", format: "pcm" },
        x_flac: { type: "audio", format: "flac" }
        'Anything else belongs in the "other" category
    }

    return mimetypes["x_" + file.path.Split().extension.mid(1)]
end function

'
' Used to playback a given video file with path and filename . Should only be
' called as playFile from the RokuMfgVideoModule() object.
'
' @param usb the usb module object which allow to access to filesystem
' @param params roAssociativeArray of parameters for the file to play.
'     Supported keys:
'     * volume the volume to for the video file.
'     * loop true-- allow to replay the video file after player finished the
'             video file playback; false-- do not allow to replay the video file
'     * file_path the file path (contain the file name) for the video file to play
'
' @return true if the video file is played successfully, otherwise return false.
function RokuMfgVideoPlayFile(usb as Object, params=invalid as Object) as Boolean
    o = {
        volume: "ext1:",
        loop: false,
        file_path: ""
    }

    if RokuMfgIsAA(params) then
        o = RokuMfgMerge(o, params)
    end if

    playlist = []

    if RokuMfgIsAA(usb) and RokuMfgHasInterface(usb.readFile, "ifFunction") then
        if o.file_path.Left(Len(o.volume)+1) = (o.volume + "/") then
            tmp = o.file_path.Mid(Len(o.volume)+1)
        else
            tmp = o.file_path
        end if

        file = usb.readFile(tmp, o.volume)
        if RokuMfgIsInvalid(file) then
            print "RokuMfgVideoPlayFile: file is not available"
            return false
        else
            file.mimetype = p_mapMimetypes(file)
        end if

        if not RokuMfgIsInvalid(file.mimetype) then
            playlist.Push(file)
        end if
    end if

    if playlist.count() > 0 then
        return m.playPlaylist({loop: o.loop, content: playlist})
    end if

    print "RokuMfgVideoPlayFile: cannot playback media file = " + o.file_path
    return false
end function

'
' Used to setup a playlist for the given folder, then start to playback the video
' files in the playlist. Should only be called as playFolder from the
' RokuMfgVideoModule() object.
'
' @param usb the usb module object which allow to access to filesystem
' @param params roAssociativeArray of parameters for the file to play.
'     Supported keys:
'     * volume the volume to for the path.
'     * loop true-- allow to replays the playlist automatically after player
'             finished all video files in playlist; false-- do not allow to
'             replay the playlist
'     * path the path within the volume to traverse
'
' @return true if the playlist has setup and the video in playlist is played
'         successfully, otherwise return false.
function RokuMfgVideoPlayFolder(usb as Object, params=invalid as Object) as Boolean
    o = {
        volume: "ext1:",
        loop: false,
        path: ""
    }

    if RokuMfgIsAA(params) then
        o = RokuMfgMerge(o, params)
    end if

    playlist = []

    if RokuMfgIsAA(usb) and RokuMfgHasInterface(usb.readDir, "ifFunction") then
        for each file in usb.readDir(o.path, o.volume)
            if file.type = "file" then
                file.mimetype = p_mapMimetypes(file)
            end if

            if not RokuMfgIsInvalid(file.mimetype) then
                playlist.Push(file)
            end if
        end for
    end if

    if playlist.count() > 0 then
        return m.playPlaylist({loop: o.loop, content: playlist})
    end if

    print "RokuMfgVideoPlayFolder: no media files in folder = " + o.path
    return false
end function

'
' Used to playback the video files in the given playlist. Should only be called
' as playPlaylist from the RokuMfgVideoModule() object.
'
' @param playlist the playlist array, the content of each element of the array
'        is a roAssociativeArray, should contain "file name", "file path",
'        "file size" and all informatin which is gotten from stat() API of
'        "roFilesystem" component.
'
' @return true if the video in playlist is played successfully, otherwise return
'         false.
function RokuMfgVideoPlaylist(playlist as Object) as Boolean
    defaults = {
        loop: false,
        current: -1,
        content: []
    }

    if RokuMfgIsAA(playlist) then
        o = RokuMfgMerge(defaults, playlist)
    else
        print "RokuMfgVideoPlaylist: Playlist is invalid!"
        return false
    end if

    if 0 = o.content.count() then
        print "RokuMfgVideoPlaylist: Playlist is empty!"
        return false
    end if

    m.playlist = o
    return m.playNextFile()
end function

'
' Used to playback the next video file in playlist. Should only be called as
' playNextFile from the RokuMfgVideoModule() object.
'
' @return true if the next file in playlist is played successfully, otherwise
'         return false.
function RokuMfgVideoPlayNextFile() as Boolean
    nextIndex = m.playlist.current + 1
    if m.playlist.loop then
        nextIndex = RokuMfgModulo(nextIndex, m.playlist.content.count())
    end if

    if nextIndex < m.playlist.content.count() then
        'Stop the video playback and clear the content of player
        m.player.stop()
        m.player.ClearContent()

        'Add the current file to play to player content list.
        m.playlist.current = nextIndex
        file = m.playlist.content[nextIndex]
        m.player.AddContent({
            title: file.name,
            stream: {
                url: "file://" + file.path
            },
            streamFormat: file.mimetype.format
        })

        m.filename = file.name
        return m.play()
    else
        m.playlist = invalid
        return false
    end if

    return true
end function

'
' Stop the video playback and clear the content of player. Should only be called
' as exitPlayback from the RokuMfgVideoModule() object.
'
' @return true if the video playback is stopped successfully, otherwise return
'         false.
function RokuMfgVideoExitPlayback() as Boolean
    ret = m.player.stop()
    m.player.ClearContent()
    m.playing = false
    m.paused = false
    return ret
end function

'
' Start to playback the current video file in content list of roVideoPlayer object.
' If Seek() was called prior to Play(), the player will start playing at the seek
' position. If Seek() was not called, the player advances its current position to
' the next item in the content list and starts playing that item. Should only be
' called as play from the RokuMfgVideoModule() object.
'
' @return true if the video file is played successfully, otherwise return false.
function RokuMfgVideoPlay() as Boolean
    ret = m.player.Play()

    if true = ret then
        m.playing = true
        m.paused = false
    end if

    return ret
end function

'
' Stops playback and resets the seek position; keeps the player’s current position
' unchanged. Should only be called as stop from the RokuMfgVideoModule() object.
'
' @return true if playback is stopped successfully, otherwise return false.
function RokuMfgVideoStop() as Boolean
    ret = m.player.Stop()

    if true = ret then
        m.playing = false
        m.paused = false
    end if

    return ret
end function

'
' Puts the roVideoPlayer object into pause mode. If the player is already in
' pause mode, this will generate an error. Should only be called as pause from
' the RokuMfgVideoModule() object.
'
' @return true if playback is paused successfully, otherwise return false.
function RokuMfgVideoPause() as Boolean
    ret = m.player.Pause()

    if true = ret then
        m.playing = false
        m.paused = true
    end if

    return ret
end function

'
' Puts the roVideoPlayer object into play mode starting from the pause point.
' This method must be called when the roVideoPlayer object is in pause mode;
' otherwise, it will generate an error. Should only be called as resume from
' the RokuMfgVideoModule() object.
'
' @return true if playback is resumed successfully, otherwise return false.
function RokuMfgVideoResume() as Boolean
    ret = m.player.Resume()

    if true = ret then
        m.playing = true
        m.paused = false
    end if

    return ret
end function

'
' Sets the start point of playback for the current video to offsetMs milliseconds.
' Should only be called as seek from the RokuMfgVideoModule() object.
'
' @param offset The number of milliseconds to offset the playback of the current
'               content item.
'
' @return true if playback is seeked successfully, otherwise return false.
function RokuMfgVideoSeek(offset as Integer) as Boolean
    ret = m.player.Seek(offset)

    if true = ret then
        m.playing = true
        m.paused = false
    end if

    return ret
end function

'
' Returns the duration of the video, in seconds. This information may not be
' available until after the video starts playing. A value of 0 is returned if
' the duration is unknown. Should only be called as getPlaybackDuration from
' the RokuMfgVideoModule() object.
'
' @return the duration of current playing video file.
function RokuMfgVideoGetPlaybackDuration() as Float
    if true = m.playing or true = m.paused then
        return m.player.getPlaybackDuration()
    end if

    print "RokuMfgVideoGetPlaybackDuration: Media file is not in playback status"
    return 0
end function