'*********************************************************************
'** (c) 2019 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************

'
' Construct a USB module object
' Provides the following functions:
' @see .volumes RokuMfgUSBVolumes
' @see .readFile RokuMfgUSBReadFile
' @see .readDir RokuMfgUSBReadDir
' @see .exists RokuMfgUSBFileExists
' @see .deleteFile RokuMfgUSBDeleteFile
' @see .check RokuMfgUSBCheckEvent
'
' @param params roAssociativeArray of parameters to merge into the returned object.
'     Supported keys:
'     * callback: ifFunction called when this module detects an roFilesystemEvent.
'         This function should accept an roFilesystemEvent message as a parameter.
'     Also allows overriding default functionality and adding features not
'     provided by the default implementation.
'     Default is invalid.
'
' @return usb a new roAssociativeArray wrapping USB functionality
function RokuMfgUSBModule(params=invalid as Object) as Object
    print "Initializing USB Module"
    usb = {
        ' Private data that should only be accessed by member functions.
        p_: {
            port: createObject("roMessagePort"),
            fs: createObject("roFilesystem"),
            path: createObject("roPath", "ext1:/")
        }

        volumes: RokuMfgUSBVolumes,
        readFile: RokuMfgUSBReadFile,
        readDir: RokuMfgUSBReadDir,
        exists: RokuMfgUSBFileExists,
        deleteFile: RokuMfgUSBDeleteFile,
        check: RokuMfgUSBCheckEvent,
        callback: invalid
    }

    usb.p_.fs.setMessagePort(usb.p_.port)

    if RokuMfgIsAA(params) then
        RokuMfgMerge(usb, params)
    end if
    return usb
end function

'
' Used to get a list of available volumes. Should only be called as volumes
' from the RokuMfgUSBModule() object.
'
' @param ext_only if true, will only return "ext#" volumes. Default is true.
'
' @return volumes an roList of volumes
function RokuMfgUSBVolumes(ext_only=true as Boolean) as Object
    volumes = m.p_.fs.getVolumeList()
    if ext_only then
        i = 0
        while i <> volumes.count()
            if "ext" <> left(volumes[i], 3) then
                volumes.delete(i)
            else
                i = i + 1
            end if
        end while
    end if

    return volumes
end function

function p_getDefaultUSBVolums(usb as Object) as String
    ' Get the default volume, USB doesn't always mount on ext1 (or other ext for that
    ' matter), if exists more, then default to ext device with smallest number.
    volume = "ext1:"
    vol_list = usb.volumes()
    if vol_list.count() > 0 then
        volume = vol_list[0]
        for each tmp_vol in vol_list
            if RokuMfgIntCast(tmp_vol.mid(3,1)) < RokuMfgIntCast(volume.mid(3,1)) then
                volume = tmp_vol
            end if
        end for
    end if

    return volume
end function
'
' Used to get list of files in a directory path. Should only be called as readFile
' from the RokuMfgUSBModule() object.
'
' @param file_path the path within the volume to traverse
' @param volume the volume to check. Default is system mounted ext device
'
' @return results list of roAssociativeArrays wrapping file details, including
'     name, path, size, and description. If the given file is not available or
'     there are issues happend, the function will return invalid.
function RokuMfgUSBReadFile(file_path as String, volume="" as String) as Object
    if volume = "" then
        volume = p_getDefaultUSBVolums(m)
    end if

    path = volume
    if file_path.left(1) <> "/" then
        path = path + "/"
    end if
    path = path + file_path
    file_object = createObject("roPath", path)

    if RokuMfgIsInvalid(file_object) then
        print "Error: cannot create roPath object for input file."
        return invalid
    end if

    temp = file_path.Split("/")
    filename = temp[temp.count()-1]

    file = {
        name: filename,
        path: file_object,
        size: invalid,
        description: invalid
    }
    ' Acquire other file information and merge it in.
    stat = m.p_.fs.stat(file.path)
    if stat.type = "file" then
        file = RokuMfgMerge(file, stat)
    else
        print "Error: input file is not available."
        return invalid
    end if

    return file
end function

'
' Used to get list of files in a directory path. Should only be called as readDir
' from the RokuMfgUSBModule() object.
'
' @param folder_path the path within the volume to traverse
' @param volume the volume to check. Default is system mounted ext device
'
' @return results list of roAssociativeArrays wrapping file details, including
'     name, path, size, and description.
function RokuMfgUSBReadDir(folder_path as String, volume="" as String) as Object
    if volume = "" then
        volume = p_getDefaultUSBVolums(m)
    end if

    path = volume
    if folder_path.left(1) <> "/" then
        path = path + "/"
    end if
    path = path + folder_path
    path = createObject("roPath", path)

    results = []
    if path.isValid()
        for each filename in m.p_.fs.getDirectoryListing(path)
            file = {
                name: filename,
                path: createObject("roPath", path + "/" + filename),
                size: invalid,
                description: invalid
            }

            ' Acquire other file information and merge it in.
            stat = m.p_.fs.stat(file.path)
            file = RokuMfgMerge(file, stat)

            results.push(file)
        end for
    else
        print "Error: path is not valid: " + path
        print m.p_.fs.getVolumeList()
    end if

    return results
end function

'
' Used to determine if a file path exists on the volume. Should only be called
' as exists from the RokuMfgUSBModule() object.
'
' @param path the path to validate
' @param volume the volume to check. Default is system mounted ext device
'
' @return exists true is path exists, false otherwise
function RokuMfgUSBFileExists(path as String, volume="" as String) as Boolean
    if volume = "" then
        volume = p_getDefaultUSBVolums(m)
    end if

    path = createObject("roPath", volume + "/" + path)
    return m.p_.fs.exists(path)
end function

'
' Used to Permanently removes the file or directory specified by the path parameter.
' If path is a directory, its contents are recursively removed. Should only be called
' as deleteFile from the RokuMfgUSBModule() object.
'
' @param path the path of the file to be deleted
' @param volume the volume to check. Default is system mounted ext device
'
' @return exists true is path exists, false otherwise
function RokuMfgUSBDeleteFile(path as String, volume="" as String) as Boolean
    if volume = "" then
        volume = p_getDefaultUSBVolums(m)
    end if

    path = createObject("roPath", volume + "/" + path)
    return m.p_.fs.delete(path)
end function

'
' Used to check for roFilesystemEvents. Should only be called as check from the
' RokuMfgUSBModule() object. If a valid callback function was provided when the
' USB object was initialized, it will be called when an event is detected.
' Otherwise, the event details will be printed to the console.
sub RokuMfgUSBCheckEvent()
    msg = wait(5, m.p_.fs.getMessagePort())
    if type(msg) = "roFilesystemEvent" then
        if RokuMfgHasInterface(m.callback, "ifFunction") then
            m.callback(msg)
        else
            msg = "Unhandled USB event"
            if msg.isStorageDeviceAdded() then
                msg = msg + ": Device inserted"
            else if msg.isStorageDeviceRemoved() then
                msg = msg + ": Device removed"
            endif

            print msg
        end if
    end if
end sub
