'*********************************************************************
'** (c) 2019 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************

'
' Construct a Timer module object
' Provides the following functions:
' @see .setup RokuMfgTimerSetup
' @see .add RokuMfgTimerAdd
' @see .delete RokuMfgTimerDelete
' @see .delAll RokuMfgTimerDeleteAll
' @see .modify RokuMfgTimerModify
' @see .update RokuMfgTimerUpdateTimers
'
' @param params roAssociativeArray of parameters to merge into the returned object.
'     Allows overriding default functionality and adding features not provided
'     by the default implementation.
'     Default is invalid.
'
' @return timer a new roAssociativeArray wrapping Timer functionality
function RokuMfgTimerModule(params=invalid as Object) as Object
    print "Initializing Timer Module"
    timer = {
        timers: {},

        sys_timer: createObject("roTimespan"),

        start: RokuMfgTimerStart,
        setup: RokuMfgTimerSetup,
        add: RokuMfgTimerAdd,
        delete: RokuMfgTimerDelete,
        delAll: RokuMfgTimerDeleteAll,
        modify: RokuMfgTimerModify,

        update: RokuMfgTimerUpdateTimers,
    }

    if RokuMfgIsAA(params) then
        RokuMfgMerge(timer, params)
    end if
    return timer
end function

'
' Used to start the system timer of timer module. Should only be called as start
' from the RokuMfgTimerModule() object.
sub RokuMfgTimerStart()
    m.sys_timer.mark()
end sub

'
' Used to setup a timer object . Should only be called as setup from the
' RokuMfgTimerModule() object.
'
' @param params the properties of timer object:
'     Supported keys:
'        id: the name of timer
'        expires: the timeout time of timer
'        loop_mode: the timer is set to loop mode or not
'        callback: the callback function after the timer is timeout
'        context: the context information when the callback function is called.
'
'
' @return o a new timer object which is merged with defauts settings.
function RokuMfgTimerSetup(params as Object) as Object
    defaults = {
        id: invalid,
        expires: 0,
        cur_time: 0,
        loop_mode: false,

        timer: m,
        context: invalid,
        callback: invalid,
    }

    o = RokuMfgMerge(defaults, params)

    ' Generate an anonymous id in case one isn't specified.
    if RokuMfgIsInvalid(o.id) then
        i = 0
        while true
            id = "anonymous" + RokuMfgStrCast(i)
            if RokuMfgIsInvalid(m.timers[id]) then
                o.id = id
                exit while
            end if
        end while
    end if

    return o
end function

'
' Used to add defined "timer" object into timer store "timers: {}" . Should only
' be called as add from the RokuMfgTimerModule() object.
'
' @param timer the defined "timer" object to add
'
' @return the defined "timer" object after it has been added into timer store
'     "timers: {}"
function RokuMfgTimerAdd(timer as Object) as Object
    obj_valid = RokuMfgIsAA(timer)
    id_valid = obj_valid and RokuMfgIsString(timer.id)
    expires_valid = obj_valid and RokuMfgIsInt(timer.expires) and (timer.expires > 0)
    has_callback = obj_valid and RokuMfgHasInterface(timer.callback, "ifFunction")
    has_context = obj_valid and (not RokuMfgIsInvalid(timer.context))

    if id_valid and expires_valid and has_callback and has_context then
        for each name in m.timers
            if timer.id = name then
                print "RokuMfgTimerAdd: duplicate timer object to be added!"
                return invalid
            end if
        end for
    else
        print "RokuMfgTimerAdd: input timer is not a valid timer object"
        return invalid
    end if

    m.timers[timer.id] = timer
    return m.timers[timer.id]
end function

'
' Used to delete one defined "timer" object from timer store "timers: {}" . Should
' only be called as delete from the RokuMfgTimerModule() object.
'
' @param id the "id" of "timer" object to be deleted
'
' @return true if the "timer" object is deleted successfully; otherwise return false.
function RokuMfgTimerDelete(id as String) as Boolean
    if RokuMfgIsAA(m.timers[id]) then
        return m.timers.delete(id)
    end if

    return false
end function

'
' Used to clear all "timer" objects from timer store "timers: {}". Should only be
' called as delAll from the RokuMfgTimerModule() object.
sub RokuMfgTimerDeleteAll()
    m.timers.clear()
end sub

'
' Used to modify the timeout time for a existing timer object. Should only be
' called as modify from the RokuMfgTimerModule() object.
'
' @param id the "id" of "timer" object to be modified.
' @param expires the new timeout time to be modify
'
' @return true if the timeout time is modified successfully; otherwise return false.
function RokuMfgTimerModify(id as String, expires as Integer) as Boolean
    if expires < 0 then
        print "RokuMfgTimerModify: expires parameter is not a valid value!"
        return false
    end if

    for each name in m.timers
        if id = name then
            m.timers[name].expires = expires
            return true
        end if
    end for

    return false
end function

'
' Used to update the current cost time for all timer objects in timer stack
' "timers: {}", new current cost time should be value of "old cost time" add the
' current system escape time . Should only be called as updateTimers from the
' RokuMfgTimerModule() object.
sub RokuMfgTimerUpdateTimers()
    escape_time = m.sys_timer.totalMilliseconds()

    for each key in m.timers
        timer = m.timers[key]
        timer.cur_time = timer.cur_time + escape_time

        if timer.cur_time >= timer.expires then
            'Call the callback function
            if RokuMfgHasInterface(timer.callback, "ifFunction") then
                timer.callback(timer.context)
            end if

            ' If the timer is loop mode timer, then update the cur_time for
            ' next running; Otherwise delete the timer after timeout.
            if timer.loop_mode = true then
                timer.cur_time = 0
            else
                m.delete(key)
                timer = invalid
            end if
        end if
    end for

    m.sys_timer.mark()
end sub