'*********************************************************************
'** (c) 2019 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************
' Screencap.brs
' Screen Capture Utilities

'
' Construct a wrapper object for MFG screen captures.
' Provides the following public member functions:
' @see .grab RokuMfgScreencapGrab()
' @see .getFilename RokuMfgScreencapGetFilename()
' @see .getFormat RokuMfgScreencapGetFormat()
' @see .getDimensions RokuMfgScreencapGetDimensions()
' @see .getScaling RokuMfgScreencapGetScaling()
' @see .getRect RokuMfgScreencapGetRectangle()
' @see .getPixel RokuMfgScreencapGetPixel()
'
' @return capture a new roAssociativeArray wrapping screencap functionality
function RokuMfgConstructScreencap() as Object
    print "Initializing Screencap Module"
    o = {
        ' Data intended to be private and only accessed by helper functions
        p_: {
            filename: "tmp:/cap.dat",
            raw: invalid,
            data: {
                format: "Unknown",
                height: 0,
                width: 0,
                pitch: 0,
                scaling: {
                    dst_h: 1.0,
                    dst_w: 1.0,
                    src_h: 1.0,
                    src_w: 1.0
                }
            }
        }

        grab: RokuMfgScreencapGrab,
        getFilename: RokuMfgScreencapGetFilename,
        getFormat: RokuMfgScreencapGetFormat,
        getDimensions: RokuMfgScreencapGetDimensions,
        getScaling: RokuMfgScreencapGetScaling,
        getRect: RokuMfgScreencapGetRectangle,
        getPixel: RokuMfgScreencapGetPixel
    }

    return o
end function

'
' Used to take a single screencap. Should only be called as "grab" from the
' RokuMfgConstructScreencap() object.
' Note: screen captures will fail if there's no valid video.
' Note: test patterns are not supported.
'
' @param height the height of the returned capture. Default is 720.
' @param width the width of the returned capture. Default is 0 since the
'              underlying capture call will automatically detemrine width based
'              on height and scaling
' @return true if the capture was successful, false otherwise
function RokuMfgScreencapGrab(height=720 as Integer, width=0 as Integer) as Boolean
    ret = RokuMfg().call("screencap", {
        fileout: m.getFilename(),
        height: height,
        width: width
    })

    if not RokuMfgCheckResponse(ret, false) then
        return false
    else if ret.data.height = 0 or ret.data.width = 0 then
        return false
    end if

    ' The raw data will be updated if necessary, such as when a caller
    ' accesses the pixel data.
    m.p_.raw = invalid
    m.p_.data = ret.data
    return true
end function

'
' Used to get the filename where the screen capture data will be stored. Should
' only be called as "getFilename" from the RokuMfgConstructScreencap() object.
'
' @return string of the filename (e.g., "tmp:/cap.dat")
function RokuMfgScreencapGetFilename() as String
    return m.p_.filename
end function

'
' Used to get the pixel format of the most recent screencap. Should only be
' called as "getFormat" from the RokuMfgConstructScreencap() object.
'
' @return string indicating the pixel format (e.g., "RGBA8888")
function RokuMfgScreencapGetFormat() as String
    return m.p_.data.format
end function

'
' Used to get the dimensions of the most recent screencap. Should only be
' called as "getDimensions" from the RokuMfgConstructScreencap() object.
'
' @return roAssociativeArray containing "height", "width", and "pitch"
function RokuMfgScreencapGetDimensions() as Object
    return {
        height: m.p_.data.height,
        width: m.p_.data.width,
        pitch: m.p_.data.pitch
    }
end function

'
' Used to get the scaling of the most recent screencap. Should only be called
' as "getScaling" from the RokuMfgConstructScreencap() object.
'
' @return roAssociativeArray containing "dst_w", "dst_h", "src_w", and "src_h"
function RokuMfgScreencapGetScaling() as Object
    return RokuMfgDeepCopy(m.p_.data.scaling)
end function

'
' Used to get a rectangle of the most recent screencap, starting at (x, y) with
' dimensions h x w. Should only be called as "getRect" from the
' RokuMfgConstructScreencap() object.
' Note: does not validate dimensions, so elements of returned object may be
'       invalid.
'
' @param x the x-coordinate of the starting point
' @param y the y-coordinate of the starting point
' @param h the height of the rectangle to get
' @param w the width of the rectangle to get
'
' @return roArray containing h roArrays, each which is size w. Allows accessing
'        individual pixels in the rectangle where (x, y) is (0, 0).
function RokuMfgScreencapGetRectangle(x as Integer, y as Integer, h as Integer, w as Integer) as Object
    rows = []
    for y_ = y to y + h - 1
        row = []
        for x_ = x to x + w - 1
            row.push(m.getPixel(x_, y_))
        end for

        rows.push(row)
    end for

    return rows
end function

'
' Used to get the value of a specific pixel from the most recent screencap.
' Should only be called as "getPixel" from the RokuMfgConstructScreencap()
' object.
' Note: (0,0) is the top-left of the capture.
' Note: the pixel coordniates are based on the capture dimensions, not
' screen resolution.
'
' @param x the x-coordinate of the pixel
' @param y the y-coordinate of the pixel
' @return roAssociativeArray containing "r", "g", "b", and "a". Will return
'         invalid if the coordinates are out of range.
function RokuMfgScreencapGetPixel(x as Integer, y as Integer) as Object
    ' Each row is m.pitch bytes
    row = m.p_.data.pitch * y

    ' Each pixel is 4 bytes (RGBA). Seek x * 4
    row = row + (x * 4)

    ' Check to see if we need to initialize the raw data
    if m.p_.raw = invalid then
        ' if the file doesn't exist, try to take a screenshot
        fs = createObject("roFilesystem")
        if not fs.exists(m.getFilename()) then
            m.grab()
        end if
        
        m.p_.raw = createObject("roByteArray")
        m.p_.raw.readFile(m.getFilename())
    end if

    raw = m.p_.raw
    if row + 3 > raw.count() -1 then
        return invalid
    end if

    b = raw[row]
    g = raw[row + 1]
    r = raw[row + 2]
    a = raw[row + 3]

    ' There's an issue with black pixels such that we get them back as a shade
    ' of purple. Detect that case and return black.
    if r = 124 and g = 0 and b = 118 and a = 255 then
        r = 0
        g = 0
        b = 0
    end if

    return {
        r: r,
        g: g,
        b: b,
        a: a
    }
end function
