'*********************************************************************
'** (c) 2019 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************

'
' Construct a GPIO module object
' Provides the following functions:
' @see .get RokuMfgGPIOGet
' @see .set RokuMfgGPIOSet
' @see .enable RokuMfgGPIOEnable
' @see .disable RokuMfgGPIODisable
'
' @param params roAssociativeArray of parameters to merge into the returned object.
'     Allows overriding default functionality and adding features not provided
'     by the default implementation. Default is invalid.
function RokuMfgGPIOModule(params=invalid as Object) as Object
    print "Initializing GPIO Module"
    gpio = {
        get: RokuMfgGPIOGet,
        set: RokuMfgGPIOSet,
        enable: RokuMfgGPIOEnable,
        disable: RokuMfgGPIODisable
    }

    if RokuMfgIsAA(params) then
        RokuMfgMerge(gpio, params)
    end if

    return gpio
end function

'
' Used to get a GPIO level. Can be called from global scope or as get from the
' RokuMfgGPIOModule() object.
'
' @param gpio string representing the GPIO name in custom_pkg.
'
' @return "ACTIVE", "UNKNOWN", or "UNDEFINED"
function RokuMfgGPIOGet(gpio as String) as String
    ret = RokuMfg().call("gpio", {
        action: "get",
        name: gpio
    })

    if RokuMfgCheckResponse(ret) then
        if ret.data then
            return "ACTIVE"
        else
            return "INACTIVE"
        end if
    end if

    return "UNDEFINED"
end function

'
' Used to set a GPIO level. Can be called from global scope or as set from the
' RokuMfgGPIOModule() object.
'
' @param gpio string representing the GPIO name in custom_pkg.
' @param enabled if true, set GPIO to ACTIVE; if false, set GPIO to INACTIVE.
'
' @return true if set was successful, false otherwise
function RokuMfgGPIOSet(gpio as String, enabled as Boolean) as Boolean
    return RokuMfgCheckResponse(RokuMfg().call("gpio", {
        action: "set",
        name: gpio,
        active: enabled
    }))
end function

'
' Used to set a GPIO level to ACTIVE. Should only be called as enable from the
' RokuMfgGPIOModule() object. Identical to RokuMfgGPIOSet(gpio, true).
'
' @param gpio string representing the GPIO name in custom_pkg.
'
' @return true if set was successful, false otherwise
function RokuMfgGPIOEnable(gpio as String) as Boolean
    return m.set(gpio, true)
end function

'
' Used to set a GPIO level to INACTIVE. Should only be called as enable from the
' RokuMfgGPIOModule() object. Identical to RokuMfgGPIOSet(gpio, false).
'
' @param gpio string representing the GPIO name in custom_pkg.
'
' @return true if set was successful, false otherwise
function RokuMfgGPIODisable(gpio as String) as Boolean
    return m.set(gpio, false)
end function
