'*********************************************************************
'** (c) 2019 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************

'
' Construct and initialize a "string" type "defined data" object
' Provides the following functions:
' @see .get RokuMfgDataGetStringValue
' @see .set RokuMfgDataUpdateStringValue,
' @see .update RokuMfgDataUpdateStringValue,
' @see .refresh RokuMfgDataRefreshStringValue,
'
' Used to construct and initialize a "string" type "defined data" object. Should
' only be called as constructString from the RokuMfgDataModule() object.
'
' @param params roAssociativeArray of parameters to merge into the returned object
'     Supported keys:
'     * id: the "id" of defined "data" object
'     * onInit: the callback function triggered when constructing a data object
'     * onRfresh: the callback function triggered when calling .get() to obtain an un-cached data object
'     * onUpdate: the callback function triggered when calling .set() to modify a data object
'     Also allows overriding default functionality and adding features not
'     provided by the default implementation.
'
' @return the defined "data" object
function RokuMfgDataConstructString(params as Object) as Object
    o = {
        class_: "String",
        value: "<UNSET>",
        cached: true,
        readOnly: false,

        get: RokuMfgDataGetStringValue,
        set: RokuMfgDataUpdateStringValue,

        update: RokuMfgDataUpdateStringValue,
        refresh: RokuMfgDataRefreshStringValue,

        onInit: invalid,
        onRefresh: invalid,
        onUpdate: invalid
    }

    if RokuMfgIsAA(params) then
        RokuMfgMerge(o, params)
    end if

    ' Add to data store if ID isn't invalid
    if not RokuMfgIsInvalid(o.id) then
        o.app = m.app
        o.data = m
        m.add(o.id, o)
    end if

    if RokuMfgHasInterface(o.onInit, "ifFunction") then
        o.onInit()
    end if

    return o
end function

'
' Used to get the current value for "string" type "defined data" object. Should
' only be called as get from the RokuMfgDataConstructString() object.
'
' @return the current value for "sting" type "defined data" object
function RokuMfgDataGetStringValue() as String
    if not m.cached then
        m.refresh()
    end if

    if false = RokuMfgIsString(m.value) then
        return RokuMfgStrCast(m.value)
    end if

    return m.value
end function

'
' Used to change the value for "string" type "defined data" object. Should
' only be called as set or update from the RokuMfgDataConstructString() object.
'
' @param value the value to change
'
' @return the read back value of "string" type "defined data" object after it
'         has been set.
function RokuMfgDataUpdateStringValue(str as String)
    if m.readOnly then
        print "ERROR: ReadOnly string being modified!"
        return invalid
    end if

    m.value = str
    if RokuMfgHasInterface(m.onUpdate, "ifFunction") then
        m.onUpdate()
    end if

    return m.get()
end function

'
' Used to update the value to platform for "sting" type "defined data" object.
' Should only be called as refresh from the RokuMfgDataConstructString() object.
sub RokuMfgDataRefreshStringValue()
    if RokuMfgHasInterface(m.onRefresh, "ifFunction") then
        m.onRefresh()
    end if
end sub