'*********************************************************************
'** (c) 2019 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************

'
' Construct and initialize a "selector" type "defined data" object
' Provides the following functions:
' @see .update RokuMfgDataUpdateSelector
' @see .setIndex RokuMfgDataSetSelectorIndex
' @see .set RokuMfgDataSetSelected,
' @see .get RokuMfgDataGetSelected,
' @see .getValue RokuMfgDataGetSelected,
'
' Used to construct and initialize a "selector" type "defined data" object. Should
' only be called as constructSelector from the RokuMfgDataModule() object.
'
' @param params roAssociativeArray of parameters to merge into the returned object
'     Supported keys:
'     * id: the "id" of defined "data" object
'     * onInit: the callback function triggered when constructing a data object
'     * onUpdate: the callback function triggered when calling .set() to modify a data object
'     Also allows overriding default functionality and adding features not
'     provided by the default implementation.
'
' @return the defined "data" object
function RokuMfgDataConstructSelector(params as Object) as Object
    o = {
        class_: "Selector",
        dataset: [],
        index: 0,
        onInit: invalid,                       ' User-supplied callback to init fields like dataset[] and index.
        onUpdate: invalid,                     ' User-supplied callback any time selector value is updated through the below methods.

        update: RokuMfgDataUpdateSelector,     ' Set selector index via offset from current spot (or zero).  Triggers onUpdate()
        setIndex: RokuMfgDataSetSelectorIndex, ' Set index directly.  Triggers onUpdate()'
        set: RokuMfgDataSetSelected,           ' Sets index by searching dataset for the supplied value. Triggers onUpdate()

        get: RokuMfgDataGetSelected,           ' Gets the "dataset" item at "index"
        getValue: RokuMfgDataGetSelected,      ' Gets a singular value (which usually is the same as what get() returns, but not always)
    }

    if RokuMfgIsAA(params) then
        RokuMfgMerge(o, params)
    end if

    ' Add to data store if ID isn't invalid
    if not RokuMfgIsInvalid(o.id) then
        o.app = m.app
        o.data = m
        m.add(o.id, o)
    end if

    if RokuMfgHasInterface(o.onInit, "ifFunction") then
        o.onInit()
    end if

    return o
end function

'
' Used to increase the index for "selector" type "defined data" object. Should
' only be called as update from the RokuMfgDataConstructSelector() object.
'
' @param offset the incremental index value to be added to current index value
'     for "selector" type "defined data" object
'
' @return the read back value of "selector" type "defined data" object after it
'         has been set.
function RokuMfgDataUpdateSelector(offset as Integer)
    newIndex = m.index + offset
    if newIndex < 0 then
        newIndex = newIndex + m.dataset.count()
    end if

    m.index = newIndex mod m.dataset.count()

    if RokuMfgHasInterface(m.onUpdate, "ifFunction") then
        m.onUpdate()
    end if

    return m.get()
end function

'
' Used to select one index for "selector" type "defined data" object. Should
' only be called as setIndex from the RokuMfgDataConstructSelector() object.
'
' @param index the index value to be selected
'
' @return the read back value of "selector" type "defined data" object after it
'         has been set.
function RokuMfgDataSetSelectorIndex(index as Integer)
    m.index = index mod m.dataset.count()

    if RokuMfgHasInterface(m.onUpdate, "ifFunction") then
        m.onUpdate()
    end if

    return m.get()
end function

'
' Used to change the value for "selector" type "defined data" object. Should
' only be called as set from the RokuMfgDataConstructSelector() object.
'
' @param value the value to change
'
' @return the read back value of "selector" type "defined data" object after it
'         has been set.
function RokuMfgDataSetSelected(value)
    for i = 0 to m.dataset.count() - 1
        if value = m.dataset[i] then
            m.index = i

            if RokuMfgHasInterface(m.onUpdate, "ifFunction") then
                m.onUpdate()
            end if

            return m.get()
        end if
    end for

    print "Error: Unable to find " + value + " in dataset "
    return invalid
end function

'
' Used to get the current value for "selector" type "defined data" object. Should
' only be called as get or getValue from the RokuMfgDataConstructSelector() object.
'
' @return the current value for "selector" type "defined data" object
function RokuMfgDataGetSelected()
    return m.dataset[m.index]
end function