'*********************************************************************
'** (c) 2018 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************
' Roku_MFG_Legacy_Tuner.brs
' Wrappers for the legacy "tuner" APIs

' @api ClearChannels
' Clear all channel database information.
' After this API is called, the tuner will require rescanning in order to be
' usable again.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_ClearChannels() as Object
    ret = RokuMfg().call("tuner", {action: "clear"})

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetChannelInfoByIndex
' Get channel information for a specific channel index.
' To get the current channel's information, use -1 for the index.
'
' @args
' Integer index - The channel index to query
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer index - The index of the selected channel.  Useful when -1 is supplied as input.
' Integer channel - The channel number.
' Integer subchannel - the subchannel number if applicable.
' Integer isCable - 0 : Channel is air; 1 : Channel is cable
' Integer isDigital - 0 : Channel is analog; 1 : Channel is digital
function RokuMfgLegacy_GetChannelInfoByIndex(index as Integer) as Object
    chInfo = {}
    if -1 = index then
        ret = RokuMfg().call("tuner", {action: "get", data: "current"})
        if not RokuMfgCheckResponse(ret) then
            return RokuMfgLegacyError(ret.header_.description)
        end if

        chInfo = ret.data
    else
        ret = RokuMfg().call("tuner", {action: "get"})
        if not RokuMfgCheckResponse(ret)
            return RokuMfgLegacyError(ret.header_.description)
        else if 0 > index or ret.data.count() <= index then
            return RokuMfgLegacyError("index out of range")
        end if

        chInfo = ret.data[index]
    end if

    data = {
        index: chInfo.index,
        channel: chInfo.channel,
        subchannel: chInfo.subchannel
    }

    if "digital" = chInfo.signal then
        data.isDigital = 1
    else
        data.isDigital = 0
    end if

    if "cable" = chInfo.medium then
        data.isCable = 1
    else
        data.isCable = 0
    end if

    return RokuMfgLegacySuccess(data)
end function

' @api GetIndexByChannelNumber
' Search the scanned channel database for a specific channel and return its index, if found.
'
' @args
' Integer phy_channel - The physical channel number to query.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer index - The index of the channel number, if found.
function RokuMfgLegacy_GetIndexByChannelNumber(phy_channel as Integer) as Object
    ret = RokuMfg().call("tuner", {action: "get"})
    if not RokuMfgCheckResponse(ret) then
        return RokuMfgLegacyError(ret.header_.description)
    end if

    for each channel in ret.data
        if phy_channel = channel.phy_channel then
            return RokuMfgLegacySuccess({index: channel.index})
        end if
    end for

    return RokuMfgLegacyError("physical channel not found")
end function

' @api TuneChannelByIndex
' Tunes to a channel, given a specific channel index.
'
' @args
' Integer index - The index of the channel to tune to.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_TuneChannelByIndex(index as Integer) as Object
    ret = RokuMfg().call("tuner", {action: "get"})
    if not RokuMfgCheckResponse(ret)
        return RokuMfgLegacyError(ret.header_.description)
    else if 0 > index or ret.data.count() <= index then
        return RokuMfgLegacyError("index out of range")
    end if

    ret = RokuMfg().call("tuner", {
        action: "set",
        data: ret.data[index]
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api ScanChannel
' Perform a scan for a specific physical channel.
' In the case where access to a specific channel is needed, but performing a full scan takes
' too muich time.  This doesn't work unless you know the physical channel (digital channel numbers
' do not provide the actual frequency to scan for).
'
' @args
' Integer phy_channel - The physical channel to scan.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_ScanChannel(phy_channel as Integer) as Object
    ret = RokuMfg().call("tuner", {
        action: "scan",
        data: phy_channel
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetAnalogTVChannel
' Tune to a specific analog channel number.
'
' @args
' Integer channel - The analog channel number.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetAnalogTVChannel(channel as Integer) as Object
    allChannels = RokuMfg().call("tuner", {action: "get"})
    if RokuMfgCheckResponse(allChannels) then
        for each chInfo in allChannels.data
            if "analog" = chInfo.signal and channel = chInfo.channel then
                ret = RokuMfg().call("tuner", {
                    action: "set",
                    data: chInfo
                })

                exit for
            end if 
        end for

        if RokuMfgCheckResponse(ret) then
            return RokuMfgLegacySuccess()
        end if
    end if

    return RokuMfgLegacyError("failed to set channel")
end function

' @api SetDigitalTVChannel
' Tune to a specific digital channel number.
'
' @args
' Integer channel - The primary channel number.
' Integer sub - The subchannel number.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetDigitalTVChannel(channel as Integer, sub_channel as Integer) as Object
    allChannels = RokuMfg().call("tuner", {action: "get"})
    if RokuMfgCheckResponse(allChannels) then
        for each chInfo in allChannels.data
            if "digital" = chInfo.signal and channel = chInfo.channel and sub_channel = chInfo.subchannel then
                ret = RokuMfg().call("tuner", {
                    action: "set",
                    data: chInfo
                })

                exit for
            end if 
        end for

        if RokuMfgCheckResponse(ret) then
            return RokuMfgLegacySuccess()
        end if
    end if

    return RokuMfgLegacyError("failed to set channel")
end function

' @api GetChannelIndex
' Retrieve the current channel's index from the channel database.
' This value is dynamically generated by Roku upon channel discovery, and should no longer be
' considered valid when another channel scan is performed.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer index - The current channel index.
function RokuMfgLegacy_GetChannelIndex() as Object
    ret = RokuMfg().call("tuner", {action: "get", data: "current"})
    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            index: ret.data.index
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api IncrementChannel
' Tune to the channel with a channel index one higher than the current.
' This API only works if the current input source is Tuner.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_IncrementChannel() as Object
    ret = RokuMfg().call("tuner", {action: "next"})
    if RokuMfgCheckResponse(ret)
        return RokuMfgLegacySuccess()
    end if

    RokuMfgLegacyError(ret.header_.description)
end function

' @api DecrementChannel
' Tune to the channel with a channel index one low than the current.
' This API only works if the current input source is Tuner.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_DecrementChannel() as Object
    ret = RokuMfg().call("tuner", {action: "prev"})
    if RokuMfgCheckResponse(ret)
        return RokuMfgLegacySuccess()
    end if

    RokuMfgLegacyError(ret.header_.description)
end function

' @api AddDtvChannel
' Scan for a specific channel.
' This API functions identically to #ScanChannel, but only scans either cable or air for the signal.
'
' @args
' String chtype - Channel type, one of: cable, air
' Integer channel - The physical channel number to add.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_AddDtvChannel(chtype as String, channel as Integer) as Object
    ret = RokuMfg().call("tuner", {
        action: "scan",
        data: channel,
        medium: chtype
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api LoadChannelListFile
' Load a channel database file.
' When an environment has already been scanned, loading a database with this API saves a considerable
' amount of time compared to performing a scan on each device during manufacturing.
'
' @args
' String name - The name of the channel database file to load.  The filename is in the format "atsc-channels-&lt;name&gt;.db"
'               Pass emtpy string ("") to load default custom_pkg atsc-channels.db.
'               Pass "usb" to load atsc-channels.db from attached USB drive.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_LoadChannelListFile(name as String) as Object
    pl = {action: "load"}
    if "usb" = lcase(name) then
        pl.location = "usb"
    else
        pl.location = "custom"
        pl.data = name
    end if

    ret = RokuMfg().call("tuner", pl)
    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SaveChannelListFile
' Save a channel database file.
' Copy the current channel database to the USB stick mounted on /media/ext1:/ (mfg mode only)
'
' @args
' String name - The name of the channel database file to save.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SaveChannelListFile(name as String) as Object
    ret = RokuMfg().call("tuner", {
        action: "save",
        data: name
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetClosedCaptioning
' Get the TV's closed captioning display mode.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer enabled - 0 : Captions are disabled; 1 : Captions are enabled
' Integer track - The preferred caption track.
function RokuMfgLegacy_GetClosedCaptioning() as Object
    ret = RokuMfg().call("captions", {action: "get"})

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            enabled: ret.data.enabled,
            track: ret.data.track
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetClosedCaptioning
' Enable/Disable closed captioning on the device.
' If -1 is supplied, the device disables closed captioning. Otherwise, it attempts to use
' the closed captioning track specified by the API.
'
' @args
' Integer captioning - Caption track to enable.
'
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetClosedCaptioning(captioning as Integer) as Object
    pl = {action: "set"}
    if -1 = captioning then
        pl.enable = false
    else
        pl.enable = true
        pl.track = captioning
    end if

    ret = RokuMfg().call("captions", pl)
    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function
