'*********************************************************************
'** (c) 2018-2019 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************
' Roku_MFG_Legacy_TV.brs
' Wrappers for the legacy "tv" APIs
Library "state/Roku_MFG_State.brs"

' @api SyncInputOnAppExit
' Automatically switch to the corresponding input source upon BrightScript application exit.
' Usually, exiting a BrightScript application returns the system to the Roku OS home screen.
' When this API is enabled, The Roku OS will immediately go to the input source currently
' active in the BrightScript application.
' This API was deprecated in February 2019, and is no longer supported.  It will always return valid = 0.
'
' @args
' Integer enable - 0 : disable input sync; 1 : enable input sync
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SyncInputOnAppExit(enable as Integer) as Object
    return RokuMfgLegacyError("deprecated")
end function

' @api SetPanelType
' Set the panel index for the TV.
' The numeric index is vendor-specific information and assigned for a specific panel. Configuration
' of the TV with the wrong panel index could result in a blank screen, or severely degraded picture
' quality.
'
' @args
' Integer panel - The panel index.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetPanelType(panel as Integer) as Object
    if RokuMfgIsSigmaPlatform() then
        ' Sigma needs to account for project base
        key = "PROJECT_BASE"
        pb = RokuMfg().call("custominfo", {
            action: "get",
            data: key
        })

        if RokuMfgCheckResponse(pb, false) and not RokuMfgIsInvalid(pb.data[key]) then
            panel = panel + val(pb.data[key], 10)
        end if
    end if

    ret = RokuMfg().call("pc", {
        action: "set",
        data: {panel: RokuMfgStrCast(panel)}
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetPanelType
' Get the panel index currently in use by the TV.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer value - The panel index currently in use.
function RokuMfgLegacy_GetPanelType() as Object
    ret = RokuMfg().call("pc", {
        action: "get",
        data: "panel"
    })

    if RokuMfgCheckResponse(ret) then
        if RokuMfgIsSigmaPlatform() then
            ' Sigma needs to account for project base
            key = "PROJECT_BASE"
            pb = RokuMfg().call("custominfo", {
                action: "get",
                data: key
            })

            if RokuMfgCheckResponse(pb, false) then
                ret.data.panel = RokuMfgStrCast(val(ret.data.panel, 10) - val(pb.data[key], 10))
            end if
        end if

        return RokuMfgLegacySuccess({
            value: val(ret.data.panel, 10)
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetSpeakerType
' Set the speaker index for the TV.
' The numeric index is vendor-specific information and assigned for a specific speaker. Configuration
' of the TV with the wrong speaker index could result in no audio, or severely degraded audio quality.
'
' @args
' Integer speaker - The speaker index.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetSpeakerType(speaker as Integer) as Object
    ret = RokuMfg().call("pc", {
        action: "set",
        data: {speaker: RokuMfgStrCast(speaker)}
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetSpeakerType
' Get the speaker index currently in use by the TV.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer value - The speaker index currently in use.
function RokuMfgLegacy_GetSpeakerType() as Object
    ret = RokuMfg().call("pc", {
        action: "get",
        data: "speaker"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            value: val(ret.data.speaker, 10)
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetTvManufacturerId
' Get an ID that uniquely represents the manufacturer of this TV.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String str - The ID corresponding to the manufacturer of this system's board/set.
function RokuMfgLegacy_GetTvManufacturerId() as Object
    key = "info/manufacturer"
    ret = RokuMfg().call("custominfo", {
        action: "get",
        data: key
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            str: ret.data[key]
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetTvModel
' Get the TV's manufacturer-designated model string.
' Note: Roku allocates its own internal model number, which is kept separate from the
' manufacturer's model string.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String str - The manufacturer's TV model identifier.
function RokuMfgLegacy_GetTvModel() as Object
    ret = RokuMfg().call("pc", {
        action: "get",
        data: "odmmodel"
    })

    if RokuMfgCheckResponse(ret) then
        str = ret.data.odmmodel
    else
        str = "Default"
    end if

    return RokuMfgLegacySuccess({str: str})
end function

' @api SetTvModel
' Set the TV's manufacturer-designated model string.
' Note: Roku allocates its own internal model number, which is kept separate from the
' manufacturer's model string.
'
' @args
' String model - The manufacturer's TV model identifier.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetTvModel(model as String) as Object
    ret = RokuMfg().call("pc", {
        action: "set",
        data: {odmmodel: model}
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetODMId
' Get an integer identifier corresponding to the TV ODM of this device.
' Correspond with Roku for the list of IDs pertinent to a manufacturer.  These IDs
' are provided on a need-to-know basis.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer value - The TV ODM's brand ID.
function RokuMfgLegacy_GetODMId() as Object
    if "TV" = getGlobalAA().mfg_constants.sysinfo.devicetype then
        key = "info/partner_id"
        ret = RokuMfg().call("custominfo", {
            action: "get",
            data: key
        })

        if RokuMfgCheckResponse(ret) then
            return RokuMfgLegacySuccess({
                ' Some platforms return a String, others an Integer
                ' Cast to a String, then rewrap as an Integer to avoid type mismatches.
                value: val(RokuMfgStrCast(ret.data[key]), 10)
            })
        end if

        return RokuMfgLegacyError(ret.header_.description)
    end if

    return RokuMfgLegacyError("Not supported (non-TV device)")
end function

' @api GetODMSerial
' Get the serial number set by the TV ODM.
' Note that Roku maintains its own serial numbers, which can be
' obtained by calling #GetSerialNumber and #GetLegacySerialNumber
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String str - The TV ODM's serial number for this device.
function RokuMfgLegacy_GetODMSerial() as Object
    key = "odmesn"
    ret = RokuMfg().call("pc", {
        action: "get",
        data: key
    })

    if RokuMfgCheckResponse(ret) then
        str = ret.data[key]
    else
        str = ""
    end if

    return RokuMfgLegacySuccess({str: str})
end function

' @api SetODMSerial
' Set the serial number as provided by the TV ODM.
' This API, on success, will set the manufacturing date in persistent storage,
' and will result in changes to the value returned by #GetLegacySerialNumber upon
' the next system boot.
'
' Note that Roku maintains its own serial numbers, and cannot be set.
'
' @args
' String serial - The TV ODM's serial number for this device.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetODMSerial(serial as String) as Object
    ret = RokuMfg().call("pc", {
        action: "set",
        data: {odmesn: serial}
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetTvBOMRev
' Get the TV's BOM (bill of materials) revision.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String str - The TV's BOM revision.
function RokuMfgLegacy_GetTvBOMRev() as Object
    ret = RokuMfg().call("syscfg", {
        action: "get",
        data: "RokuTv_BOM_Revision"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            str: ret.data.RokuTv_BOM_Revision
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetTvBOMRev
' Set the TV's BOM (bill of materials) revision.
'
' @args
' String revision - The TV's BOM revision.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetTvBOMRev(revision as String) as Object
    if "" = revision then
        return RokuMfgLegacyError("revision string empty")
    else if 32 < len(revision) then
        return RokuMfgLegacyError("Supplied revision string exceeds maximum length (32 chars)")
    end if

    RokuMfg().call("syscfg", {
        action: "set",
        data: {"RokuTv_BOM_Revision": revision}
    })

    return RokuMfgLegacySuccess()
end function

' @api GetFactoryRemote
' Get the TV's "factory remote control mode."
' This API allows the manufacturing app to change behavior of key presses or other
' operations based on a value.  It could theoretically be implemented directly in
' the BrightScript application, but this is also tied into certain firmware behaviors as well.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer value - The TV's factory "remote control mode." 0 : disabled; 1 : enabled
function RokuMfgLegacy_GetFactoryRemote() as Object
    ret = RokuMfg().call("syscfg", {
        action: "get",
        data: "RokuTv_Factory_Remote"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            value: val(ret.data.RokuTv_Factory_Remote, 10)
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetFactoryRemote
' Set the TV's "factory remote control mode."
' This API allows the manufacturing app to change behavior of key presses or other
' operations based on a value.  It could theoretically be implemented directly in
' the BrightScript application, but this is also tied into certain firmware behaviors as well.
'
' @args
' Integer enable - 0 : disable "factory remote control mode"; 1 : enable "factory remote control mode"
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetFactoryRemote(enable as Integer) as Object
    RokuMfg().call("syscfg", {
        action: "set",
        data: {"RokuTv_Factory_Remote": enable}
    })

    key = "allow_autoboot_override"
    ret = RokuMfg().call("custominfo", {
        action: "get",
        data: key
    })

    if RokuMfgCheckResponse(ret, false) then
        if ("true" = ret.data[key]) then
            info = getGlobalAA().mfg_constants.legacy
            if not RokuMfgBoolCast(enable) then
                state = info.PC_AUTOBOOT_OFF
            else
                state = info.PC_AUTOBOOT_ON
            end if

            RokuMfg().call("pc", {
                action: "set",
                data: {autobootoverride: state}
            })
        end if
    end if

    return RokuMfgLegacySuccess()
end function

' @api GetAgingMode
' Get the TV's aging mode.
' TV's undergo aging at the factory, which requires behavior that differs from Roku OS's
' default.  This configuration parameter is stored by the firmware, to allow for such
' differences to be implemented outside of BrightScript.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer value - 0 : aging mode is disabled; 1 : aging mode is enabled
function RokuMfgLegacy_GetAgingMode() as Object
' tfranklin: Do we want to remove it now?
    ' FIXME
    ' XXX TBD! We will deprecate the CONF_TVAgingMode configuration
    '          In favor of CONF_TvMfgBscMode. However, so as not to
    '          break anything that might be in-flux, we'll support
    '          CONF_TvAgingMode for a short duration.
    ret = RokuMfg().call("syscfg", {
        action: "get",
        data: "RokuTv_Aging_Mode"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            value: val(ret.data.RokuTv_Aging_Mode, 10)
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetAgingMode
' Set the TV's aging mode.
' TV's undergo aging at the factory, which requires behavior that differs from Roku OS's
' default.  This configuration parameter is stored by the firmware, to allow for such
' differences to be implemented outside of BrightScript.
'
' @args
' Integer enable - 0 : disable aging mode; 1 : enable aging mode
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetAgingMode(enable as Integer) as Object
    RokuMfg().call("syscfg", {
        action: "set",
        data: {
            "RokuTv_Aging_Mode": enable,
            "RokuTv_Mfg_Bsc_Mode": enable
        }
    })

    if true = enable and RokuMfgIsSigmaPlatform() then
        ' This applies to platforms using the legacy custom_pkg.
        ' Checking against Simga should suffice.
        key = "DISABLE_WIFI_IN_AGING_MODE"
        ret = RokuMfg().call("custominfo", {
            action: "get",
            data: key
        })

        if RokuMfgCheckResponse(ret) then
            if true = ret.data[key] then
                print "Disable wireless in aging mode"
                RokuMfg().call("network", {
                    component: "wifi",
                    action: "config",
                    enable: false,
                    block: false
                })
            end if
        end if
    end if

    return RokuMfgLegacySuccess()
end function

' @api SaveODMInformation
' Save specific ODM information to persistent storage.
'
' @args
' String name - Which type of ODM-specific data to save, one of: ODM_ESN, ODM_SERVICE, ODM_WB_DB
' String value - The associated data to be stored, for ODM_ESN and ODM_SERVICE.  ODM_WB_DB ignores this argument.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SaveODMInformation(name as String, value as String) as Object
    if "ODM_ESN" = name then
        return RokuMfgLegacy_SetODMSerial(value)
    else if "ODM_SERVICE" = name then
        ret = RokuMfg().call("pc", {
            action: "set",
            data: {odmservice: value}
        })

        if RokuMfgCheckResponse(ret) then
            return RokuMfgLegacySuccess()
        end if

        return RokuMfgLegacyError(ret.header_.description)
    else if "ODM_WB_DB" = name then
        ret = RokuMfg().call("pq", {
            action: "save",
            component: "panelcalib"
        })

        if RokuMfgCheckResponse(ret) then
            return RokuMfgLegacySuccess()
        end if

        return RokuMfgLegacyError(ret.header_.description)
    end if

    return RokuMfgLegacyError("Unknown ODM data name: " + name)
end function

' @api FetchODMInformation
' Get specific ODM information.
'
' @args
' String name - The type of ODM-specific data to fetch, one of: ODM_ESN, ODM_SERVICE
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String value - The corresponding data from persistent storage.
function RokuMfgLegacy_FetchODMInformation(name as String) as Object
    ' ODM_WB_DB is a more complicated lookup. Check for it separately
    if "ODM_WB_DB" = name then
' tfranklin: need PQ get functionality
        return RokuMfgLegacyError("incomplete implementation")
    end if

    lut = {
        "ODM_ESN":      "odmesn",
        "ODM_SERVICE":  "odmservice",
        "ODM_MODEL":    "odmmodel",
        "PANEL_INDEX":  "panel"
    }

    if not RokuMfgIsInvalid(lut[name]) then
        ret = RokuMfg().call("pc", {
            action: "get",
            data: lut[name]
        })

        if RokuMfgCheckResponse(ret) then
            return RokuMfgLegacySuccess({
                value: ret.data[lut[name]]
            })
        end if

        return RokuMfgLegacyError(ret.header_.description)
    end if

    return RokuMfgLegacyError("Unknown ODM data name: " + name)
end function

' @api GetDisplaySize
' Get the TV panel's display current resolution.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer w - Resolution width.
' Integer h - Resolution height.
' Integer native_width - Native resolution width.
' Integer native_height - Native resolution height.
function RokuMfgLegacy_GetDisplaySize() as Object
    di = CreateObject("roDeviceInfo")
    size = di.getDisplaySize()
    data = {
        h: size.h,
        w: size.w
    }

    key_w = "panel/Width"
    key_h = "panel/Height"
    ret = RokuMfg().call("custominfo", {
        action: "get",
        data: [key_w, key_h]
    })

    if RokuMfgCheckResponse(ret, false) then
        data.native_width = ret.data[key_w]
        data.native_height = ret.data[key_h]
    end if

    return RokuMfgLegacySuccess(data)
end function

' @api GetPanelFrequency
' Get the current panel backlight frequency.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer value - The frequency that the panel's backlight PWM is driven at.
function RokuMfgLegacy_GetPanelFrequency() as Object
    key = "Disp_Backlight_Frequency"
    ret = RokuMfg().call("pq", {
        action: "get",
        component: "get",
        data: key
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            value: val(ret.data[key], 10)
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetPanelFrequency
' Set the panel backlight frequency.
'
' @args
' Integer frequency - The frequency that the panel's backlight PWM should be driven at.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetPanelFrequency(frequency as Integer) as Object
    ret = RokuMfg().call("pq", {
        action: "set",
        component: "setting",
        data: {
            "Disp_Backlight_Frequency": frequency
        }
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetPanelFlip
' Get the current state of the TV panel's vertical inversion.
' This API was deprecated in April 2016, and is no longer supported.  It will always return valid = 0.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_GetPanelFlip() as Object
    return RokuMfgLegacyError("deprecated")
end function

' @api SetPanelFlip
' Set the current state of the TV panel's vertical inversion.
' This API was deprecated in April 2016, and is no longer supported.  It will always return valid = 0.
'
' @args
' Integer flip - 0 : do not flip; 1 : flip
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetPanelFlip(flip as Integer) as Object
    return RokuMfgLegacyError("deprecated")
end function

' @api GetPanelMirror
' Get whether the TV panel data is inverted horizontally.
' This API was deprecated in April 2016, and is no longer supported.  It will always return valid = 0.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_GetPanelMirror() as Object
    return RokuMfgLegacyError("deprecated")
end function

' @api SetPanelMirror
' Set whether the TV panel data is inverted horizontally.
' This API was deprecated in April 2016, and is no longer supported.  It will always return valid = 0.
'
' @args
' Integer mirror - 0 : do not flip; 1 : flip
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetPanelMirror(mirror as Integer) as Object
    return RokuMfgLegacyError("deprecated")
end function

' @api SetTestPatternsVideoPlane
' Set the test pattern to be rendered in the video plane.
' This API is only intended for MStar TV platforms currently.
'
' @args
' Integer en - 0 : render to graphics plane; 1 : render to video plane
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetTestPatternsVideoPlane(en as Integer) as Object
    if not RokuMfgBoolCast(en) then
        plane = "graphics"
    else
        plane = "video"
    end if

    RokuMfgState().set("TestPatternPlane", plane)
    return RokuMfgLegacySuccess()
end function

' @api SetTestPatternColor
' Generate and potentially display a test pattern of the chosen color on the device.
' This API can also be used to disable any active test pattern by setting the second parameter to 0.
'
' @args
' String col - One of: red, green, blue, white black, ire20, ire80, ire100
' Integer en - 0 : disable pattern; 1 : enable pattern
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetTestPatternColor(col as String, en as Integer) as Object
    plane = RokuMfgState().get("TestPatternPlane")
    if not RokuMfgBoolCast(en) then
        ret = RokuMfg().call("testpattern", {
            action: "disable",
            plane: plane
        })
    else
        if RokuMfgIsChipType("Sigma UXL-38") then
            ' UXL uses YCbCr (ordered as CrYCb in the registers).
            ' Use Cr = r, Cb = b, and Y = g for this case instead
            ' of maintaining separate implementations
            lut = {
                red:    {red: &hff, green: &h00, blue: &h80},
                green:  {red: &h00, green: &h00, blue: &h00},
                blue:   {red: &h80, green: &h00, blue: &hff},
                white:  {red: &h80, green: &hff, blue: &h80},
                black:  {red: &h80, green: &h00, blue: &h80},
                ire20:  {red: &h80, green: &h33, blue: &h80},
                ire80:  {red: &h80, green: &hcc, blue: &h80},
                ire100: {red: &h80, green: &hff, blue: &h80}
            }
        else if RokuMfgIsChipType("Sigma SX7") then
            ' According to James Z on 11/14/2015, the new SX register value takes RBG 10:10:10.
            ' Note: That is _RBG_ not rgb as you would expect.
            lut = {
                red:    {red: &h3ff, blue: &h000, green: &h000},
                green:  {red: &h000, blue: &h000, green: &h3ff},
                blue:   {red: &h000, blue: &h3ff, green: &h000},
                white:  {red: &h3ff, blue: &h3ff, green: &h3ff},
                black:  {red: &h000, blue: &h000, green: &h000},
                ire20:  {red: &h0CC, blue: &h0CC, green: &h0CC},
                ire80:  {red: &h332, blue: &h332, green: &h332},
                ire100: {red: &h300, blue: &h300, green: &h300}
            }
        else
            ' Assume MStar case covers general case, too
            ' Note: Due to definition of the SetTestPatternRGB,
            ' only bottom 8 bits needed though MStar supports 12 bits
            lut = {
                red:    {red: &hff, green: &h00, blue: &h00},
                green:  {red: &h00, green: &hff, blue: &h00},
                blue:   {red: &h00, green: &h00, blue: &hff},
                white:  {red: &hff, green: &hff, blue: &hff},
                black:  {red: &h00, green: &h00, blue: &h00},
                ire20:  {red: &h33, green: &h33, blue: &h33},
                ire80:  {red: &hCC, green: &hCC, blue: &hCC},
                ire100: {red: &hff, green: &hff, blue: &hff}
            }
        end if

        if RokuMfgIsInvalid(lut[col]) then
            return RokuMfgLegacyError("unsupported color: " + col)
        end if

        ret = RokuMfg().call("testpattern", {
            action: "rgb",
            plane: plane,
            data: lut[col]
        })
    end if

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetPeripheral
' Set up peripheral (device). For example, enable/disable RGB Bypass for MEMC.
'
' @args
' roAssociativeArray input {
'     String device   - name of the device (peripheral)
'     String function - what needs to be done on the device
'     String data     - data associated with the function
' }
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetPeripheral(input as Object) as Object
    if not RokuMfgIsString(input.device) then
        return RokuMfgLegacyError("expected input.device as string")
    else if not RokuMfgIsString(input.function) then
        return RokuMfgLegacyError("expected input.function as string")
    else if not RokuMfgIsString(input.data) then
        return RokuMfgLegacyError("expected input.data as string")
    end if

    ' For now, expect all data to be stringified booleans
    data = {}
    data[input.function] = RokuMfgBoolCast(input.data)
    ret = RokuMfg().call("hwmodule", {
        action: "set",
        component: input.device,
        data: data
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetUBKeyValue
' Set a Key/Value pair in the Shared Persistent Configuration area
'
' @args
' String key - the name of the key to set
' String value - the value to set it to, use the empty string to remove the setting
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - the error code if an error occurred
function RokuMfgLegacy_SetUBKeyValue(key as String, value as String) as Object
    data = {}
    data[key] = value
    ret = RokuMfg().call("ubapp", {
        action: "set",
        data: data
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetUBKeyValue
' Get a Key's Value in the Shared Persistent Configuration area
'
' @args
' String key - the name of the key to get
'
' @return
' Integer valid - 0 : not valid (not set); 1 : valid
' String value - the value of the key
' String error - the error code if an error occurred
function RokuMfgLegacy_GetUBKeyValue(key as String) as Object
    ret = RokuMfg().call("ubapp", {
        action: "get",
        data: key
    })

    if RokuMfgCheckResponse(ret) then
        if "" = ret.data[key] then
            return RokuMfgLegacyError(key + " not set")
        end if

        return RokuMfgLegacySuccess({value: ret.data[key]})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetTestPatternRGBColor
' Generate and potentially display a test pattern of a specific RGB color.
' This API can also be used to disable any active test pattern by setting the "enable" parameter to 0.
'
' @args
' roAssociativeArray input {
'     Integer red - Red component of color (0:255)
'     Integer green - Green component of color (0:255)
'     Integer blue - Blue component of color (0:255)
'     Integer enable - 0 : disable pattern; 1 : enable pattern
' }
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetTestPatternRGBColor(input as Object) as Object
    if not RokuMfgIsAA(input) then
        return RokuMfgLegacyError("expected input as associative array")
    else if RokuMfgIsInvalid(input.red) or RokuMfgIsInvalid(input.green) or RokuMfgIsInvalid(input.blue) or RokuMfgIsInvalid(input.enable) then
        return RokuMfgLegacyError("expected input to contain red, green, blue, and enable memebers")
    end if

    plane = RokuMfgState().get("TestPatternPlane")
    if not RokuMfgBoolCast(input.enable) then
        ret = RokuMfg().call("testpattern", {
            action: "disable",
            plane: plane
        })
    else
        if RokuMfgIsChipType("Sigma SX7") then
            mask = &h3ff
        else
            mask = &hff
        end if

        ret = RokuMfg().call("testpattern", {
            action: "rgb",
            plane: plane,
            data: {
                red: input.red and mask,
                green: input.green and mask,
                blue: input.blue and mask
            }
        })
    end if

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetTestPatternIRE
' Generate and potentially display an IRE-percentage (white) test pattern.
' This API can also be used to disable any active test pattern by setting the first parameter to 0.
'
' @args
' Integer en - 0 : disable pattern; 1 : enable pattern
' Integer ire_percent - IRE percent (0:100)
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetTestPatternIRE(en as Integer, ire_percent as Integer) as Object
    plane = RokuMfgState().get("TestPatternPlane")
    if not RokuMfgBoolCast(en) then
        ret = RokuMfg().call("testpattern", {
            action: "disable",
            plane: plane
        })
    else
        ' Limit 0-100. Allow the platform implementation to handle
        ' number of bits and how to set them.
        ire = RokuMfgMax(0, RokuMfgMin(100, ire_percent))
        ret = RokuMfg().call("testpattern", {
            action: "ire",
            plane: plane,
            data: ire
        })
    end if

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetHdmiEdid
' Retrieve the HDMI Extended Display Identification Data.
' This can be done on a per-port basis, and can access extension blocks as well.
'
' @args
' Integer port - The HDMI port.
' Integer block - EDID block number.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' roByteArray edid_data - A byte array containing the requested EDID block.
function RokuMfgLegacy_GetHdmiEdid(port as Integer, block as Integer) as Object
    ' GetHdmiEdidByMode supported undocumented mode=-1
    ' to handle both GetHdmiEdid implementations in one function
    return RokuMfgLegacy_GetHdmiEdidByMode(-1, port, block)
end function

' @api GetHdmiEdidByMode
' Get the HDMI's Extended Display Identification Data for a specific HDMI mode.
' HDMI 1.4 has differing EDID information when compared to HDMI 2.2, and this API can query
' either mode.  This can be done on a per-port basis, and can access extension blocks as well.
'
' @args
' Integer mode - 0 : HDMI 1.4; 1 : HDMI 2.2
' Integer port - The HDMI port.
' Integer block - The EDID block number.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' roByteArray edid_data - A byte array containing the requested EDID block.
function RokuMfgLegacy_GetHdmiEdidByMode(mode as Integer, port as Integer, block as Integer) as Object
    ' GetHdmiEdidByMode supported undocumented mode=-1
    ' to handle both GetHdmiEdid implementations in one function
    version_map = {
        "0": "HDMI 1.4",
        "1": "HDMI 2.0"
    }

    if -1 <> mode and RokuMfgIsInvalid(version_map[RokuMfgStrCast(mode)]) then
        return RokuMfgLegacyError("invalid hdmi mode")
    end if

    ret = RokuMfg().call("hdmi", {
        action: "get",
        component: "count"
    })

    if RokuMfgCheckResponse(ret) then
        max_port = ret.data.rx
        if (1 > port or max_port < port) then
            return RokuMfgLegacyError("invalid port (expected: 1-" + RokuMfgStrCast(max_port))
        end if

        pl = {
            action: "get",
            component: "rxrawedid",
            port: port,
            block: block
        }

        if not RokuMfgIsInvalid(version_map[RokuMfgStrCast(mode)]) then
            pl.mode = version_map[RokuMfgStrCast(mode)]
        end if

        ret = RokuMfg().call("hdmi", pl)

        if RokuMfgCheckResponse(ret) then
            return RokuMfgLegacySuccess({
                edid_data: ret.data
            })
        end if
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetHdmiHdcp14Ksv
' Get the TV's HDMI HDCP 1.4 KSV section.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' roByteArray hdcp_data - A byte array containing HDCP 1.4 KSV data.
function RokuMfgLegacy_GetHdmiHdcp14Ksv() as Object
    ret = RokuMfg().call("hdmi", {
        action: "get",
        component: "hdcp14ksv"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            hdcp_data: ret.data
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetHdmiHdcp22ReceiverId
' Get the TV's HDMI HDCP 2.2 receiver id section.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' roByteArray hdcp_data - A byte array containing HDCP 2.2 receiver ID data.
function RokuMfgLegacy_GetHdmiHdcp22ReceiverId() as Object
    ret = RokuMfg().call("hdmi", {
        action: "get",
        component: "hdcp22rxid"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            hdcp_data: ret.data
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetHdmiArcEnable
' Get whether HDMI ARC (audio return channel) is enabled.
' Audio return channel is a CEC feature that allows the TV to route audio to a specific
' HDMI port, where an audio amplifier or receiver is connected.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer enabled - 0 : not enabled; 1 : enabled
function RokuMfgLegacy_GetHdmiArcEnable() as Object
    ret = RokuMfg().call("cec", {
        action: "get",
        data: {component: "arc"}
    })

    if RokuMfgCheckResponse(ret) then
        if true = ret.data then
            enabled = 1
        else
            enabled = 0
        end if

        return RokuMfgLegacySuccess({enabled: enabled})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetHdmiArc
' Enable/disable HDMI ARC (audio return channel).
' Audio return channel is a CEC feature that allows the TV to route audio to a specific
' HDMI port, where an audio amplifier or receiver is connected.
'
' @args
' Integer enable - 0 : disable ARC; 1 : enable ARC
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetHdmiArc(enable as Integer) as Object
    ret = RokuMfg().call("cec", {
        action: "set",
        data: {
            component: "arc",
            state: RokuMfgBoolCast(enable)
        }
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetSystemAudioControl
' Get whether System Audio Control is enabled.
' System Audio Control is a CEC feature that allows an audio amplifier or receiver
' to be used with a TV.  The volume can be controlled by using the remote of any compatible
' devices in the CEC environment.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer enabled - 0 : not enabled; 1 : enabled
function RokuMfgLegacy_GetSystemAudioControl() as Object
    ret = RokuMfg().call("cec", {
        action: "get",
        data: {component: "sac"}
    })

    if RokuMfgCheckResponse(ret) then
        if true = ret.data then
            enabled = 1
        else
            enabled = 0
        end if

        return RokuMfgLegacySuccess({enabled: enabled})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetSystemAudioControl
' Enable/disable System Audio Control on the TV.
' System Audio Control is a CEC feature that allows an audio amplifier or receiver
' to be used with a TV.  The volume can be controlled by using the remote of any compatible
' devices in the CEC environment.
'
' @args
' Integer enable - 0 : disable System Audio Control; 1 enable System Audio Control
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetSystemAudioControl(enable as Integer) as Object
    ret = RokuMfg().call("cec", {
        action: "set",
        data: {
            component: "sac",
            state: RokuMfgBoolCast(enable)
        }
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api deprecated GetSupportedLanguageCodes
' Get all language codes supported by the device.
' This API was deprecated in August 2018, and is no longer supported.  It will always return valid = 0.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' roArray codes - A list of all supported language codes.
function RokuMfgLegacy_GetSupportedLanguageCodes() as Object
    return RokuMfgLegacyError("deprecated")
end function

' @api deprecated GetLanguage
' Get the current system language setting.
' For a list of possible language settings, use #GetSupportedLanguageCodes.
' This API was deprecated in August 2018, and is no longer supported.  It will always return valid = 0.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String language - The current language code.
' String country - The current country code.
function RokuMfgLegacy_GetLanguage() as Object
    return RokuMfgLegacyError("deprecated")
end function

' @api deprecated SetLanguage
' Set the system language.
' The system country code is automatically updated when language is selected.  To get
' a list of supported languages, use #GetSupportedLanguageCodes.
' This API was deprecated in August 2018, and is no longer supported.  It will always return valid = 0.
'
' @args
' String language - The language code to apply.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetLanguage(language as String) as Object
    return RokuMfgLegacyError("deprecated")
end function

' @api GetAllSources
' Get all supported input sources on the device.
' The exact list varies from platform to platform, but will likely contain some or all of the following:
'
' ui, tuner, cvbs, hdmi1, hdmi2, hdmi3, hdmi4, dtv, atv, composite
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' roArray sources - A list containing all supported input sources on the device.
function RokuMfgLegacy_GetAllSources() as Object
    ret = RokuMfg().call("inputsource", {
        action: "get",
        data: "all"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            sources: ret.data
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetSource
' Get the current input source.
' For a list of possible inputs, use #GetAllSources
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String value - The current input source.
function RokuMfgLegacy_GetSource() as Object
    ret = RokuMfg().call("inputsource", {action: "get"})
    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            value: ret.data
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetSource
' Set the current input source.
' For a list of possible inputs, use #GetAllSources
'
' @args
' String srcstr - The input source to switch to.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetSource(srcstr as String) as Object
    ret = RokuMfg().call("inputsource", {
        action: "set",
        data: srcstr
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetLastSource
' Get the previous active input source.
' For a list of possible inputs, use #GetAllSources
'
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String value - The previous input source used.
function RokuMfgLegacy_GetLastSource() as Object
    ret = RokuMfg().call("inputsource", {
        action: "get",
        data: "last"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            value: ret.data
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetLvdsSscEnable
' Get whether LVDS SSC is enabled or not.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer enabled - 0 : disabled; 1 : enabled
function RokuMfgLegacy_GetLvdsSscEnable() as Object
    ret = RokuMfg().call("lvds", {
        action: "get",
        component: "ssc"
    })

    if RokuMfgCheckResponse(ret) then
        if false = ret.data.enabled then
            enabled = 0
        else
            enabled = 1
        end if

        return RokuMfgLegacySuccess({enabled: enabled})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetLvdsDriveStrength
' Get the panel LVDS drive strength.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer value - The drive strength of the panel's LVDS. 0 : 270mV; 1 : 330mV; 2 : 390mV; 3 : 450mV
function RokuMfgLegacy_GetLvdsDriveStrength() as Object
    key = "Disp_SSC_Strength"
    ret = RokuMfg().call("pq", {
        action: "get",
        component: "setting",
        data: key
    })

    if RokuMfgCheckResponse(ret) then
        value = ret.data[key]
        if RokuMfgIsSigmaPlatform() then
            if 0 > value or 3 < value then
                return RokuMfgLegacyError("failed to retrieve setting")
            end if
        end if

        return RokuMfgLegacySuccess({value: value})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetLvdsDriveStrength
' Set the panel LVDS's drive strength.
'
' @args
' Integer drive - The drive strength of the panel's LVDS. 0 : 270mV; 1 : 330mV; 2 : 390mV; 3 : 450mV
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetLvdsDriveStrength(drive as Integer) as Object
    if RokuMfgIsSigmaPlatform() then
        if 0 > drive or 3 < drive then
            return RokuMfgLegacyError("supplied value out of range")
        end if
    end if

    ret = RokuMfg().call("pq", {
        action: "set",
        component: "setting",
        data: {"Disp_SSC_Strength": drive}
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetLvdsSscParameters
' Get the panel LVDS SSC parameters, if the device supports it.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer strength - The LVDS drive strength. 0 : 270mV; 1 : 330mV; 2 : 390mV; 3 : 450mV
' Integer frequency - The LVDS frequency. 0 : fm ~ 3.9-7.8Hz; 1 : fm ~ 5.2-10.4kHz; 2 : fm ~ 7.8-15.6kHz; 3 : fm ~ 15.6-31.3kHz; 4 : fm ~ 32.3-64.5kHz; 5 : fm ~ 62.5-125kHz; 6 : fm ~ 83.3-166.6kHz; 7 : fm ~ 125-250kHz
' Integer depth - The LVDS depth. 0 : no SSC; 1 : kss ~ 1; 2 : kss ~ 1.5; 3 : kss ~ 2; 4 : kss ~ 3; 5 : kss ~ 4; 6 : kss ~ 6; 7 : kss ~ 8
function RokuMfgLegacy_GetLvdsSscParameters() as Object
    ret = RokuMfg().call("lvds", {
        action: "get",
        component: "ssc"
    })

    if RokuMfgCheckResponse(ret) then
        keys = ["depth", "frequency", "strength"]
        pl = {}
        for each key in keys
            if RokuMfgIsInvalid(ret.data[key])
                return RokuMfgLegacyError("failed to get key: " + key)
            end if

            pl[key] = ret.data[key]
        end for

        return RokuMfgLegacySuccess(pl)
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetLvdsSscParameters
' Set panel LVDS SSC parameters, if the device supports it.
'
' @args
' roAssociativeArray data {
'     Integer strength - The LVDS drive strength. 0 : 270mV; 1 : 330mV; 2 : 390mV; 3 : 450mV
'     Integer frequency - The LVDS frequency. 0 : fm ~ 3.9-7.8Hz; 1 : fm ~ 5.2-10.4kHz; 2 : fm ~ 7.8-15.6kHz; 3 : fm ~ 15.6-31.3kHz; 4 : fm ~ 32.3-64.5kHz; 5 : fm ~ 62.5-125kHz; 6 : fm ~ 83.3-166.6kHz; 7 : fm ~ 125-250kHz
'     Integer depth - The LVDS depth. 0 : no SSC; 1 : kss ~ 1; 2 : kss ~ 1.5; 3 : kss ~ 2; 4 : kss ~ 3; 5 : kss ~ 4; 6 : kss ~ 6; 7 : kss ~ 8
' }
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetLvdsSscParameters(data as Object) as Object
    if not RokuMfgIsAA(data) or not RokuMfgIsInt(data.strength) or not RokuMfgIsInt(data.frequency) or not RokuMfgIsInt(data.depth) then
        return RokuMfgLegacyError("expected data as associative containing strength, frequency, and depth integers")
    end if

    pl = {
        "Disp_SSC_Depth": data.depth,
        "Disp_SSC_Frequency": data.frequency,
        "Disp_SSC_Strength": data.strength
    }

    ret = RokuMfg().call("pq", {
        action: "set",
        component: "setting",
        data: pl
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetVCOMLevel
' Get the current VCOM level.
' VCOM is a voltage drop measured across the LCD pixel.  It's a parameter tuned on a per-panel basis
' to remove flickering from an LCD screen.
'
' The only platform that currently supports this API is longview.
'
' This API was deprecated in August 2018, and is no longer supported.  It will always return valid = 0.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer value - The current VCOM level.
function RokuMfgLegacy_GetVCOMLevel() as Object
    return RokuMfgLegacyError("deprecated")
end function

' @api SetVCOMLevel
' Set the VCOM level.
' VCOM is a voltage drop measured across the LCD pixel.  It's a parameter tuned on a per-panel basis
' to remove flickering from an LCD screen.
'
' The only platform that currently supports this API is longview.
'
' This API was deprecated in August 2018, and is no longer supported.  It will always return valid = 0.
'
' @args
' Integer level - The new VCOM level.
' Integer save - 0 : set (but do not save) save VCOM level; 1 : set and save new VCOM level to TCON non-volatile memory.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetVCOMLevel(level as Integer, save as Integer) as Object
    return RokuMfgLegacyError("deprecated")
end function

' @api GetODMGammaTable
' Get a gamma table from persistent storage.
' This API specifically returns part or all of a single color lookup table for
' a particular gamma table.
'
' @args
' roAssociativeArray input {
'     String table - Name of table to look up, one of: Cool-2.2, Normal-2.2, Warm-2.2, Vivid
'     String color - Color of table to look up, one of: red, green, blue
'     Integer start - Starting offset of target table (0:255).
'     Integer length - The number of elements to retrieve (1:256).
' }
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' roByteArray data - A byte array containing the color data per pixel (16 bits big-endian, H L H L).
function RokuMfgLegacy_GetODMGammaTable(input as Object) as Object
    msg = invalid
    if not RokuMfgIsString(input.table) or not RokuMfgIsString(input.color) then
        msg = "expected input.table and input.color as strings"
    else if not RokuMfgIsInt(input.start) or not RokuMfgIsInt(input.length) then
        msg = "expected input.start and input.length as integers"
    end if

    if not RokuMfgIsInvalid(msg) then
        return RokuMfgLegacyError(msg)
    end if

    ret = RokuMfg().call("gamma", {
        component: "persistent",
        action: "get",
        table: input.table
        color: input.color
        offset: input.start,
        length: input.length
    })

    if RokuMfgCheckResponse(ret) then
        data = CreateObject("roByteArray")
        for each el in ret.data
            data.push((el and &hff00) >> 8)
            data.push(el and &h00ff)
        end for

        return RokuMfgLegacySuccess({
            data: data
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api UpdateODMGammaTable
' Update the gamma table in persistent storage.
' Note that the "clean" option below will ignore any other parameters and delete all data.
'
' @args
' roAssociativeArray input {
'     String clean - If "yes", clears all PC gamma table data instead of writing data.
'     String table - Name of table to look up, one of: Cool-2.2, Normal-2.2, Warm-2.2, Vivid
'     String color - Color of table to look up, one of: red, green, blue
'     Integer start - Starting offset of target table (0:255).
'     Boolean combine - (Optional) If true, combine auto gamma and preset gamma. False by default.
'     roByteArray data - A byte array containing the color data per pixel (16 bits big-endian, H L H L).
' }
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_UpdateODMGammaTable(input as Object) as Object
    action = "update"
    data = []
    if RokuMfgIsString(input.clean) and "yes" = lcase(input.clean) then
        action = "clear"
    else if not RokuMfgIsString(input.table) or not RokuMfgIsString(input.color) then
        return RokuMfgLegacyError("expected input.table and input.color as strings")
    else if not RokuMfgIsInt(input.start) then
        return RokuMfgLegacyError("expected input.start as integer")
    else if not RokuMfgHasInterface(input.data, "ifByteArray") then
        return RokuMfgLegacyError("expected input.data as roByteArray")
    else
        for i=0 to input.data.count() - 1 step 2
            data.push((input.data[i] << 8) or input.data[i + 1])
        end for
    end if

    ret = RokuMfg().call("gamma", {
        component: "persistent",
        action: action,
        table: input.table,
        color: input.color,
        offset: input.start,
        combine: RokuMfgBoolCast(input.combine),
        data: data
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api ApplyODMGammaTable
' Apply the ODM gamma tables currently stored in persistent storage.
' This API is useful after updating gamma table data, as the device needs to be restarted otherwise
' for the new gamma values to take effect.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_ApplyODMGammaTable() as Object
    ret = RokuMfg().call("gamma", {
        component: "persistent",
        action: "apply"
    })

    if RokuMfgCheckResponse(ret) then
        ' If we were successful applying the table, refresh PQ
        RokuMfgCheckResponse(RokuMfg().call("pq", {action: "apply"}))
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api WriteHDCP14RxEncryptedKey
' Write HDCP 1.4 Rx key in persistent storage.
' This API is intended for use on MStar platforms that don't have HDCP1.4 keys
' burned into the SOC OTP.
' It uses an encrypted key generated by the MStar SecureStorm tool.
'
' @args
' roByteArray input - The encrypted HDCP 1.4 Rx key (289 bytes).
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_WriteHDCP14RxEncryptedKey(input as Object) as Object
    ret = RokuMfg().call("hdmi", {
        action: "set",
        component: "hdcp14key",
        key: input
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api ReadHDCP14RxKey
' Indicates whether there is a valid HDCP 1.4 Rx key stored on the device.
' This API does not reveal the HDCP 1.4 Rx key contents, despite the name.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_ReadHDCP14RxKey() as Object
    ret = RokuMfg().call("hdmi", {
        action: "get",
        component: "hdcp14key"
    })

    if RokuMfgCheckResponse(ret) then
        ba = createObject("roByteArray")
        ba.fromBase64String(ret.data.data)
        return RokuMfgLegacySuccess({hdcp_key: ba})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api WriteHDCP22RxEncryptedKey
' Write HDCP 2.2 Rx key in persistent storage.
' This API is intended for use on longview platforms, and uses an encrypted key
' generated by the MStar SecureStorm tool.
'
' @args
' roByteArray input - The encrypted HDCP 2.2 Rx key (1044 bytes).
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_WriteHDCP22RxEncryptedKey(input as Object) as Object
    ret = RokuMfg().call("hdmi", {
        action: "set",
        component: "hdcp22key",
        key: input
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api ReadHDCP22RxKey
' Indicates whether there is a valid HDCP 2.2 Rx key stored on the device.
' This API does not reveal the HDCP 2.2 Rx key contents, despite the name.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_ReadHDCP22RxKey() as Object
    ret = RokuMfg().call("hdmi", {
        action: "get",
        component: "hdcp22key"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({hdcp_key: ret.data})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api CleanPCPanelCalib
' Delete all white balance calibration data from persistent storage.
' A reset is requried after using this API for changes to take effect.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_CleanPCPanelCalib() as Object
    info = getGlobalAA().mfg_constants.sysinfo
    if true = info.ismanufacturing then
        ret = RokuMfg().call("pc", {
            action: "set",
            data: {panelcalib: ""}
        })

        if RokuMfgCheckResponse(ret) then
            return RokuMfgLegacySuccess()
        end if

        return RokuMfgLegacyError(ret.header_.description)
    end if

    return RokuMfgLegacyError("not in manufacturing mode")
end function

' @api UnzipData
' Unzip data compressed with zlib and return uncompressed contents.
'
' @args
' roByteArray input - zlib-compressed data.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' roByteArray data - Decompressed data.
function RokuMfgLegacy_UnzipData(input as Object) as Object
    filein = "tmp:/input_gz"
    fileout = "tmp:/output_unzipped"

    input.writeFile(filein)
    ret = RokuMfg().call("filesys", {
        action: "unzip",
        filein: filein,
        fileout: fileout
    })

    if RokuMfgCheckResponse(ret) then
        unzipped = CreateObject("roByteArray")
        unzipped.readFile(fileout)
        return RokuMfgLegacySuccess({
            data: unzipped
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetSupportedModels
' Return a list of all supported models for this platform/manufacturer.
'
' This API is supported on TV projects beginning with Longview and Midland.  Liberty and Fort Worth do not
' support this API.
'
' This API analyzes the "projects" file within custom package and extracts model information
' for each entry. The idea behind this API is to deduplicate data that is already stored in custom package,
' and remove the need to manually maintain manufacturing application structures.
'
' The structure of the projects file allows for manufacturer modification.  It is not rigid.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' roArray data - an array of roAssociativeArray objects containing model and project information.
function RokuMfgLegacy_GetSupportedModels() as Object
    ret = RokuMfg().call("projects")

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            data: ret.data
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetRokuModel
' Get the current Roku model.
'
' This API is supported on TV projects beginning with Longview and Midland.  Liberty and Fort Worth do not
' support this API.
'
' This model string is allocated by Roku and is not strictly tied to ODM model.
'
' The Roku model string is used by the Roku OS to configure many aspects of the system,
' including feature and component support.  Changing this to the wrong model may impact
' performance of the device.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String str - The Roku model string.
function RokuMfgLegacy_GetRokuModel() as Object
    ret = RokuMfg().call("pc", {
        action: "get",
        data: "rokumodel"
    })

    if RokuMfgCheckResponse(ret) then
        str = ret.data.rokumodel
    else
        str = "Default"
    end if

    return RokuMfgLegacySuccess({str: str})
end function

' @api SetRokuModel
' Set the Roku model.
'
' This API is supported on TV projects beginning with Longview and Midland.  Liberty and Fort Worth do not
' support this API.
'
' The Roku model string is used by the Roku OS to configure many aspects of the system,
' including feature and component support.  Changing this to the wrong model may impact
' performance of the device.
'
' @args
' String model - The Roku model string.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetRokuModel(model as String) as Object
    ret = RokuMfg().call("pc", {
        action: "set",
        data: {rokumodel: model}
    })

    if RokuMfgCheckResponse(ret) then
        RokuMfg().call("pq", {action: "clear"})
        RokuMfg().call("aq", {action: "clear"})
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetCPUXTAL
' Set the CPU XTAL level.
'
' @args
' Integer level - CPU XTAL level (0:3)
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetCPUXTAL(level as Integer) as Object
    ret = RokuMfg().call("cpuxtal", {
        action: "set",
        data: level
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetCPUXTAL
' Get the CPU XTAL level.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer level - CPU XTAL level (0:3)
function RokuMfgLegacy_GetCPUXTAL() as Object
    ret = RokuMfg().call("cpuxtal", {action: "get"})

    if RokuMfgCheckResponse(ret)
        return RokuMfgLegacySuccess({
            level: val(ret.data, 10)
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api ExtendCustomConfig
' Provide model configuration extensions in persistent storage, so otherwise-identical
' models from manufacturers (models with identical Roku model strings) can support different
' feature sets.
'
' @args
' roAssociativeArray input {
'     String MODEL_HAS_WIFI_REMOTE - Model has WiFi remote support, one of: true, false
'     String MODEL_SHIPS_WITH_ENHANCED_REMOTE - Model ships with WiFi remote, one of: true, false
'     String MODEL_SHIPS_WITH_VOICE_REMOTE - Model ships with voice remote, one of: true, false
'     String MODEL_HAS_REMOTE_FINDER - Model has remote finder support, one of: true, false
'     String DEFAULT_LANG - Model's default language (en/it/de/es)
'     String DEFAULT_LOCALE - Model's default locale (en_US/en_CA/es_MX)
' }
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_ExtendCustomConfig(input as Object) as Object
    if not RokuMfgIsAA(input) then
        return RokuMfgLegacyError("expected input data as associative array")
    end if

    data = {}
    for each flag in getGlobalAA().mfg_constants.legacy.flag_custom_pkg_ext
        if not RokuMfgIsInvalid(input[flag]) then
            data["flag_"+flag] = input[flag]
        end if
    end for

    for each field in getGlobalAA().mfg_constants.legacy.field_custom_pkg_ext
        if not RokuMfgIsInvalid(input[field]) then
            data[field] = input[field]
        end if
    end for

    ret = RokuMfg().call("pc", {
        action: "set",
        data: data
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetExtendedCustomConfig
' Look up any model configuration extensions present in persistent storage.  This exists to
' support models with identical Roku model strings, but different ODM model strings, allowing
' different feature sets between them.
'
' Some of the return values below may not exist, if no such parameter is saved in persistent storage.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String MODEL_HAS_WIFI_REMOTE - Model has WiFi remote support, one of: true, false
' String MODEL_SHIPS_WITH_ENHANCED_REMOTE - Model ships with WiFi remote, one of: true, false
' String MODEL_SHIPS_WITH_VOICE_REMOTE - Model ships with voice remote, one of: true, false
' String MODEL_HAS_REMOTE_FINDER - Model has remote finder support, one of: true, false
function RokuMfgLegacy_GetExtendedCustomConfig() as Object
    data = []

    for each flag in getGlobalAA().mfg_constants.legacy.flag_custom_pkg_ext
        data.push("flag_" + flag)
    end for

    data.append(getGlobalAA().mfg_constants.legacy.field_custom_pkg_ext)

    ret = RokuMfg().call("pc", {
        action: "get",
        data: data
    })

    if RokuMfgCheckResponse(ret) then
        ' Need to get this again because we don't want to prepend "flag_" in the response
        data = getGlobalAA().mfg_constants.legacy.flag_custom_pkg_ext
        data.append(getGlobalAA().mfg_constants.legacy.field_custom_pkg_ext)

        response = {}
        for each item in data
            if not RokuMfgIsInvalid(ret.data[item]) and "" <> ret.data[item] then
                response[item] = ret.data[item]
            end if
        end for

        return RokuMfgLegacySuccess(response)
    end if

    return RokuMfgLegacyError("failed to get extended config")
end function

' @api StartChecksumProcess
' Start the background process of computing the checksums. This may take a while, so run it in the background
'
' @args
' None
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' Integer started - 0 : not started; 1 : started
function RokuMfgLegacy_StartChecksumProcess() as Object
    ret = RokuMfg().call("checksum", {action: "start"})
    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({started: 1})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetAllChecksumValues
' Retrieve the checksum values for various parts of the filesystem. The sums should
' be identical for a given version of software regardless of any crypto signing keys.
'
' @args
' None
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' Integer done - 0 : computations not complete; 1 : computations completed
' Integer image - checksum of both partital acramfs images (1k - 48MB)
' Integer custom - checksum current custom package image
' Integer boot - checksum of boot partition without NAND specific info
' Integer main - checksum of uncompressed Main.bin (from /tmp/pq)
' Integer panel - checksum of uncompressed Main_Color.bin for the currently selected panel (from /tmp/pq)
' Integer audio - checksum of all audio database files (/tmp/A_*.aqdb)
' Integer video - checksum of current video database file (/tmp/V_Default.pqdb)
' Integer extra - checksum of DLC.ini (from /tmp/pq)
' String msg - message from a thrown exception
' Integer progress - how far into the checksum did we get before failing?
function RokuMfgLegacy_GetAllChecksumValues() as Object
    ret = RokuMfg().call("checksum", {action: "get"})
    if RokuMfgCheckResponse(ret) then
        if true = ret.data.valid then
            data = {
                image: ret.data.image,
                custom: ret.data.custom,
                boot: ret.data.boot,
                main: ret.data.pqmain,
                panel: ret.data.pqpanel,
                audio: ret.data.audio,
                video: ret.data.video,
                extra: ret.data.extra,
                msg: ret.data.msg,
                progress: ret.data.progress
            }

            if true = ret.data.done then
                data.done = 1
            else
                data.done = 0
            end if

            return RokuMfgLegacySuccess(data)
        end if

        return {
            valid: 0,
            msg: ret.data.msg
        }
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetGPIO
' Set a GPIO output to active high or low state
'
' @args
' String name - The name of the GPIO to control
' Integer state - The state to set the GPIO to 1 = active, 0 = inactive
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - error message if not valid
function RokuMfgLegacy_SetGPIO(name as String, state as Integer) as Object
    ret = RokuMfg().call("gpio", {
        action: "set",
        name: name,
        active: RokuMfgBoolCast(state)
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api TristateGPIO
' Set a GPIO output to tristate (e.g., make it an input), currently a one-way trip
'
' @args
' String name - The name of the GPIO to control
' Integer state - The state to set the GPIO to 1 = tristated, 0 = normal [not supported at the moment]
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - error message if not valid
function RokuMfgLegacy_TristateGPIO(name as String, state as Integer) as Object
    ret = RokuMfg().call("gpio", {
        action: "tristate",
        name: name,
        active: RokuMfgBoolCast(state)
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetRasterPattern
' Displays a raster pattern on screen, overriding any UI while displayed
'
' @args
' Integer enable - 1: enable the pattern, 0: disable the pattern
' Integer r - 0..1023 raster level for R color
' Integer g - 0..1023 raster level for G color
' Integer b - 0..1023 raster level for B color
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - error message if not valid
function RokuMfgLegacy_SetRasterPattern(enable as Integer, r as Integer, g as Integer, b as Integer) as Object
    if not RokuMfgBoolCast(enable) then
        ret = RokuMfg().call("testpattern", {
            action: "disable",
            plane: "raster"
        })
    else
        ret = RokuMfg().call("testpattern", {
            action: "rgb",
            plane: "raster",
            data: {
                red: r,
                green: g,
                blue: b
            }
        })
    end if

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetIRModeRaw
' Set the IR receive mode to Raw (shot) mode. In this mode, we get an event for each transition
' between IR seen and IR not seen and the event contains the duration in us for each state. We
' assemble this data to support the Funai serial over IR protocol.
'
' @args
' Integer enable - If 0, IR is set to normal mode, else it's set to raw mode
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - error message if not valid
function RokuMfgLegacy_SetIRModeRaw(enable as Integer) as Object
    if false = getGlobalAA().mfg_constants.sysinfo.ismanufacturing then
        return RokuMfgLegacyError("requires mfg mode")
    end if

    ret = RokuMfg().call("ir_cap", {
        action: "set",
        component: "raw",
        enable: RokuMfgBoolCast(enable)
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetInputSignalState
' Retrieve the current signal state of the curent input (e.g., valid video received or not)
'
' @args
' None
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' Integer width - width in pixels if valid
' Integer height - height in pixels if valid
' Boolean interlaced - true if signal is interlaced and valid
' Float refresh - field rate if valid
function RokuMfgLegacy_GetInputSignalState() as Object
    ret = RokuMfg().call("inputsource", {
        action: "get",
        data: "signalstate"
    })

    if RokuMfgCheckResponse(ret) then
        if "validsignal" = ret.data.state then
            ' If refresh is 60, we convert the response to
            ' Integer instead of float.
            refresh = ret.data.mode.refresh
            if RokuMfgIsInt(refresh) then
                refresh = Cdbl(refresh)
            end if

            return RokuMfgLegacySuccess({
                width: ret.data.mode.width,
                height: ret.data.mode.height,
                interlaced: ret.data.mode.interlaced,
                refresh: refresh
            })
        end if

        return RokuMfgLegacyError("signal state not valid")
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetPCKeyValue
' Set a Key/Value pair in the Persistent Configuration area
'
' @args
' String key - the name of the key to set
' String value - the value to set it to, use the empty string to remove the setting
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - the error code if an error occurred
function RokuMfgLegacy_SetPCKeyValue(key as String, value as String) as Object
    data = {}
    data[key] = value
    ret = RokuMfg().call("pc", {
        action: "set",
        data: data
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetPCKeyValue
' Get a Key's Value in the Persistent Configuration area
'
' @args
' String key - the name of the key to get
'
' @return
' Integer valid - 0 : not valid (not set); 1 : valid
' String value - the value of the key
' String error - the error code if an error occurred
function RokuMfgLegacy_GetPCKeyValue(key as String) as Object
    ret = RokuMfg().call("pc", {
        action: "get",
        data: key
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            value: ret.data[key]
        })
    end if

    return RokuMfgLegacyError("failed to get key: " + key)
end function

' @api GetCustomKeyValue
' Set a Key/Value pair in the custom package data
'
' @args
' String key - the name of the key to get
'
' @return
' Integer valid - 0 : not valid (not set); 1 : valid
' String value - the value of the key
' String type - the lowercase BRS type of the returned value
' String error - the error code if an error occurred
function RokuMfgLegacy_GetCustomKeyValue(key as String) as Object
    ret = RokuMfg().call("custominfo", {
        action: "get",
        data: key
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            value: ret.data[key],
            type: lcase(type(ret.data[key]))
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetRTCWakeup
' Set a wakeup alarm in the RTC for when we go to suspend
'
' @args
' Integer delay : how long to wait before waking back up, in seconds
'
' @return
' Integer valid - 0 : not valid, 1 : valid
function RokuMfgLegacy_SetRTCWakeup(delay as Integer) as Object
    ret = RokuMfg().call("rtc", {
        action: "set",
        component: "wakeup",
        data: delay
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetBacklightFrequency
' Set the PWM frequency of the backlight
'
' @args
' Integer frequency : desired frequency in Hz of the PWM
'
' @return
' Integer valid - 0 : not valid, 1 : valid
function RokuMfgLegacy_SetBacklightFrequency(frequency as Integer) as Object
    ret = RokuMfg().call("backlight", {
        action: "set",
        component: "frequency",
        data: frequency
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetBacklightFrequency
' Get the PWM frequency of the backlight
'
' @args
' None
'
' @return
' Integer valid - 0 : not valid, 1 : valid
' Integer frequency - frequency of the backlight PWM
function RokuMfgLegacy_GetBacklightFrequency() as Object
    ret = RokuMfg().call("backlight", {action: "get"})
    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({frequency: ret.data.frequency})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetBacklightLevel
' Directly set the level of the backlight via the PWM hardware
'
' @args
' Integer level : desired percentage of the backlight level
'
' @return
' Integer valid - 0 : not valid, 1 : valid
function RokuMfgLegacy_SetBacklightLevel(level as Integer) as Object
    ret = RokuMfg().call("backlight", {
        action: "set",
        component: "duty",
        data: level
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetBacklightLevel
' Directly get the backlight level from the PWM hardware
'
' @args
' None
'
' @return
' Integer valid - 0 : not valid, 1 : valid
' Integer level - current PWM level as a percentage
function RokuMfgLegacy_GetBacklightLevel() as Object
    ret = RokuMfg().call("backlight", {action: "get"})
    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({level: ret.data.duty})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetBoot2Suspend
' Directly set boot2suspend flag in UBAPP
' This api allow mfg to change the TV bootup to (seudo)suspend, normally TV will either enter display off or standby mode
' Mfg can therefore can quickly start measure the power consumption
'
' @args
' Integer mode   - 0 : disable "boot to suspend";
'                  1 : enable  "boot to suspend + resume on rc power key press + be effective till flag reset"
'                  2 : enable  "boot to suspend + resume on rc power key press + be effective only once"
'                  3 : enable  "boot to suspend + reboot on rc power key press + be effective till flag reset"
'                  4 : enable  "boot to suspend + reboot on rc power key press + be effective only once"
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetBoot2Suspend(mode as Integer) as Object
    if RokuMfgIsSigmaPlatform() then
        return RokuMfgLegacyError("not supported on this platform")
    else if true <> getGlobalAA().mfg_constants.sysinfo.ismanufacturing then
        return RokuMfgLegacyError("requires MFG mode")
    else if 0 > mode or 4 < mode then
        return RokuMfgLegacyError("mode is out of range")
    end if

    ret = RokuMfg().call("ubapp", {
        action: "set",
        data: {
            "boot2suspend": mode
        }
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetBoot2Suspend
' Get boot2suspend flag in UBAPP
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' Integer mode   - 0 : disable "boot to suspend";
'                  1 : enable  "boot to suspend + resume on rc power key press + be effective till flag reset"
'                  2 : enable  "boot to suspend + resume on rc power key press + be effective only once"
'                  3 : enable  "boot to suspend + reboot on rc power key press + be effective till flag reset"
'                  4 : enable  "boot to suspend + reboot on rc power key press + be effective only once"
' String error - Description of error condition.
function RokuMfgLegacy_GetBoot2Suspend() as Object
    if RokuMfgIsSigmaPlatform() then
        return RokuMfgLegacyError("not supported on this platform")
    else if true <> getGlobalAA().mfg_constants.sysinfo.ismanufacturing then
        return RokuMfgLegacyError("requires MFG mode")
    end if

    key = "boot2suspend"
    ret = RokuMfg().call("ubapp", {
        action: "get",
        data: key
    })

    if RokuMfgCheckResponse(ret) then
        mode = val(ret.data[key], 10)
        if 0 > mode or 4 < mode then
            mode = 0
        end if

        return RokuMfgLegacySuccess({
            mode: mode
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetPOSFileStatus
' Get NAND POS File Status (FW-12011)
'
' @args
' None
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String POSFileStatusGood - "true" for good and "false" for not good
' Rest of the entries are mainly for debug purposes;
' See PointOfSalesModeMonitor::GetPOSFileStatus() for complete list of data entries
function RokuMfgLegacy_GetPOSFileStatus() as Object
    ret = RokuMfg().call("pos", {
        action: "get",
        component: "status"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess(ret.data)
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api POSSilentUpdate
' Perform USB Silent Update to NAND POS files
'
' @args
' roAssociativeArray input {
'     String function - what needs to be done
'     String precheck - if MD5 checksum needs to be checked before the copy. If "true",
'                       check the md5sum file on USB against POS file on NAND.
' }
'
' @return
' Integer valid - 0 : function failed; 1 : function succeeded
' String error - Description of error condition.
' String MfgUsbPOSUpdtStatus - see below
'     "succeeded"
'     "failed"
'     "update in progress"
'
' Rest of the entries are mainly for debug purposes;
' See PointOfSalesModeMonitor::POSSilentUpdate() for complete list of data entries
function RokuMfgLegacy_POSSilentUpdate(input as Object) as Object
    if not RokuMfgIsString(input.function) then
        return RokuMfgLegacyError("expected input.function as string")
    else if not RokuMfgIsInvalid(input.precheck) and not RokuMfgIsString(input.precheck) then
        return RokuMfgLegacyError("input.precheck must be a string if provided")
    end if

    pl = {
        action: "update",
        function: input.function
    }

    if RokuMfgIsString(input.precheck) then
        pl.precheck = RokuMfgBoolCast(input.precheck)
    end if

    ret = RokuMfg().call("pos", pl)
    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess(ret.data)
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api VerticalFreqState
' Get/Set the vertical frequency state.
'
' roAssociativeArray input {
'     String action - The action to perform, one of: [ "get", "set" ]
'     String freq - The vertical frequency setting to apply when action="set", one of: [ "50", "60", "panel" (default), "source" ]
'     Boolean force - If true, Force the vFreq to the provided "freq" value when action="set", one of: [ true, false (default) ]
' }
'
' @return
' Integer valid - 0 : not valid (not set); 1 : valid
' Boolean force - current "vfreq force" state if action="get"
' String error - the error description if an error occurred
function RokuMfgLegacy_VerticalFreqState(input as Object) as Object
    ret = RokuMfg().call("vfreq", input)
    if RokuMfgCheckResponse(ret) then
        if not RokuMfgIsInvalid(ret.data) then
            data = ret.data
        else
            data = {}
        end if

        return RokuMfgLegacySuccess(data)
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api ProgramTconI2CDevice
' Program TCON PM/Gamma IC
'
' @args
' String key - the name of the device tp grogram
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - the error code if an error occurred
function RokuMfgLegacy_ProgramTconI2CDevice(key as String) as Object
    ret = RokuMfg().call("hwmodule", {
        action: "program",
        component: "tcon",
        data: key
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function
