'*********************************************************************
'** (c) 2018 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************
' Roku_MFG_Legacy_Common.brs
' Wrappers for the legacy "common" APIs

' @api GetApiVersion
' Get the current version of the MFG API on the system.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String version - Version of the API currently running.
function RokuMfgLegacy_GetApiVersion() as Object
    return RokuMfgLegacySuccess({
        version: getGlobalAA().mfg_constants.libinfo.version
    })
end function

' @api GetApiBuildDate
' Get the API build date.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String str - API build date.
function RokuMfgLegacy_GetApiBuildDate() as Object
    return RokuMfgLegacySuccess({
        str: getGlobalAA().mfg_constants.sysinfo.builddate
    })
end function

' @api GetChipType
' Get the SOC chip family for the current system.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String str - SOC chip family.
function RokuMfgLegacy_GetChipType() as Object
    return RokuMfgLegacySuccess({
        str: RokuMfgGetChipType()
    })
end function

' @api GetUnameString
' Get the kernel information string.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String uname - The kernel uname identifier.
function RokuMfgLegacy_GetUnameString() as Object
    un = getGlobalAA().mfg_constants.sysinfo.uname
    uname_str = un.sysname
    uname_str += " " + un.nodename
    uname_str += " " + un.release
    uname_str += " " + un.version
    uname_str += " " + un.machine
    uname_str += " " + un.domainname
    return RokuMfgLegacySuccess({
        uname: uname_str
    })
end function

' @api GetTVSoftwareVersion
' Get the TV software version.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String str - The current TV software version.
function RokuMfgLegacy_GetTVSoftwareVersion() as Object
    info = getGlobalAA().mfg_constants.sysinfo
    return RokuMfgLegacySuccess({
        str: info.platformid + info.softwareversion + "E" + info.buildnumber + "X"
    })
end function

' @api GetComponentSoftwareVersion
' Get a device component's software version.
' If a component is not supported on this system, it will return an error.
'
' @args
' String component - Name of component to retrieve version information.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String str - Version of the software running on the component.
function RokuMfgLegacy_GetComponentSoftwareVersion(component as String) as Object
    info = getGlobalAA().mfg_constants.platforminfo
    if ("memc" = lcase(component)) then
        if true = info.memcsupported then
            return RokuMfgLegacySuccess({
                str: info.memcversion
            })
        else
            return RokuMfgLegacyError("This platform/device does not support MEMC")
        end if
    else if ("tcon" = lcase(component)) then 
        if true = info.tconsupported then
            return RokuMfgLegacySucess({
                str: info.tconversion
            })
        else
            return RokuMfgLegacyError("This platform/device does not support TCON")
        end if
    end if

    return RokuMfgLegacyError("Unsupported component name given.")
end function

' @api GetSerialNumber
' Get the system's serial number.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String serial - The serial number for this system.
function RokuMfgLegacy_GetSerialNumber() as Object
    return RokuMfgLegacySuccess({
        serial: getGlobalAA().mfg_constants.sysinfo.esn
    })
end function

' @api GetLegacySerialNumber
' Get the system's legacy serial number (device ID).
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String serial - The legacy serial number (device ID) for this system.
function RokuMfgLegacy_GetLegacySerialNumber() as Object
    return RokuMfgLegacySuccess({
        serial: getGlobalAA().mfg_constants.sysinfo.deviceid
    })
end function

' @api GetAQPQVersion
' Get version information for the audio and video databases.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String AQVersion - OEM version string for Audio DB.
' Integer AQMajor - Internal version number for Audio DB.
' String PQVersion - OEM version string for PQ DB.
' Integer PQMajor - Internal major version number for PQ DB.
' Integer PQMinor - Internal minor version number for PQ DB.
' String TSEVersion - TSE (Panel DB) version number, if applicable. "Unknown" on unsupported systems.
' String TSEBuildTime - TSE (Panel DB) build time, if applicable. "Unknown" on unsupported systems.
function RokuMfgLegacy_GetAQPQVersion() as Object
    TSEBuildTime = getGlobalAA().mfg_constants.platforminfo.tsebuildtime
    if RokuMfgIsInvalid(TSEBuildTime) then
        TSEBuildTime = "Unknown"
    end if

    TSEVersion = getGlobalAA().mfg_constants.platforminfo.tseversion
    if RokuMfgIsInvalid(TSEVersion) then
        TSEVersion = "Unknown"
    end if

    pl = {
        action: "get",
        component: "version"
    }

    mfg = RokuMfg()
    pq = mfg.call("pq", pl)
    if not RokuMfgCheckResponse(pq) then
        return RokuMfgLegacyError("error reading PQ version")
    end if

    aq = mfg.call("aq", pl)
    if not RokuMfgCheckResponse(aq) then
        return RokuMfgLegacyError("error reading AQ version")
    end if

    return RokuMfgLegacySuccess({
        AQVersion: aq.data.oem,
        AQMajor: val(aq.data.major, 10),
        AQModelVersion: val(aq.data.model, 10),
        PQVersion: pq.data.oem,
        PQMajor: val(pq.data.major, 10),
        PQMinor: val(pq.data.minor, 10),
        TSEVersion: TSEVersion,
        TSEBuildTime: TSEBuildTime
    })
end function

' @api IsManufacturingMode
' Get whether the system is currently in manufacturing mode or not.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer manufacturingMode - 0 : production mode; 1 : manufacturing mode
function RokuMfgLegacy_IsManufacturingMode() as Object
    if true = getGlobalAA().mfg_constants.sysinfo.ismanufacturing then
        mfgmode = 1
    else
        mfgmode = 0
    end if

    return RokuMfgLegacySuccess({
        manufacturingMode: mfgmode
    })
end function

' @api GetManufacturingYear
' Get the year of manufacture for the system.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer year - Year of manufacture.
function RokuMfgLegacy_GetManufacturingYear() as Object
    ret = RokuMfg().call("pc", {
        action: "get",
        data: "mfgdateyear"
    })

    if RokuMfgCheckResponse(ret) then
        year = val(ret.data.mfgdateyear, 10)

        if 0 <> year then
            return RokuMfgLegacySuccess({
                year: year
            })
        end if
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetManufacturingYear
' Set the year of manufacture for the system.
'
' @args
' Integer year - Year of manufacture.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetManufacturingYear(year as Integer) as Object
    if 2015 > year or 2025 < year then
        return RokuMfgLegacyError("invalid year supplied")
    end if

    ret = RokuMfg().call("pc", {
        action: "set",
        data: {
            mfgdateyear: year
        }
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetManufacturingMonth
' Get the month of manufacture for the system.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer month - Month of manufacture.
function RokuMfgLegacy_GetManufacturingMonth() as Object
    ret = RokuMfg().call("pc", {
        action: "get",
        data: "mfgdatemonth"
    })

    if RokuMfgCheckResponse(ret) then
        month = val(ret.data.mfgdatemonth, 10)

        if 0 <> month then
            return RokuMfgLegacySuccess({
                month: month
            })
        end if
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetManufacturingMonth
' Set the month of manufacture for the system.
'
' @args
' Integer month - Month of manufacture.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetManufacturingMonth(month as Integer) as Object
    if 1 > month or 12 < month then
        return RokuMfgLegacyError("invalid month supplied")
    end if

    ret = RokuMfg().call("pc", {
        action: "set",
        data: {
            mfgdatemonth: month
        }
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetVendorDataVersions
' Get vendor-specific version strings.
' On longview products, these are the main and small MStar qmap versions.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String main - Main vendor value.
' String second - Secondary vendor value.
function RokuMfgLegacy_GetVendorDataVersions() as Object
    info = getGlobalAA().mfg_constants.platforminfo

    if not RokuMfgIsInvalid(info.mainqmap) and not RokuMfgIsInvalid(info.smallqmap) then
        return RokuMfgLegacySuccess({
            main: info.mainqmap,
            second: info.smallqmap
        })
    end if

    return RokuMfgLegacyError("data unavailable")
end function

' @api GetAutoStartMode
' Get whether the system will automatically start the manufacturing application mode.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer value - 0 : does not automatically start app; 1 : app automatically starts
function RokuMfgLegacy_GetAutoStartMode() as Object
    ret = RokuMfg().call("syscfg", {
        action: "get",
        data: "RokuTv_Mfg_Bsc_Mode"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            value: val(ret.data.RokuTv_Mfg_Bsc_Mode, 10)
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetAutoStartMode
' Configure the device to start the manufacturing application automatically.
'
' @args
' Integer enable - 0 : do not start app automatically; 1 : automatically start app
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetAutoStartMode(enable as Integer) as Object
    RokuMfg().call("syscfg", {
        action: "set",
        data: {RokuTv_Mfg_Bsc_Mode: enable}
    })

    return RokuMfgLegacySuccess()
end function

' @api GetBackLitTime
' Get the amount of time the device has been on, and how long the backlight has been on.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer BackLit - The number of minutes the system's back light has been on.
' Integer Total - The number of minutes the system has been on or in display off mode.
function RokuMfgLegacy_GetBackLitTime() as Object
    ret = RokuMfg().call("uptime", {action: "get"})
    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            BackLit: ret.data.Backlight,
            Total: ret.data.Power
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api ClearBackLitTime
' Clear the amount of time the device and backlight has been on.
'
' @args
' String timer - Timer to clear, one of: BackLit, Total
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_ClearBackLitTime() as Object
    ret = RokuMfg().call("uptime", {action: "clear"})
    if RokuMfgCheckResponse(ret)
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api RebootSystem
' Reboot the system.
' Please note that, on success, not all devices will return a value
' before the system reboots.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_RebootSystem() as Object
    ret = RokuMfg().call("systempower", {
        action: "set",
        data: "reboot"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GoToStandby
' Set the system to standby power mode.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_GoToStandby() as Object
    ret = RokuMfg().call("systempower", {
        action: "set",
        data: "standby"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GoToSuspend
' Set the system to suspend power mode.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_GoToSuspend() as Object
    ret = RokuMfg().call("systempower", {
        action: "set",
        data: "suspend"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetNextBootPartition
' Report the partition that is going to be used in the next boot cycle.
'
' Two partitions are available Active(current executing software -typically with mfg image-) and
' Update(typically with prod image), this API reports the boot partition that is going to
' be used in the next booting cycle. After executing API FactoryShopInit, this function should report
' BOOT_PART = Update, indicating that in next boot cycle the system is going to boot with Update partition.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String boot_part - Boot partition to use in the next boot cycle, one of: Active, Update.
function RokuMfgLegacy_GetNextBootPartition() as Object
    ret = RokuMfg().call("partitions", {action: "getnext"})

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            boot_part: ucase(left(ret.data, 1)) + lcase(right(ret.data, len(ret.data) - 1))
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetBootAlternatePartition
' Mark the alternate partition as the next boot candidate.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetBootAlternatePartition() as Object
    ret = RokuMfg().call("partitions", {action: "setalternate"})

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api CalcPartitionCRC
' Calculate a CRC32 value on a partition
'
' @args
' String name - Partition name, one of: Active, Update
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' roArray crc - CRC of the specified partition.
function RokuMfgLegacy_CalcPartitionCRC(partition as String) as Object
    ret = RokuMfg().call("partitions", {
        action: "crc",
        data: partition
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            crc: ret.data
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api WriteTest
' Write random content and validate integrity.
'
' Data is written to the specified path, if possible, and is read back to assess whether
' the data was written successfully.
'
' @args
' String path - The file path to write to.
' Integer kilobytes - The number of kilobytes to write.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_WriteTest() as Object
    return RokuMfgLegacyError("deprecated")
end function

' @api FactoryFlipImage
' Mark the alternate partition as the boot candidate, for the next boot.
' This API fails if the device is not in manufacturing mode.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_FactoryFlipImage() as Object
    return m.SetBootAlternatePartition()
end function

' @api FactoryNVMReset
' Clear the system's non-volatile memory.
' The implementation details of this operation may be vendor/ODM-specific.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_FactoryNVMReset() as Object
    ret = RokuMfg().call("resetnvm")

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api FactoryShopInit
' Finalize factory manufacturing process and set the system to production mode.
' The implementation details of this operation may be vendor/ODM-specific.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_FactoryShopInit() as Object
    ret = RokuMfg().call("shopinit")

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api ClearSettingsData
' Clear the selected set of data settings (partial or full).
' The implementation details of this operation may be vendor/ODM-specific.
'
' @args
' Integer full - 0 : partial clear; 1 : full clear
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_ClearSettingsData(full as Integer) as Object
    ret = RokuMfg().call("clearsettings", {data: {cleanall: true}})

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetCoreTemperature
' Read the core temperature, in degrees Celsius.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer temperature - Temperature of the core, in Celsius.
function RokuMfgLegacy_GetCoreTemperature() as Object
    ret = RokuMfg().call("temperature")

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            temperature: ret.data.coretemp
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetChassisTemperature
' Read the device chassis temperature, in degrees Celsius.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer temperature - Temperature of the chassis, in Celsius.
function RokuMfgLegacy_GetChassisTemperature() as Object
    ret = RokuMfg().call("temperature")

    if RokuMfgCheckResponse(ret) then
        if not RokuMfgIsInvalid(ret.data.chassistemp) then
            return RokuMfgLegacySuccess({
                temperature: ret.data.chassistemp
            })
        else
            return RokuMfgLegacyError("Device does not support chassis temperature")
        end if
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetLED
' Set the LED state.
'
' @args
' String setting - One of: off, on, blink
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetLED(setting as String) as Object
    ret = RokuMfg().call("led", {state: setting})

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetLED2
' Set the second LED state, if applicable.
'
' @args
' String setting - One of: off, on
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetLED2(setting as String) as Object
    ret = RokuMfg().call("led", {
        led: 1,
        state: setting
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api Sync
' Flush file system buffers.
' Force changed blocks back to nand.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_Sync() as Object
    ret = RokuMfg().call("filesys", {action: "sync"})

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api CECConnected
' Return whether the device has CEC connected and has a valid logical address.
'
' @return
' Integer valid - 0 : not valid (CEC is not OK); 1 : valid (CEC is OK)
' String error - Description of error condition.
function RokuMfgLegacy_CECConnected() as Object
    return RokuMfgLegacyError("deprecated")
end function

' @api internal GetDevMode
' Determine if the system has Roku-internal developer mode enabled.
' Note: This is not the same as the channel developer mode.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer devmode - 0 : developer mode is off; 1 : developer mode is on
function RokuMfgLegacy_GetDevMode() as Object
    if false <> getGlobalAA().mfg_constants.sysinfo.issecure then
        devmode = 0
    else
        devmode = 1
    end if

    return RokuMfgLegacySuccess({
        devmode: devmode
    })
end function

' @api GetUSBDevices
' Return known USB devices from the USB subsystem.
'
' This function accepts no arguments. Note that device 0 (e.g., "1-0")
' corresponds to the host controller. Devices ending in any other number
' represent connected devices. This API does not show if a device is a
' hub with devices further down the chain.
'
' Note: Device 1-1 is the wifi module
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' roArray devices - an array of USB "<bus>-<device>" strings.
function RokuMfgLegacy_GetUSBDevices() as Object
' tfranklin: Implementation is broken
    ret = RokuMfg().call("filesys", {
        action: "get",
        data: "usbdevices"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            devices: ret.data
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetPidOverride
' Mark or clear the device with a PID (product ID) override value, in persistent storage.
'
' @args
' String pid - Specifies the PID override value.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetPidOverride(pid as String) as Object
    ret = RokuMfg().call("pc", {
        action: "set",
        data: {
            pid: pid
        }
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetPidOverride
' Determine whether the device has a PID (product ID) override value.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String pid_override - One of: True, False
function RokuMfgLegacy_GetPidOverride() as Object
    ret = RokuMfg().call("pc", {
        action: "get",
        data: "pid"
    })

    if RokuMfgCheckResponse(ret) then
        if "" <> ret.data.pid then
            return RokuMfgLegacySuccess({
                pid_override: ret.data.pid
            })
        end if
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api ResetPcData
' Clear persistent storage and reboot the system.
' Note, not all devices will return a value before reboot process begins.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_ResetPcData() as Object
    ret = RokuMfg().call("pc", {action: "reset"})

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api RescanCustomModel
' Cause the custom package model definition to be reevaluated.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_RescanCustomModel() as Object
    return RokuMfgLegacyError("deprecated")
end function

' @api SetLEDFrame
' Set an LED animation frame.
'
' @args
' roArray obj - Array of integer LED values.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetLEDFrame(obj as Object) as Object
    ret = RokuMfg().call("ledanim", {
        action: "setframe",
        data: obj
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api StartLEDAnimation
' Start an LED animation, driven by a PPM file.
'
' @args
' String anim_file - Thee filename of the PPM file that contains the animation.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_StartLEDAnimation() as Object
    ret = RokuMfg().call("ledanim", {action: "start"})

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api StopLEDAnimation
' Stop a currently-running dynamic LED animation.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_StopLEDAnimation() as Object
    ret = RokuMfg().call("ledanim", {action: "stop"})

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetCannedAnimation
' Start a preset LED animation pattern.
'
' @args
' Integer anim - Preset animation number.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetCannedAnimation(anim as Integer) as Object
    ret = RokuMfg().call("ledanim", {
        action: "setcanned",
        data: anim
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api DoButtonTestLatch
' Start the latch button test.
'
' @args
' String target - mute
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_DoButtonTestLatch(target="" as String) as Object
' Note: target wasn't used in the API, except for a log message
    ret = RokuMfg().call("buttontest", {action: "latch"})

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api DoButtonTestRE
' Start the rotary encoder test.
'
' @args
' Integer threshold - Encoder threshold for success.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_DoButtonTestRE(threshold as Integer) as Object
' tfranklin: need to test this still
    ret = RokuMfg().call("buttontest", {
        action: "re",
        data: threshold
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api internal StartChild
' Starts a child process (in this case, to drive btc/rtlbtmp) over UART.
'
' @args
' roUartConsole uart - UART instance.
' String cmd - One of: btc, rtlbtmp
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_StartChild(uart as Object, cmd as String) as Object
    ret = RokuMfg().call("proc", {
        action: "start",
        uart: uart,
        command: cmd
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api internal SendChild
' Send data to existing child process.
'
' @args
' roAssociativeArray data {
'     roArray args - An array of arguments to send to child process.
' }
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' ksassenrath: Why do args have to be wrapped inside a roAssociativeArray?
function RokuMfgLegacy_SendChild(data as Object) as Object
    if not RokumfgIsAA(data) then
        return RokuMfgLegacyError("expected data as roAssociativeArray")
    else if RokuMfgIsInvalid(data.args) or not RokuMfgIsArray(data.args) then
        return RokuMfgLegacyError("expected data[args] as roArray")
    end if

    ret = RokuMfg().call("proc", {
        action: "send",
        args: data.args
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetDeviceType
' Get the device type
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer tv - 1, if device is a TV.  Otherwise not present.
function RokuMfgLegacy_GetDeviceType() as Object
    retdata = {}
    if "TV" = getGlobalAA().mfg_constants.sysinfo.devicetype then
        retdata.tv = 1
    end if

    return RokuMfgLegacySuccess(retdata)
end function

' @api internal GetBoardRevision
' Return strapping values for all board(s) in the product
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String revision - All revisions, space separated
function RokuMfgLegacy_GetBoardRevision() as Object
' tfranklin: implementation needs ported still
    return RokuMfgLegacyError("needs ported")
end function

' @api EnableCpuBurn
' Starts/Stops cpuburn application.
' Note, api available only on manufacturing images.
'
' @args
' Integer enable - 0 : disable CPU Burn; 1 : enable CPU Burn
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_EnableCpuBurn() as Object
' tfranklin: implementation needs ported still
    return RokuMfgLegacyError("needs ported")
end function

' @api CpuBurnStatus
' Returns whether or not cpuburn app is running.
' Note, api available only on manufacturing images.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' Integer running - 0 : not running; 1 : running
' String error - Description of error condition.
function RokuMfgLegacy_CpuBurnStatus() as Object
' tfranklin: implementation needs ported still
    return RokuMfgLegacyError("needs ported")
end function

' @api GetSPDIFInputStatus
' Returns whether or not a source is connected to the SPDIF input port.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' Integer connected - 0 : source not connected; 1 : source connected
' String error - Description of error condition.
function RokuMfgLegacy_GetSPDIFInputStatus() as Object
' tfranklin: implementation needs ported still
    return RokuMfgLegacyError("needs ported")
end function
