'*********************************************************************
'** (c) 2018-2019 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************
' Roku_MFG_Legacy_AQ.brs
' Wrappers for the legacy "aq" APIs

' @api SaveAQSettings
' Save the current AQ Database to persistent storage.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SaveAQSettings() as Object
    ret = RokuMfg().call("aq", {action: "save"})

    if RokuMfgCheckResponse(ret)
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError()
end function

' @api GetAudioVolume
' Get the current TV volume.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer value - The current volume setting.
function RokuMfgLegacy_GetAudioVolume() as Object
    ret = RokuMfg().call("aq", {
        action: "get",
        component: "volume"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({value: ret.data})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetAudioVolume
' Set the TV volume.
'
' @args
' Integer volume - The desired volume value.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetAudioVolume(volume as Integer) as Object
    ret = RokuMfg().call("aq", {
        action: "set",
        component: "volume",
        data: volume
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError()
end function

' @api EnableSetAudioVolume
' NOTE: This API is deprecated as of December 2018
' Enable changing the  volume.
'
' @args
' None
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_EnableSetAudioVolume() as Object
    return RokuMfgLegacyError("Deprecated as of December 2018")
end function

' @api SetAudioVolumeFade
' NOTE: This API is deprecated as of December 2018
' Set the TV volume fader.
' This function will instruct the platform to transition from the current volume level to the target
' volume level smoothly, over a set duration.  Additionally, a fade type can be specified, which will
' apply an easing function to the fade.
'
' @args
' Integer volume - The target volume value.
' Integer duration - The duration of the fade, in milliseconds.
' Integer type - The type of volume fade (easing function). 0 for linear, 1 for cubic in, 2 for cubic out.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetAudioVolumeFade(volume as Integer, duration as Integer, f_type as Integer) as Object
    return RokuMfgLegacyError("Deprecated as of December 2018")
end function

' @api GetAudioDelay
' Get the delay value for the provided audio path, in milliseconds.
'
' @args
' String path - The audio path to get audio delay information for. Supported paths: speakers, headphones, lineout, spdif
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer delay - The audio delay, in milliseconds.
function RokuMfgLegacy_GetAudioDelay(path as String) as Object
    ret = RokuMfg().call("aq", {
        action: "get",
        component: "delay",
        data: path
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({delay: ret.data})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetAudioDelay
' Set the delay for the provided audio path, in milliseconds.
'
' @args
' String path - The audio path to adjust delay for. Supported paths: speakers, headphones, lineout, spdif
' Integer delay - The amount of delay, in milliseconds.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetAudioDelay(path as String, delay as Integer) as Object
    ret = RokuMfg().call("aq", {
        action: "set",
        component: "delay",
        dest: path,
        data: delay
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetAudioMute
' Get the current audio mute state.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer state - 0 : not muted; 1 : muted
function RokuMfgLegacy_GetAudioMute() as Object
    ret = RokuMfg().call("aq", {
        action: "get",
        component: "mute"
    })

    if RokuMfgCheckResponse(ret) then
        if true = ret.data then
            state = 1
        else
            state = 0
        end if

        return RokuMfgLegacySuccess({state: state})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetAudioMute
' Set the current audio mute state.
'
' @args
' Integer muted - 0 : not muted; 1 : muted
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetAudioMute(muted as Integer) as Object
    ret = RokuMfg().call("aq", {
        action: "set",
        component: "mute",
        data: RokuMfgBoolCast(muted)
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetAudioCurve
' (Deprecated as of January 2016) Get the current audio curve, given the set of data points.
'
' @args
' String name - The name of audio curve: "headphone" or "speaker"
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer count - The number of reference points in this array.
' Integer refPtN - The UI value used as a reference point (x-axis) in audio curve calculations.  N is a digit in the range [0:7]
' Integer valueN - The internal hardware value associated with the equivalent refPtN.  N is a digit within the range [0:7]
function RokuMfgLegacy_GetAudioCurve(name as String) as Object
    return RokuMfgLegacyError("Deprecated as of January 2016")
end function

' @api SetAudioCurve
' (Deprecated as of January 2016) Set the audio curve.
' The audio curve is specified as a set of data points (max: 8).
'
' @args
'     roAssociativeArray data {
'     String curve - The name of audio curve: "headphone" or "speaker"
'     Integer refPtN - The UI value used as a reference point (x-axis) in audio curve calculations.  N is a digit in the range [0:7]
'     Integer valueN - The internal hardware value associated with the equivalent refPtN.  N is a digit within the range [0:7]
' }
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetAudioCurve(data as Object) as Object
    return RokuMfgLegacyError("Deprecated as of January 2016")
end function

' @api GetAudioPrescalerGain
' Get the audio prescaler gain value for a given input.
'
' @args
' String input - The name of the input source to query.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer value - The current prescaler gain setting.
function RokuMfgLegacy_GetAudioPrescalerGain(input as String) as Object
    ret = RokuMfg().call("aq", {
        action: "get",
        component: "prescalergain",
        data: input
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({value: ret.data})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetAudioPrescalerGain
' Set the audio prescaler gain value for a given input.
'
' @args
' String input - The name of the input source to query.
' Integer value - The prescaler gain setting to use.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetAudioPrescalerGain(input as String, value as Integer) as Object
    ret = RokuMfg().call("aq", {
        action: "set",
        component: "prescalergain",
        source: input,
        data: value
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetAudioAmplifierGain
' Get the current audio amplifier gain value.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer value - The current amplifier gain setting.
function RokuMfgLegacy_GetAudioAmplifierGain() as Object
    ret = RokuMfg().call("aq", {
        action: "get",
        component: "amplifiergain"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({value: ret.data})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetAudioAmplifierGain
' Set the audio amplifier gain value.
'
' @args
' Integer gain - Amplifier gain setting.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetAudioAmplifierGain(gain as Integer) as Object
    ret = RokuMfg().call("aq", {
        action: "set",
        component: "amplifiergain",
        data: gain
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetAudioAmplifierRegister
' Get the specified audio amplifier register value.
'
' @args
' Integer regaddr - The address of the audio amplifier register to access.
' Integer length - the number of bytes to read from the audio amplifier.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String value - a colon-spearated list of hexadecimal values representing the read values.
function RokuMfgLegacy_GetAudioAmplifierRegister(regaddr as Integer, length as Integer) as Object
    ret = RokuMfg().call("aq", {
        action: "get",
        component: "amplifierregister",
        address: regaddr,
        length: length
    })

    if RokuMfgCheckResponse(ret) then
        value = ""
        for each el in ret.data
            to_append = stri(el, 16)
            while (len(to_append) < 2)
                to_append = "0" + to_append
            end while

            value = value + to_append + ":"
        end for

        if len(value) > 0 then
            value = left(value, len(value) - 1)
        end if

        return RokuMfgLegacySuccess({value: value})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetAudioAmplifierRegister
' Set the specified audio amplifier register value.
'
' @args
' roAssociativeArray data {
'     Integer address - The address of the audio amplifier register to access.
'     roArray values - An array of values representing data to write.
' }
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetAudioAmplifierRegister(data as Object) as Object
    if not RokuMfgIsAA(data) then
        return RokuMfgLegacyError("expected data as roAssociativeArray")
    else if not RokuMfgIsInt(data.address) then
        return RokuMfgLegacyError("expected data.address as Integer")
    else if not RokuMfgIsArray(data.values) then
        return RokuMfgLegacyError("expected data.values as roArray")
    end if

    ret = RokuMfg().call("aq", {
        action: "set",
        component: "amplifierregister",
        address: data.address
        data: data.values
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetAudioAmplifierEQBypass
' Get the current audio amplifier EQ bypass setting.
' When this setting is enabled, all audio EQ settings are effectively disabled.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer state - The current EQ bypass setting: 0: bypass inactive; 1: bypass active
function RokuMfgLegacy_GetAudioAmplifierEQBypass() as Object
    ret = RokuMfg().call("aq", {
        action: "get",
        component: "amplifiereqbypass"
    })

    if RokuMfgCheckResponse(ret) then
        if true = ret.data then
            state = 1
        else
            state = 0
        end if

        return RokuMfgLegacySuccess({state: state})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetAudioAmplifierEQBypass
' Set the current Audio Amplifier EQ bypass setting.
' When this setting is enabled, all audio EQ settings are effectively disabled.
'
' @args
' Integer state - 0: disable bypass; 1: enable bypass
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetAudioAmplifierEQBypass(state as Integer) as Object
    ret = RokuMfg().call("aq", {
        action: "set",
        component: "amplifiereqbypass",
        enable: RokuMfgBoolCast(state)
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetAudioAmplifierDRCEnable
' Get the current audio amplifier's DRC enable setting.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer state - 0: DRC disabled; 1: DRC enabled
function RokuMfgLegacy_GetAudioAmplifierDRCEnable() as Object
    ret = RokuMfg().call("aq", {
        action: "get",
        component: "amplifierdrcenable"
    })

    if RokuMfgCheckResponse(ret) then
        if true = ret.data then
            state = 1
        else
            state = 0
        end if

        return RokuMfgLegacySuccess({state: state})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetAudioAmplifierDRCEnable
' Set the current audio amplifier's DRC enable setting.
'
' @args
' Integer state - 0: DRC disabled; 1: DRC enabled
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetAudioAmplifierDRCEnable(state as Integer) as Object
    ret = RokuMfg().call("aq", {
        action: "set",
        component: "amplifierdrcenable",
        data: RokuMfgBoolCast(state)
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError()
end function

' @api GetAudioAmplifierParameters
' (Deprecated as of June 2017) Get the specific audio amplifier parameter.
'
' When provided with an input file of addresses and length, this function fetches the associated values.
'
' The input format of this file is as follows:
' <br/><br/>
' address # : # length in bytes<br/>
' (ex.)<br/>
' cat test.cfg<br/>
' 0x0:1<br/>
' 0x1:1<br/>
' 0x2:1<br/>
' ...<br/>
' 0x20:4<br/>
'
' @args
' String file - Path to a file in the aforementioned format.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String value - A list of values returned by the amplifier.
function RokuMfgLegacy_GetAudioAmplifierParameters(file as String) as Object
    return RokuMfgLegacyError("Deprecated as of June 2017")
end function

' @api SetAudioAmplifierParameters
' (Deprecated as of June 2017) Set audio amplifier parameters from an input file.
'
' The format of the input file is as follows:
' address # : # length in bytes : contents, ... |
' (ex.)
' cat test.cfg
' 0x0:0x1:74|
' 0x1:0x1:c1|
' 0x2:0x1:00|
' ...
' 0x20:0x4:00, 01, 77, 72|
'
' @args
' String file - Path to a file in the aforementioned format.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetAudioAmplifierParameters(file as String) as Object
    return RokuMfgLegacyError("Deprecated as of June 2017")
end function

' @api deprecated GetAllAudioEQModes
' (Deprecated as of July 2018) Get all available audio EQ modes.
' This just returns an error.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer count - The number of audio modes found.
' String names - A comma-separated list of EQ mode names.
' String [EQ] - A commma-separated list of curve points for a particular [EQ] mode.
function RokuMfgLegacy_GetAllAudioEQModes() as Object
    return RokuMfgLegacyError("Deprecated as of June 2018")
end function

' @api GetAudioEQMode
' Get the current EQ mode.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String mode - The current EQ mode.
function RokuMfgLegacy_GetAudioEQMode() as Object
    ret = RokuMfg().call("aq", {
        action: "get",
        component: "eqmode"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({mode: ret.data})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetAudioEQMode
' Set the audio EQ mode.
'
' @args
' String mode - The audio EQ mode to be changed to.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetAudioEQMode(mode as String) as Object
    ret = RokuMfg().call("aq", {
        action: "set",
        component: "eqmode",
        data: mode
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api deprecated GetAudioEQParameters
' (Deprecated as of July 2018) Get the parameters for the given audio EQ mode.
' This just returns an error.
'
' @args
' String mode - the audio EQ mode to query.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String value - A comma-separated string of 5 values.
function RokuMfgLegacy_GetAudioEQParameters(mode as String) as Object
    return RokuMfgLegacyError("Deprecated as of June 2018")
end function

' @api deprecated SetAudioEQParameters
' (Deprecated as of July 2018) Set the audio EQ parameters for a particular mode.
' This just returns an error.
'
' @args
' roAssociativeArray data {
'     String mode - Name of EQ mode to change.  Can also be "current" to modify current EQ mode.
'     roArray values - Array of five values to associate with the given mode.
' }
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetAudioEQParameters(data as Object) as Object
    return RokuMfgLegacyError("Deprecated as of June 2018")
end function

' @api GetAudioPEQEnabled
' Determine if audio PEQ is enabled.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer value - 0 : disabled; 1 : enabled
function RokuMfgLegacy_GetAudioPEQEnabled() as Object
    ret = RokuMfg().call("aq", {
        action: "get",
        component: "peq"
    })

    if RokuMfgCheckResponse(ret) then
        if true = ret.data then
            state = 1
        else
            state = 0
        end if

        return RokuMfgLegacySuccess({value: state})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api EnableAudioPEQ
' Enable or disable audio PEQ.
'
' @args
' Integer enable - 0 : disable; 1 : enable
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_EnableAudioPEQ(enable as Integer) as Object
    ret = RokuMfg().call("aq", {
        action: "set",
        component: "peq",
        enable: RokuMfgBoolCast(enable)
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetAudioSurroundModeEnabled
' Determine if surround sound is enabled.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer value - 0 : disabled; 1 : enabled
function RokuMfgLegacy_GetAudioSurroundModeEnabled() as Object
    ret = RokuMfg().call("aq", {
        action: "get",
        component: "surroundmode"
    })

    if RokuMfgCheckResponse(ret) then
        if true = ret.data then
            state = 1
        else
            state = 0
        end if

        return RokuMfgLegacySuccess({value: state})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetAudioSurroundBypassEnabled
' Determine if surround sound bypass is enabled.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer value - 0 : disabled; 1 : enabled
function RokuMfgLegacy_GetAudioSurroundBypassEnabled() as Object
    ret = RokuMfg().call("aq", {
        action: "get",
        component: "surroundmodebypass"
    })

    if RokuMfgCheckResponse(ret) then
        if true = ret.data then
            state = 1
        else
            state = 0
        end if

        return RokuMfgLegacySuccess({value: state})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api EnableAudioSurroundMode
' Enable or disable surround sound.
'
' @args
' Integer enable - 0 : disable; 1 : enable
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_EnableAudioSurroundMode(enable as Integer) as Object
    ' Some platforms enable/disable Surround Mode by toggling the bypass state.
    bypass = {
        "Sigma UXL-38": "",
        "MStar-T10": "",
        "MStar-T14": "",
        "MStar-T22": ""
    }

    flag = RokuMfgBoolCast(enable)
    component = "surroundmode"
    if not RokuMfgIsInvalid(bypass[RokuMfgGetChipType()]) then
        flag = not flag
        component = component + "bypass"
    end if

    ret = RokuMfg().call("aq", {
        action: "set",
        component: component,
        enable: flag
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetAudioDestinations
' Get all possible audio destinations.
' Each destination supported on this platform will be added as a key in the returned roAssociativeArray.
' Possible audio destinations:
'    hdmi, wifiremote, headphones, speakers, arc, spdif, lineout, datagram
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer [destination] - [destination] is supported on this platform.
function RokuMfgLegacy_GetAudioDestinations() as Object
    ret = RokuMfg().call("aq", {
        action: "get",
        component: "destinations"
    })

    if RokuMfgCheckResponse(ret) then
        data = {}
        for each dest in ret.data
            data[dest] = 1
        end for

        return RokuMfgLegacySuccess(data)
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api DoTFTPFetchSetSinglePtMicEQ
' Get Single point microphone EQ values from TFTP server and install
'       in Persistent Configuration.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_DoTFTPFetchSetSinglePtMicEQ() as Object
    mfg = RokuMfg()
    info = mfg.sysInfo()
    if false = info.isFarField then
        return RokuMfgLegacyError("unsupported on this platform")
    end if

    serverIP = RokuMfgState().get("TFTPServer")
    if RokuMfgIsInvalid(serverIP) or "" = serverIP then
        return RokuMfgLegacyError("Need to set tftp server")
    end if

    local = "tmp:/miceqSinglePt"
    ret = mfg.call("tftp", {
        action: "get",
        local: local,
        remote: "miceqSinglePt" + info.esn,
        address: serverIP
    })

    if not RokuMfgCheckResponse(ret) then
        return RokuMfgLegacyError(ret.header_.description)
    end if

    ba = createObject("roByteArray")
    ba.readFile(local)
    ret = mfg.call("pc", {
        action: "set",
        data: {"miceqSinglePt": ba.toAsciiString()}
    })

    if RokuMfgCheckResponse() then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api DoTFTPFetchSetMultiPtMicEQ
' Get Multi point microphone EQ values from TFTP server and install
'       in Persistent Configuration.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' TODO: For now only the calibration from 1KHz is being used. Calibration values
'         from 300Hz and 4KHz are being stored to PC, but not being used.
function RokuMfgLegacy_DoTFTPFetchSetMultiPtMicEQ() as Object
    mfg = RokuMfg()
    info = mfg.sysInfo()
    if false = info.isFarField then
        return RokuMfgLegacyError("unsupported on this platform")
    end if

    serverIP = RokuMfgState().get("TFTPServer")
    if RokuMfgIsInvalid(serverIP) or "" = serverIP then
        return RokuMfgLegacyError("Need to set tftp server")
    end if

    local = "tmp:/miceqMultiPt"
    ret = mfg.call("tftp", {
        action: "get",
        local: local,
        remote: "miceqMultiPt" + info.esn,
        address: serverIP
    })

    if not RokuMfgCheckResponse(ret) then
        return RokuMfgLegacyError(ret.header_.description)
    end if

    ba = createObject("roByteArray")
    ba.readFile(local)
    ret = mfg.call("pc", {
        action: "set",
        data: {"miceqMultiPt": ba.toAsciiString()}
    })

    if RokuMfgCheckResponse() then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetMicCompare
' Capture audio from all microphones and calculate relative gain settings for each
'
' @args
' None
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_GetMicCompare() as Object
    ' FIXME: tfranklin: I wasn't able to build and verify this functionality after
    ' moving it from the cltvmanufacturing AQ APIs because it depends on
    ' BUILD_SUPPORT_FARFIELD_BOARD, which isn't currently used for any
    ' available Roku platforms. Someone more familiar with this area should
    ' provide a cleaner implementation that goes through this function when
    ' support is necessary again
    return RokuMfgLegacyError("deprecated until needed")
end function

' @api StartAudioCapture
' Capture audio from one or all microphones.
'
' @args
' Integer micno - The number of the microphone to capture from, starting with 0. -1 will capture from all mics.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_StartAudioCapture(micno as Integer) as Object
    ' FIXME: tfranklin: I wasn't able to build and verify this functionality after
    ' moving it from the cltvmanufacturing AQ APIs because it depends on
    ' BUILD_SUPPORT_FARFIELD_BOARD, which isn't currently used for any
    ' available Roku platforms. Someone more familiar with this area should
    ' provide a cleaner implementation that goes through this function when
    ' support is necessary again
    return RokuMfgLegacyError("deprecated until needed")
end function

' @api EndAudioCapture
' Finish an active audio capture.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_EndAudioCapture() as Object
    ' FIXME: tfranklin: I wasn't able to build and verify this functionality after
    ' moving it from the cltvmanufacturing AQ APIs because it depends on
    ' BUILD_SUPPORT_FARFIELD_BOARD, which isn't currently used for any
    ' available Roku platforms. Someone more familiar with this area should
    ' provide a cleaner implementation that goes through this function when
    ' support is necessary again
    return RokuMfgLegacyError("deprecated until needed")
end function

' @api SetAudioOutput
' Set the audio output to the given destination.
'
' @args
' String path - The audio path to send audio output to.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetAudioOutput(path as String) as Object
    ret = RokuMfg().call("aq", {
        action: "set",
        component: "output",
        data: path
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetAudioInput
' Set the audio input to the given destination
'
' @args
' String path - The audio path to send audio input to.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetAudioInput(path as String) as Object
    ret = RokuMfg().call("aq", {
        action: "set",
        component: "input"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetDTSParameter
' Set a DTS parameter.
' The DTS parameter itself is currently specified as an index.  Additionally,
' this api expects a String to be passed for the value, but the value itself
' may be an integer or double.  In this case, pass a string representing the
' actual value (e.g. "3.14" for a floating value, or "4" for an integer).
'
' @args
' Integer index - DTS tuning parameter index.
' String value - A value of the appropriate type for the specified parameter.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetDTSParameter(index as Integer, value as String) as Object
    ret = RokuMfg().call("dts", {
        action: "set",
        component: "params",
        index: index,
        value: value
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetDTSParameter
' Get the DTS parameter index's name, data type, and value.
'
' @args
' Integer index - The DTS tuning parameter index.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String name - DTS parameter name associated with the given index.
' String type - The data type of the DTS parameter.  One of "Integer", "Double", or "String".
' Dynamic value - The value for the DTS parameter.
' Dynamic min - The minimum DTS value, if the type is "Integer" or "Double"
' Dynamic max - The maximum DTS value, if the type is "Integer" or "Double"
' Dynamic step - The step size for the DTS vaslue, if the type is "Integer" or "Double"
function RokuMfgLegacy_GetDTSParameter(index as Integer) as Object
    ret = RokuMfg().call("dts", {
        action: "get",
        component: "params",
        index: index
    })

    if RokuMfgCheckResponse(ret) then
        data = {
            name: ret.name,
            type: ret.type,
            value: ret.value
        }

        ' It's possible ret.range won't be populated, so we need to validate
        if RokuMfgIsAA(ret.range) then
            data.min = ret.range.min
            data.max = ret.range.max
            data.step = ret.range.step
        end if

        return RokuMfgLegacySuccess(data)
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetDTSUnitEnable
' Determine if a DTS SRS unit function is enabled, by index.
'
' @args
' Integer index - DTS unit function index.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer value - 0 : disabled; 1 : enabled
function RokuMfgLegacy_GetDTSUnitEnable(index as Integer) as Object
    ret = RokuMfg().call("dts", {
        action: "get",
        component: "enable",
        index: index
    })

    if RokuMfgCheckResponse(ret) then
        if true = ret.data.enable then
            state = 1
        else
            state = 0
        end if

        return RokuMfgLegacySuccess({value: state})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetDTSUnitEnable
' Enable/Disable a DTS SRS unit function, by index.
'
' @args
' Integer index - An index value in the range [0:17]
' Integer enable - 0 : disable; 1 : enable
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetDTSUnitEnable(index as Integer, enable as Integer) as Object
    ret = RokuMfg().call("dts", {
        action: "set",
        component: "enable",
        index: index,
        enable: RokuMfgBoolCast(enable)
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetTFTPAddress
' Set the IP address of the TFTP server.
'
' @args
' String serverIP - The IPv4 address of the TFTP server.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String ip - The formatted IPv4 address just set, if valid.
function RokuMfgLegacy_SetTFTPAddress(serverIP as String) as Object
    regex = createObject("roRegex", "^[0-9.]*$", "i")
    if not regex.isMatch(serverIP) then
        return RokuMfgLegacyError("invalid IP address format")
    end if

    tok = serverIP.tokenize(".")
    if 4 <> tok.count() then
        return RokuMfgLegacyError("invalid IP address format")
    end if

    for each el in tok
        intval = val(el, 10)
        if 0 > intval or 255 < intval then
            return RokuMfgLegacyError("invalid IP address format")
        end if
    end for

    state = RokuMfgState()
    state.set("TFTPServer", serverIP)
    state.set("TFTPAudioGood", false)
    state.set("TFTPPath", "")
    return RokuMfgLegacySuccess({ip: serverIP})
end function

' @api SetTFTPAudioName
' Set the remote name of the file used by the play audio command.
'
' @args
' String remoteFilename - The name of the file to fetch from TFTP server.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetTFTPAudioName(remoteFilename as String) as Object
    if not RokuMfgIsFilenameGood(remoteFilename) then
        return RokuMfgLegacyError("invalid filename format")
    end if

    state = RokuMfgState()
    state.set("TFTPRemoteFilename", remoteFilename)
    state.set("TFTPAudioGood", false)
    return RokuMfgLegacySuccess()
end function

' @api SetTFTPPath
' Set the path on the TFTP server.
'
' @args
' String path - The path on the TFTP server.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetTFTPPath(path as String) as Object
    if not RokuMfgIsFilenameGood(path) then
        return RokuMfgLegacyError("invalid path format")
    end if

    state = RokuMfgState()
    state.set("TFTPPath", path)
    return RokuMfgLegacySuccess()
end function

' @api DoTFTPFetchAudio
' Fetch audio sample from TFTP server and report the local name of file.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String filename - The name of the local file, if valid.
' String extension - The extension of the local file, if valid.
function RokuMfgLegacy_DoTFTPFetchAudio() as Object
    state = RokuMfgState()
    remoteFilename = state.get("TFTPRemoteFilename")
    serverIP = state.get("TFTPServer")
    if RokuMfgIsInvalid(remoteFilename) or "" = remoteFilename then
        return RokuMfgLegacyError("Need to set tftp audio")
    else if RokuMfgIsInvalid(serverIP) or "" = serverIP then
        return RokuMfgLegacyError("Need to set tftp server")
    end if

    name = "tmp:/tftpAudio."
    tok = remoteFilename.tokenize(".")
    if tok.count() > 1 then
        ext = tok[tok.count() - 1]
    else
        ext = ""
    end if

    name = name + ext
    audioGood = state.get("TFTPAudioGood")
    if false = audioGood then
        ret = RokuMfg().call("tftp", {
            action: "get",
            local: name,
            remote: remoteFilename,
            address: serverIP
        })

        if not RokuMfgCheckResponse(ret) then
            return RokuMfgLegacyError(ret.header_.description)
        end if
    end if

    state.set("TFTPAudioGood", true)
    return RokuMfgLegacySuccess({
        filename: name,
        extension: ext
    })
end function

' @api DoInstallOOBEFromFile
' Fetch OOBE blob from OOBE file (indexed by chipID)
'
' @args
' String path - The path to the OOBE file
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_DoInstallOOBEFromFile(path as String) as Object
    ret = RokuMfg().call("oobe", {
        action: "install",
        component: "file",
        data: path
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api DoInstallOOBE
' Fetch OOBE blob from the TFTP server and store it in persistent storage.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_DoInstallOOBE() as Object
    ret = RokuMfg().call("oobe", {
        action: "install",
        component: "tftp",
        data: RokuMfgState().get("TFTPServer")
        path: RokuMfgState().get("TFTPPath")
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api DoVerifyOOBE
' Check validity of the OOBE block from #DoInstallOOBE
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_DoVerifyOOBE() as Object
    ret = RokuMfg().call("oobe", {action: "verify"})
    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api DspOutCapture
' Capture audio from all channels of DSP out and send to TFTP server.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_DspOutCapture() as Object
    ' FIXME: tfranklin: I wasn't able to build and verify this functionality after
    ' moving it from the cltvmanufacturing AQ APIs because it depends on
    ' BUILD_SUPPORT_FARFIELD_BOARD, which isn't currently used for any
    ' available Roku platforms. Someone more familiar with this area should
    ' provide a cleaner implementation that goes through this function when
    ' support is necessary again
    return RokuMfgLegacyError("deprecated until needed")
end function

' @api SetSpeakerGainCalibration
' Write calibration values for Main speaker and Tweeter
'
' @args
' String woofercalibtarget: Calibration target for main speaker
' String woofercalib: Calibration for main speaker
' String tweetercalibtarget: Calibraiton target for tweeter
' String tweetercalib: Calibraiton for tweeter
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String result - detailed description
' String error - Description of error condition.
function RokuMfgLegacy_SetSpeakerGainCalibration(w_target as String, w_calib as String, t_target as String, t_calib as String) as Object
    data = {
        w_target: {
            asString: w_target,
            asFloat: w_target.toFloat()
        },
        w_calib: {
            asString: w_calib,
            asFloat: w_calib.toFloat()
        },
        t_target: {
            asString: t_target,
            asFloat: t_target.toFloat()
        },
        t_calib: {
            asString: t_calib,
            asFloat: t_calib.toFloat()
        }
    }

    regex = createObject("roRegex", "^[0-9.]*$", "i")
    for each el in data
        if not regex.isMatch(data[el].asString) then
            return RokuMfgLegacyError(el + " has an invalid format")
        end if
    end for

    pl = {
        action: "set",
        data: {
            "woofercalibtarget": data[w_target].asFloat,
            "woofercalib": data[w_calib].asFloat,
            "tweetercalibtarget": data[t_target].asFloat,
            "tweetercalib": data[t_calib].asFloat
        }
    }

    ret = RokuMfg().call("pc", pl)
    if RokuMfgCheckResponse(ret) then
        resultstr = "Got input woofercalibtarget: "
        resultstr = resultstr + RokuMfgStrCast(data[w_target].asFloat) + ", woofercalib: "
        resultstr = resultstr + RokuMfgStrCast(data[w_calib].asFloat) + ", tweetercalibtarget: "
        resultstr = resultstr + RokuMfgStrCast(data[t_target].asFloat) + ", tweetercalib: "
        resultstr = resultstr + RokuMfgStrCast(data[t_calib].asFloat)
        return RokuMfgLegacySuccess({result: resultstr})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

