'*********************************************************************
'** (c) 2018-2019 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************
' Roku_MFG_Events.brs
' Handle library-specifc event messages from clmanufacturing port
Library "state/Roku_MFG_State.brs"

function RokuMfgEvents_CheckEvent() as Object
    msg = m.port.getMessage()
    if "roManufacturingEvent" = type(msg) then
        if RokuMfgEvents_HandleEvent(msg) then
            return invalid
        end if
    end if

    return msg
end function

function RokuMfgEvents_ParseEventRoot(msg as Object) as Object
    if invalid <> findMemberFunction(msg, "getInfo") then
        info = msg.getInfo()
        if invalid <> info and RokuMfgIsString(info.root) then
            return parseJSON(info.root)
        end if
    end if

    return {
        type: invalid,
        data: {}
    }
end function

function RokuMfgEvents_HandleEvent(msg as Object) as Boolean
    ' Handler implementations should return true if the event should be
    ' eaten and false if we should return the message to the caller.
    ' This allows us to obfuscate events we don't want outside apps to know
    ' about (e.g., signal states) and sniff data to maintain internal state
    ' while still passing the event to the caller (e.g., wifi state).
    handlers = {
        childproc: RokuMfgEvents_ChildProc,
        tvsignalstate: RokuMfgEvents_TVSignalState,
        tvsignalmode: RokuMfgEvents_TVSignalMode,
        wifidevice: RokuMfgEvents_WifiDevice,
        wifistatus: RokuMfgEvents_WifiStatus
    }

    event = RokuMfgEvents_ParseEventRoot(msg)
    if invalid <> event.type then
        if not RokuMfgIsInvalid(handlers[event.type]) then
            return handlers[event.type](event.data)
        end if
    end if

    return false
end function

function RokuMfgEvents_ChildProc(data as Object) as Boolean
    uart = RokuMfgState().get("ProcUart")
    if RokuMfgHasInterface(uart, "ifUartConsole") then
        uart.SendOutputWithNL(data)
    end if

    ' This event is handled, don't forward to caller
    return true
end function

function RokuMfgEvents_TVSignalState(data as Object) as Boolean
    states = [
        "nosignal",
        "validsignal",
        "audioonly",
        "audioonly_nosignal",
        "error",
        "unknown"
    ]

    ' Unknown is state "-1", so wrap it at the end for easier index access
    RokuMfgState().set("TVSignalState", states[RokuMfgModulo(data, states.count())])

    ' This event is handled, don't forward to caller
    return true
end function

function RokuMfgEvents_TVSignalMode(data as Object) as Boolean
    RokuMfgState().set("TVSignalMode", data)

    ' This event is handled, don't forward to caller
    return true
end function

function RokuMfgEvents_WifiDevice(data as Object) as Boolean
    if not RokuMfgIsInvalid(data.status) and 0 = data.status then
        RokuMfgState().set("WifiDevicePresent", data.present)
    end if

    ' Forward this event to caller
    return false
end function

function RokuMfgEvents_WifiStatus(data as Object) as Boolean
    if not RokuMfgIsInvalid(data.status) and 0 = data.status then
        RokuMfgState().set("WifiAPList", data.ap_list)
    end if

    ' Forward this event to caller
    return false
end function
