'*********************************************************************
'** (c) 2018-2019 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************
' Roku_MFG_API_Tuner.brs
' Implementation for Tuner-related APIs
Library "state/Roku_MFG_State.brs"

function RokuMfgApi_Tuner(mfg as Object, pl as Object) as Object
    state = RokuMfgState()

    restoreTuner = false
    needStateRefresh = false
    if "scan" = pl.action then
        if RokuMfgIsString(pl.medium) and (pl.medium <> "air" and pl.medium <> "cable") then
            return RokuMfgBadDataStatus("expected medium as one of [ air, cable ]")
        end if

        ' We can't scan if we're using the tuner already. Move away from the
        ' tuner, scan, then come back.
        if "dtv" = state.get("CurrentInputSource") then
            RokuMfg().call("inputsource", {action: "set", data: "ui"})
            restoreTuner = true
        end if

        needStateRefresh = true
    else if "load" = pl.action then
        locations = {
            "custom": "/custom/presets/",
            "usb": "/media/ext1:/"
        }

        location = "custom"
        if not RokuMfgIsInvalid(pl.location) then
            if not RokuMfgIsString(pl.location) then
                return RokuMfgBadDataStatus("expected location as string")
            else
                if RokuMfgIsInvalid(locations[pl.location]) then
                    return RokuMfgBadDataStatus("unknown location")
                else if "usb" = pl.location and (not pl.header_.is_manufacturing and pl.header_.is_secure) then
                    return RokuMfgMfgModeStatus()
                end if

                location = pl.location
            end if
        end if

        suffix = ""
        if not RokuMfgIsInvalid(pl.data) then
            if not RokuMfgIsString(pl.data) then
                return RokuMfgBadDataStatus("expected data as filename suffix string")
            end if

            ' Empty data is a valid string, but indicates we don't want a suffix
            if "" <> pl.data then
                suffix = "-" + pl.data
            end if
        end if

        standard = RokuMfgApi_Tuner_GetStandard()

        ret = RokuMfgCallExternal(mfg, "tuner", {
            header_: pl.header_,
            action: pl.action,
            data: locations[location] + standard + "-channels" + suffix + ".db"
        })

        if RokuMfgCheckResponse(ret) then
            ' Invalidate the current tuner cache data and force a refresh
            state.tuner.invalidate(true)
        end if

        return ret
    else if "save" = pl.action then
        if not pl.header_.is_manufacturing and pl.header_.is_secure then
            return RokuMfgMfgModeStatus()
        end if

        suffix = ""
        if not RokuMfgIsInvalid(pl.data) then
            if not RokuMfgIsString(pl.data) then
                return RokuMfgBadDataStatus("expected data as filename suffix string")
            end if

            ' Empty data is a valid string, but indicates we don't want a suffix
            if "" <> pl.data then
                suffix = "-" + pl.data
            end if
        end if

        standard = RokuMfgApi_Tuner_GetStandard()

        ret = RokuMfgCallExternal(mfg, "tuner", {
            header_: pl.header_,
            action: pl.action,
            data: "/media/ext1:/" + standard + "-channels" + suffix + ".db"
        })

        RokuMfg().call("filesys", {action: "sync"})
        return ret
    else if "get" = pl.action then
        if RokuMfgIsInvalid(pl.data) then
            component = ""
        else if RokuMfgIsString(pl.data) then
            component = pl.data
        else
            return RokuMfgBadDataStatus("expected data as string")
        end if

        if ("" = component or "all" = component) then
            data = state.tuner.get("ChannelList")
            if not RokuMfgIsInvalid(data) then
                return RokuMfgSuccessStatus(data)
            end if
            needStateRefresh = true
        else if "current" = component then
            data = state.tuner.get("CurrentChannel")
            if not RokuMfgIsInvalid(data) then
                return RokuMfgSuccessStatus(data)
            end if
            needStateRefresh = true
        end if
    else if "set" = pl.action then
        if not RokuMfgIsAA(pl.data) then
            return RokuMfgBadDataStatus("expected data as channel info associative array")
        end if

        tuneTo = RokuMfgStrCast(pl.data.channel)
        standard = RokuMfgApi_Tuner_GetStandard()

        if "digital" = pl.data.signal and standard <> "dvb" then
            tuneTo = tuneTo + "." + RokuMfgStrCast(pl.data.subchannel)
        end if

        print "Tune to " + tuneTo
        ret = RokuMfgCallExternal(mfg, "tuner", {
            header_: pl.header_,
            action: "set",
            data: tuneTo
        })

        if RokuMfgCheckResponse(ret) then
            state.tuner.set("CurrentChannel", pl.data)
        end if

        return ret
    else if "next" = pl.action or "prev" = pl.action then
        allChannels = RokuMfg().call("tuner", {action: "get"})
        current = RokuMfg().call("tuner", {action: "get", data: "current"})
        if not RokuMfgCheckResponse(allChannels) or not RokuMfgCheckResponse(current) then
            return RokuMfgGenericStatus("unable to inc/dec channel")
        end if

        change = 1
        if "prev" = pl.action then
            change = -1
        end if

        for each c in  allChannels.data
            print c
        end for
        print current.data.index, change

        newIdx = RokuMfgModulo(current.data.index + change, allChannels.data.count())
        print allChannels.data[newIdx]
        ret = RokuMfg().call("tuner", {
            action: "set",
            data: allChannels.data[newIdx]
        })

        return ret
    end if

    ret = RokuMfgCallExternal(mfg, "tuner", pl)

    if RokuMfgCheckResponse(ret) then
        if needStateRefresh then
            state.tuner.refresh()
        end if
    end if

    if restoreTuner then
        RokuMfg().call("inputsource", {action: "set", data: "tuner"})
    end if

    ' If we get here when asking for the current channel, it means that we
    ' had to initialize the cache and the response we got doesn't contain
    ' the index. Update the repsonse data for completeness.
    if "get" = pl.action and "current" = pl.data then
        data = state.tuner.get("CurrentChannel")

        ' CurrentChannel should never be invalid at this point, but if it is,
        ' don't override the current data.
        if not RokuMfgIsInvalid(data) then
            ret.data = data
        end if
    end if

    return ret
end function

function RokuMfgApi_Tuner_GetStandard()
    ret = RokuMfg().call("tuner", {
        action: "get",
        data: "standard"
    })

    if not RokuMfgCheckResponse(ret) or not RokuMfgIsString(ret.data) then
        ' Assume ATSC if we don't get a reasonable response
        standard = "atsc"
    else
        standard = ret.data
    end if
    return standard
end function
