'*********************************************************************
'** (c) 2018 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************
' Roku_MFG_API_Network.brs
' Implementation for network-specific APIs
Library "state/Roku_MFG_State.brs"

function RokuMfgApi_Network(mfg as Object, pl as Object) as Object
    iface_lut = {
        wired: "ethmac",
        wifi: "wifimac",
        bluetooth: "btmac",
        p2p: "p2pmac"
    }

    if "mac" = pl.component and "set" = pl.action then
        if RokuMfgIsChipType("Sigma UXL-38") then
            return RokuMfgGenericStatus("support deprecated")
        else if not RokuMfgIsString(pl.iface) then
            return RokuMfgBadDataStatus("expected iface as string")
        else if not RokuMfgIsString(pl.data) then
            return RokuMfgBadDataStatus("expected data as mac address string")
        else if RokuMfgIsInvalid(iface_lut[pl.iface]) then
            return RokuMfgGenericStatus("unknown interface name")
        end if

        ' Check each octet to determine if it's valid. Also extend if necessary.
        tok = pl.data.tokenize(":")
        if 6 <> tok.count() then
            return RokuMfgBadDataStatus("invalid mac format: expected 6 octets")
        end if

        regex = createObject("roRegex", "^[a-f0-9]{1,2}$", "i")
        for i = 0 to 5
            if not regex.isMatch(tok[i]) then
                return RokuMfgBadDataStatus("octet invalid: " + tok[i])
            end if

            ' Prepend a 0 if necessary
            if len(tok[i]) < 2 then
                tok[i] = "0" + tok[i]
            end if
        end for

        if 1 = (val(tok[0], 16) MOD 2) then
            return RokuMfgBadDataStatus("multicast addresses not allowed")
        end if

        data = {}
        data[iface_lut[pl.iface]] = pl.data
        pc = RokuMfgCallExternal(mfg, "pc", {
            header_: pl.header_,
            action: "set",
            data: data
        })

        if not RokuMfgCheckResponse(pc) then
            return RokuMfgGenericStatus("failed to set MAC address")
        end if

        return RokuMfgSuccessStatus({}, "requires reboot to take effect")
    end if

    ret = RokuMfgCallExternal(mfg, "network", pl)

    if RokuMfgCheckResponse(ret) then
        if "mac" = pl.component and "get" = pl.action then
            if RokuMfgIsInvalid(iface_lut[pl.data]) then
                return RokuMfgBadDataStatus("unknown interface")
            end if

            pc = RokuMfgCallExternal(mfg, "pc", {
                header_: pl.header_,
                action: "get",
                data: iface_lut[pl.data]
            })

            ret.data.pcaddress = pc.data[iface_lut[pl.data]]
            if "p2p" = pl.data then
                ret.data.address = pc.data[iface_lut[pl.data]]
            end if
        end if
    end if

    if "wifi" = pl.component and "rssi" = pl.action then
        connected = RokuMfgCheckResponse(ret)

        if not connected then
            if "TV" <> pl.header_.device_type then
                retries = 0
                while not connected and retries < 2
                    reconn = RokuMfgCallExternal(mfg, "network", {
                        component: "wifi",
                        action: "reconnect",
                        block: true
                    })

                    if RokuMfgCheckResponse(reconn) then
                        for i=0 to 20 step 1
                            ret = RokuMfgCallExternal(mfg, "network", pl)
                            connected = RokuMfgCheckResponse(ret)
                            if connected then
                                exit for
                            end if

                            sleep(500)
                        end for
                    end if

                    retries = retries + 1
                end while
            else
                reconn = RokuMfgCallExternal(mfg, "network", {
                    component: "wifi",
                    action: "reconnect",
                    block: true
                })

                ' Sleep for 2s to allow things to settle down
                sleep(2000)
                if RokuMfgCheckResponse(reconn) then
                    sleep(500)
                    ret = RokuMfgCallExternal(mfg, "network", pl)
                    connected = RokuMfgCheckResponse(ret)
                end if
            end if
        end if

        if connected then
            strong = RokuMfgIsWifiRssiStrong(ret.data.rssi1, ret.data.rssi2)
        else
            strong = false
        end if

        if connected and false = strong then
            return RokuMfgGenericStatus("weak signal", ret.data)
        end if
    end if

    return ret
end function
