'*********************************************************************
'** (c) 2018-2019 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************
' Roku_MFG_API_Common.brs
' Implementation of APIs common to all platforms
#const debug = false

function RokuMfgApi_ClearSettingsData(mfg as Object, pl as Object) as Object
    RokuMfgCallExternal(mfg, "tuner", {
        header_: pl.header_,
        action: "reset"
    })

    if not RokuMfgIsInvalid(pl.data) and true = pl.data["cleanall"] then
        ' Full clean is just resetting all of nvram
        return RokuMfgApi_NvmReset(mfg, {header_: pl.header_})
    end if

    if "TV" = pl.header_.device_type then
' tfranklin: need to have PQ reset here
' 1. Restore default level-5 pq settings across all inputs.
'       PQ API call to reset single layer for all inputs, all presets
' 2. Reset important pqdb settings for the "Network" input.
'                 auto const tvm = PlatformAV::GetInstance()->getTVManager();
'                 if (tvm) {
'                     auto const input = tvm->getInput();
'                     // FIXME -- Sigma and MStar differ on setInput logic (they shouldn't).
'                     tvm->setInput("network");
' #ifdef SIGMA
'                     Singleton<PQDriver>().refresh();
' #endif
'                     tvm->setInput(input);
' #ifdef SIGMA
'                     Singleton<PQDriver>().refresh();
' #endif
' 3. Reset SSC_* values.
'             std::shared_ptr<PQConfiguration> cfg = pqdb.get(PQCONFIG_BASE);
'             cfg->unset("DISP_SSC_Depth");
'             cfg->unset("DISP_SSC_Frequency");
'             cfg->unset("DISP_SSC_Strength");
    end if

' tfranklin: need AQ API access
' 4. Reset audio settings
'             AppConfig::factoryResetVolume();
'             AppConfig::factoryResetMute();
'             AppConfig::factoryResetSpeakersEnabled();
'             m_audioDev.setSoundMode("Normal");

    ' 5. Reset some non-pq/aq configuration settings.
    ret = RokuMfg().call("syscfg", {
        action: "set",
        data: {"RokuTv_Mfg_Bsc_Mode": 0, "RokuTv_Aging_Mode": 0}
    })

    return RokuMfgSuccessStatus()
end function

function RokuMfgApi_FactoryShopInit(mfg as Object, pl as Object) as Object
    ' MALONE-2785: Make sure the WiFi MAC address is valid
    ret = RokuMfg().call("network", {
        component: "mac",
        action: "get",
        data: "wifi"
    })

    if not RokuMfgCheckResponse(ret) or "00:00:00:00:00:00" = ret.data.address then
        return RokuMfgGenericStatus("WiFi MAC address is invalid")
    end if

    ' test for turnkey project or not 
    ' retrieve all custom_pkg flags then test manually
    ret = RokuMfg().call("custominfo", {
        action: "get",
        data: "flags"
    })

    if not RokuMfgCheckResponse(ret) or not RokuMfgIsAA(ret.data["flags"]) then
        return RokuMfgGenericStatus("failure in obtaining custominfo flags")
    end if 

    if RokuMfgBoolCast(ret.data.flags["MODEL_SUPPORTS_TURNKEY"]) then
        ' perform Turnkey related extra checks

        ' obtain certain custom_pkg for later comparisons
        ret_custominfo = RokuMfg().call("custominfo", {
            action: "get",
            data: [
                "info/manufacturer",
                "info/model",
                "info/display_name"
            ]
        })

        if not RokuMfgCheckResponse(ret_custominfo) or not RokuMfgIsAA(ret_custominfo.data) then
            return RokuMfgGenericStatus("custom info failure")
        end if

        ' check PC data
        keys = [
                "manufacturer",
                "DEFAULT_LOCALE",
                "brand",
                "odmmodel",
                "displayname",
                "rokumodel"
            ]
        ret = RokuMfg().call("pc", {
            action: "get", 
            data: keys
        }) ' partial failure allowed for this call

        if not RokuMfgIsAA(ret.data) then
            return RokuMfgGenericStatus("PC API failure")
        end if 

        for each key in keys
            if "displayname" = key then 
                ' pass
            else if "odmmodel" = key then
                ' pass
            else if not RokuMfgIsString(ret.data[key]) or 0 = ret.data[key].len() then
                return RokuMfgGenericStatus(key + " data failure")
            end if
        end for

        if 5 <> ret.data["DEFAULT_LOCALE"].len() then
            return RokuMfgGenericStatus("default locale format bad")
        else if ret.data["manufacturer"] <> ret_custominfo.data["info/manufacturer"] then
            return RokuMfgGenericStatus("manufacturer data inconsistent")
        end if

        ' check the user-facing co-branded model number
        brand_model = ""  
        if RokuMfgIsString(ret.data["displayname"]) and 0 < ret.data["displayname"].len() then
            brand_model = ret.data["displayname"]
        else if RokuMfgIsString(ret.data["odmmodel"]) and 0 < ret.data["odmmodel"].len() then
            brand_model = ret.data["odmmodel"]
        end if 

        if 0 = brand_model.len() then
            return RokuMfgGenericStatus("brand model incorrect: empty")
        else if ret_custominfo.data["info/display_name"] = brand_model then
            return RokuMfgGenericStatus("brand model incorrect: same as roku default")
        else if ret.data["rokumodel"] = brand_model then
            return RokuMfgGenericStatus("brand model incorrect: same as roku model")
        else if ret_custominfo.data["info/model"] = brand_model then
            return RokuMfgGenericStatus("brand model incorrect: same as CP fallback")
        end if


        ' check for support info

        default_region = Mid(ret.data["DEFAULT_LOCALE"], 4, 2)
        cs_types = ["supporturl", "supportphone"] 
        cs_regions = ["", default_region]
        cs_keys = []

        for each cs_type in ["supporturl", "supportphone"] 
            for each cs_region in [default_region, ""]
                cs_keys.push(cs_type + cs_region)
            end for
        end for

        ret = RokuMfg().call("pc", {
            action: "get", 
            data: cs_keys
        }) ' partial failure allowed for this call

        if not RokuMfgIsAA(ret.data) then
            return RokuMfgGenericStatus("PC API failure")
        end if 

        for each cs_type in cs_types
            cs = ""

            for each cs_region in cs_regions
                key = cs_type + cs_region
                if RokuMfgIsString(ret.data[key]) and 0 < ret.data[key].len() then
                    cs = ret.data[key]
                end if 
            end for

            if 0 = cs.len() then
                return RokuMfgGenericStatus(cs_type + " not set")
            end if
        end for
    end if

    msu = RokuMfg().sysInfo().ismsu
    if "TV" = pl.header_.device_type then
        if not msu and pl.header_.is_manufacturing then
            ret = RokuMfgCallExternal(mfg, "partitions", {
                header_: pl.header_
                action: "gettype",
                data: "update"
            })
            if RokuMfgCheckResponse(ret) or "APP" <> ret.data then
                if "APP" <> ret.data then
                    return RokuMfgGenericStatus("APP image not in Update partition")
                end if
            end if
        else if msu then
            ' Make sure we have a good rescue image
            key = "rescuemfgverify"
            ret = RokuMfg().call("ubapp", {
                action: "get",
                data: key
            })

            if not RokuMfgCheckResponse(ret) or "good" <> ret.data[key] then
                return RokuMfgGenericStatus("failed to confirm rescue image")
            end if
        end if

        ret = RokuMfgCallExternal(mfg, "pc", {
            header_: pl.header_,
            action: "get",
            data: ["rokumodel", "panel", "GammaTable_Number"]
        })

        ' Check for AA here becuase failing to get a key
        ' (e.g., GammaTable_Number) results in an API failure, but the data for
        ' other keys are still valid. Any failed keys should be returned as
        ' empty strings, so we can still check against them.
        ' tfranklin: We should revisit how PC returns a failure, but I believe
        ' there's scenarios that rely on this functionality.
        if RokuMfgIsAA(ret.data) then
            if not RokuMfgIsSigmaPlatform() then
                if "" = ret.data.rokumodel or "Default" = ret.data.rokumodel then
                    return RokuMfgGenericStatus("rokumodel is not set")
                end if
            end if

            if "" = ret.data.panel then
                return RokuMfgGenericStatus("panel is not set")
            end if

            tables = val(ret.data.GammaTable_Number, 10)
            if 0 < tables and 3 > tables then
                return RokuMfgGenericStatus("Only " + RokuMfgStrCast(tables) + " gamma tables found; should be at least 3")
            else if 3 <= tables then
' tfranklin: Need to add gamma table handling here
'               FIXME:
'               Still no API for AV call
'               if (PlatformAV::GetInstance()->call("apply_odm_gamma_table")!="OK")
'               {
'                   return Error("invalid gamma tables");
'               }
                print "would check for apply error here"
            end if
        end if
    end if

'       FIXME
'       XXX TBD! We will deprecate the "aging" configuration
'           In favor of "bsc". However, so as not to
'           break anything that might be in-flux, we'll support
'           "aging" for a short duration.
'       tfranklin: do we want to remove it now?
    ret = RokuMfg().call("syscfg", {
        action: "set",
        data: {"RokuTv_Mfg_Bsc_Mode": 0, "RokuTv_Aging_Mode": 0}
    })

    ' If this returns false, then the key isn't available so
    ' there's no action to take.
    key = "shop_init/delete_channels"
    ret = RokuMfg().call("custominfo", {
        action: "get",
        data: key
    })

    if RokuMfgCheckResponse(ret) and "true" = ret.data[key] then
        RokuMfgCallExternal(mfg, "tuner", {
            header_: pl.header_,
            action: "reset"
        })
    end if

    if pl.header_.is_manufacturing then
        if msu then
            ret = RokuMfg().call("ubapp", {
                action: "set",
                data: {mfgmode: ""}
            })
            if not RokuMfgCheckResponse(ret) then
                return RokuMfgGenericStatus("failed to clear MFG mode")
            end if
        else
            ret = RokuMfgCallExternal(mfg, "partitions", {action: "setalternate"})
            if not RokuMfgCheckResponse(ret) then
                return RokuMfgResponse(ret.header_.status, ret.header_.description)
            end if
        end if
    end if

    file_flags = []

    if "TV" = pl.header_.device_type then
        file_flags.push("/nvram/factoryresetflag3")
    end if

    file_flags.push("/nvram/factoryresetflag2")
    if pl.header_.is_manufacturing then
        key = "factory_reset_fast_boot"
        ret = RokuMfg().call("custominfo", {
            action: "get",
            data: key
        })

        if RokuMfgCheckResponse(ret) and ret.data[key] then
            file_flags.push("/nvram/factoryresetfastboot")
        end if

        if not msu and "TV" <> pl.header_.device_type then
            RokuMfgCallExternal(mfg, "partitions", {
                header_: pl.header_,
                action: "erase",
                data: "active"
            })
        end if
    end if

    if "TV" <> pl.header_.device_type then
        prolonged_wait = true
    else
        prolonged_wait = false
    end if

    key = "shop_init/prolonged_wait"
    ret = RokuMfg().call("custominfo", {
        action: "get",
        data: key
    })

    if RokuMfgCheckResponse(ret) and true = ret.data[key] then
        prolonged_wait = true
    end if

#if debug
' tfranklin: REMOVE THIS DEBUGGING ''''''''''''''''''''''''''''''''''''''''''''''''''''
    print "prolonged_wait: " + RokuMfgStrCast(prolonged_wait)
    return RokuMfgSuccessStatus({}, "shopinit will occur when AC power is cycled")
'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
#end if

    pl_reset = {header_: pl.header}
    if not prolonged_wait then
        pl_reset.action = "defaults"
    end if

    RokuMfgCallExternal(mfg, "factoryreset", pl_reset)
    RokuMfgApi_Filesys(mfg, {
        header_: pl.header_,
        action: "touch",
        data: file_flags
    })

    if prolonged_wait then
        print "Waiting for FactoryResetController to reboot system"
        while true
            sleep(20)
        end while
    end if

    return RokuMfgSuccessStatus({}, "shopinit will occur when AC power is cycled")
end function

function RokuMfgApi_Filesys(mfg as Object, pl as Object) as Object
    ret = RokuMfgGenericStatus()
    if RokuMfgIsString(pl.action) then
        if "touch" = pl.action or "read" = pl.action then
            if RokuMfgIsArray(pl.data) then
                arr = pl.data
            else
                arr = [pl.data]
            end if

            allowed = ["/nvram/", "/tmp/"]
            for each path in arr
                if not RokuMfgIsString(path) then
                    return RokuMfgBadDataStatus("expected data as string or array of strings")
                end if

                match = false
                for each prefix in allowed
                    if 0 = path.instr(prefix) then
                        match = true
                        exit for
                    end if
                end for

                if not match or -1 <> path.instr("/..") then
                    return RokuMfgBadDataStatus("Unsupported file path: " + path)
                end if
            end for
        end if

        ret = RokuMfgCallExternal(mfg, "filesys", pl)

        if "get" = pl.action and "usbdevices" = pl.data and RokuMfgCheckResponse(ret) then
            ' Sample output from `ls /sys/bus/usb/devices` includes the following:
            ' 1-0:1.0, 1-1, 1-1:1.0, 2-0:1.0, usb1, usb2
            ' For the purpose of this API, we ignore "n-m:x.y".
            ' Also, usb<n> is always present even if the port isn't populated.
            ' We want to return a list of just "n-m" devices.
            data = []
            for each filename in ret.data
                if -1 = filename.instr(":") and -1 = filename.instr("usb") then
                    data.push(filename)
                end if
            end for

            ret.data = data
        end if
    else
        return RokuMfgBadActionStatus("expected action as string")
    end if

    return ret
end function

function RokuMfgApi_Hdmi(mfg as Object, pl as Object) as Object
    setters = {
        "hdcp22key": "",
        "hdcp14key": ""
    }

    if "set" = pl.action and not RokuMfgIsInvalid(setters[pl.component]) then
        if not RokuMfgHasInterface(pl.key, "ifByteArray") then
            return RokuMfgBadDataStatus("expected data key as roByteArray")
        end if

        ' Convert byte array to base64 string
        pl.key = pl.key.toBase64String()
    end if

    ret = RokuMfgCallExternal(mfg, "hdmi", pl)

    if RokuMfgCheckResponse(ret) then
        getters = {
            "hdcp14ksv":    "",
            "hdcp22rxid":   "",
            "hdcp22key":    ""
        }

        if "get" = pl.action and not RokuMfgIsInvalid(getters[pl.component]) then
            if "ok" <> ret.data.result then
                return RokuMfgGenericStatus(RokuMfgStrCast(ret.data.error))
            end if

            ' data needs converted from a base64 string
            ba = createObject("roByteArray")
            ba.fromBase64String(ret.data.data)

            if "hdcp22key" = pl.component and 1040 > ba.count() then
                return RokuMfgGenericStatus("invalid size")
            end if

            ret.data = ba
        end if
    end if

    return ret
end function

function RokuMfgApi_List(mfg as Object, pl as Object) as Object
    data = []
    libApis = getGlobalAA().mfg_lib_apis
    for each api in libApis
        data.push(api)
    end for

    ret = RokuMfgCallExternal(mfg, "list", pl)
    if RokuMfgCheckResponse(ret) then
        for each api in ret.data
            if invalid = libApis[api] then
                data.push(api)
            end if
        end for
    end if

    data.sort()
    return RokuMfgSuccessStatus(data)
end function

function RokuMfgApi_NvmReset(mfg as Object, pl as Object) as Object
'       FIXME
'       XXX TBD! We will deprecate the "aging" configuration
'           In favor of "bsc". However, so as not to
'           break anything that might be in-flux, we'll support
'           "aging" for a short duration.
'       tfranklin: do we want to remove it now?
    ret = RokuMfg().call("syscfg", {
        action: "set",
        data: {"RokuTv_Mfg_Bsc_Mode": 0, "RokuTv_Aging_Mode": 0}
    })

    RokuMfgCallExternal(mfg, "tuner", {
        header_: pl.header_,
        action: "reset"
    })

    key = "shop_init/NVM_reset_clear_nvram"
    ret = RokuMfg().call("custominfo", {
        action: "get",
        data: key
    })

    take_action = false
    if RokuMfgCheckResponse(ret) then
        if ("true" = ret.data[key]) then
            take_action = true
        end if
    end if

    info = getGlobalAA().mfg_constants.sysInfo
    file_flags = []

    if "TV" = pl.header_.device_type then
        if true = take_action or true = pl.header_.is_manufacturing then
            file_flags.push("/nvram/factoryresetflag3")
        end if
    end if

    file_flags.push("/nvram/factoryresetflag2")
    RokuMfgApi_Filesys(mfg, {
        header_: pl.header_,
        action: "touch",
        data: file_flags
    })

    if true = take_action and RokuMfgIsSigmaPlatform() then
' tfranklin: Do we still want to use autoboot flag? If so, it should be moved entirely into BRS
' FIXME: enable autoboot here
'             //Since M mode will be enabled here, enable the autoboot override.
'             auto&& pc = Singleton<PersistentConfiguration>();
'             pc.set("autobootoverride", PC_AUTOBOOT_ON);
'             pc.commit();
    end if

    return RokuMfgSuccessStatus()
end function

function RokuMfgApi_PersistentConfig(mfg as Object, pl as Object) as Object
    ret = RokuMfgGenericStatus()

    ignored_keys = {
        ' Key           Reason
        "mfgdatecheck": "Roku internal checksum"
    }

    if not RokuMfgIsInvalid(pl.action) and RokuMfgIsString(pl.action) then
        if "set" = pl.action then
            if RokuMfgIsAA(pl.data) then
                for each key in pl.data
                    if not RokuMfgIsString(pl.data[key]) then
                        return RokuMfgBadDataStatus("expected values as string")
                    end if
                end for
            else
                return RokuMfgBadDataStatus("expected data as associative array")
            end if

            get_data = []
            for each el in pl.data
                if RokuMfgIsInvalid(ignored_keys[el]) then
                    if "manufacturer" = lcase(el) and 0 < pl.data[el].len() then
                        ' always use custom_pkg value when setting manufacturer field
                        key = "info/manufacturer"
                        mfr_cp = RokuMfg().call("custominfo", {
                            action: "get",
                            data: key
                        })

                        if RokuMfgCheckResponse(mfr_cp) and RokuMfgIsString(mfr_cp.data[key]) then
                            if mfr_cp.data[key] <> pl.data[el] then
                                print "input manufacturer invalid; will use: " + mfr_cp.data[key]
                                pl.data[el] = mfr_cp.data[key]
                            end if
                        else
                            print "Note: failed to determine manufacturer; will do nothing"
                            pl.data.delete(el)
                        end if
                    end if

                    get_data.push(el)
                else
                    print "Note: ignore key " + el + ": " + ignored_keys[el]
                    pl.data.delete(el)
                end if
            end for

            get_pl = {
                header_: pl.header_,
                action: "get",
                data: get_data
            }

            if invalid <> pl.type then
                get_pl.type = pl.type
            end if
            get = RokuMfgApi_PersistentConfig(mfg, get_pl)

            if RokuMfgCheckResponse(get) then
                for each el in pl.data
                    ' Only update the values that changed
                    if get.data[el] = pl.data[el] then
                        print "Note: not updating " + el + ": identical value"
                        pl.data.delete(el)
                    end if
                end for
            end if
        else if "get" = pl.action then
            if RokuMfgIsString(pl.data) then
                pl.data = [pl.data]
            else if not RokuMfgIsArray(pl.data) then
                return RokuMfgBadDataStatus("expected data as string or array")
            end if
        end if

        ret = RokuMfgCallExternal(mfg, "pc", pl)
        if "ubapp" <> pl.type and RokuMfgCheckResponse(ret) then
            if "reset" = pl.action then
                RokuMfgCallExternal(mfg, "systempower", {
                    header_: pl.header_,
                    action: "reboot"
                })
            else if "get" = pl.action then
                for each key in ret.data
                    if not RokuMfgIsInvalid(ignored_keys[key]) then
                        print "Note: ignore key " + key + ": " + ignored_keys[key]
                        ret.data.delete(key)
                    else if "" = ret.data[key] then
' tfranklin: do we want to return a failure here? Some legacy implementations expect it (e.g., extended custom config)
                        print "Note: failed to get key from PC: " + key
                    end if
                end for
            else if "set" = pl.action then
                if not RokuMfgIsInvalid(pl.data.panel) then
' tfranklin: need to whack PQ here
                    print "Note: clear PQ to force reload on reboot"
                else if not RokuMfgIsInvalid(pl.data.mfgdateyear) or not RokuMfgIsInvalid(pl.data.mfgdatemonth) or not RokuMfgIsInvalid(pl.data.mfgdateweek) then
                    date = RokuMfgCallExternal(mfg, "pc", {
                        header_: pl.header_,
                        action: "get",
                        data: ["mfgdateyear", "mfgdatemonth", "mfgdateweek"]
                    })

                    if RokuMfgCheckResponse(date) then
                        year = val(date.data.mfgdateyear, 10)
                        month = val(date.data.mfgdatemonth, 10)
                        week = val(date.data.mfgdateweek, 10)

                        expected_check = RokuMfgXor(((year MOD 100) + (month MOD 13) + (week MOD 54)), &hff) and &hff
                        check = RokuMfgXor((year + month + week), &hff)

                        if (check = expected_check) then
                            RokuMfgCallExternal(mfg, "pc", {
                                header_: pl.header_,
                                action: "set",
                                data: {"mfgdatecheck": check}
                            })
                        end if
                    end if
                end if
            end if
        end if
    else
        return RokuMfgBadActionStatus("expected action as string")
    end if

    return ret
end function

function RokuMfgApi_Proc(mfg as Object, pl as Object) as Object
    if not RokuMfgIsString(pl.action) then
        return RokuMfgBadActionStatus("expected action as string")
    else if "start" = pl.action then
        if not RokuMfgHasInterface(pl.uart, "ifUartConsole") then
            return RokuMfgBadDataStatus("expected uart as roUartConsole")
        end if

        ' Save a copy of the UART reference to receive responses. Delete it
        ' from the payload so we don't have to rebuild to format as valid JSON.
        RokuMfgState().set("ProcUart", pl.uart)
        pl.delete("uart")
    end if

    return RokuMfgCallExternal(mfg, "proc", pl)
end function

function RokuMfgApi_Temperature(mfg as Object, pl as Object) as Object
    ' Sometimes the sensors don't respond correctly. Retry a few times
    for i=0 to 5
        ret = RokuMfgCallExternal(mfg, "temperature", {header_: pl.header_})
        if RokuMfgCheckResponse(ret) then
            return ret
        end if
    end for

    ' If we get here, the API returned an error each time.
    ' Just return the last error we got
    return ret
end function

function RokuMfgApi_UbApp(mfg as Object, pl as Object) as Object
    copy = RokuMfgDeepCopy(pl)
    copy.type = "ubapp"
    return RokuMfg().call("pc", copy)
end function
