'*********************************************************************
'** (c) 2018-2019 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************
' Roku_MFG_API.brs
' List of APIs supported/maintained in the Roku MFG BRS Library
#const debug = false

Library "apis/Roku_MFG_API_Barcode.brs"
Library "apis/Roku_MFG_API_Common.brs"
Library "apis/Roku_MFG_API_Network.brs"
Library "apis/Roku_MFG_API_PQ.brs"
Library "apis/Roku_MFG_API_TV.brs"
Library "apis/Roku_MFG_API_Tuner.brs"

function RokuMfgLibApis() as Object
    apis = {
        barcode: RokuMfgApi_Barcode,
        clearsettings: RokuMfgApi_ClearSettingsData,
        filesys: RokuMfgApi_Filesys,
        gamma: RokuMfgApi_Gamma,
        hdmi: RokuMfgApi_Hdmi,
        inputsource: RokuMfgApi_InputSource,
        list: RokuMfgApi_List,
        network: RokuMfgApi_Network,
        pc: RokuMfgApi_PersistentConfig,
        proc: RokuMfgApi_Proc,
        projects: RokuMfgApi_Projects,
        resetnvm: RokuMfgApi_NvmReset,
        shopinit: RokuMfgApi_FactoryShopInit,
        temperature: RokuMfgApi_Temperature,
        tuner: RokuMfgApi_Tuner,
        ubapp: RokuMfgApi_UbApp
    }

    return apis
end function

function RokuMfgLibraryInfo() as Object
    field = "libinfo"
    constants = getGlobalAA().mfg_constants

    if not constants.hasField(field) then
        print "Init Library Info"
        name = "Roku MFG library"
        version = "Unknown"
        mfst = readAsciiFile("libpkg:/manifest")
        if mfst <> "" then
            vers = {major_version: "0", minor_version: "0", build_version: "0"}
            for each line in mfst.tokenize(chr(10)+chr(13)):
                keyval = line.tokenize("=")
                if keyval.count() > 1:
                    vers[keyval[0].trim()] = keyval[1].trim()
                end if
            end for

            major_str = RokuMfgStrCast(val(vers.major_version, 10))
            minor_str = RokuMfgStrCast(val(vers.minor_version, 10))
            build_str = RokuMfgStrCast(val(vers.build_version, 10))
            version = major_str + "." + minor_str + "." + right("0000" + build_str, 4)
        end if

        f = {}
        f[field] = {
            name: name,
            version: version
        }
        constants.addFields(f)
    end if

    return constants.getField(field)
end function

function RokuMfgSystemInfo() as Object
    field = "sysinfo"
    constants = getGlobalAA().mfg_constants

    if not constants.hasField(field) then
        print "Init System Info"
        f = {}
        f[field] = {
            devicetype: "Unknown",
            ismanufacturing: false,
            issecure: true
        }
        constants.addFields(f)

        ret = m.call("systeminfo")
        if m.checkResponse(ret) then
            constants.setField(field, ret.data)
        end if
    end if

    return constants.getField(field)
end function

function RokuMfgPlatformInfo() as Object
    field = "platforminfo"
    constants = getGlobalAA().mfg_constants

    if not constants.hasField(field) then
        print "Init Platform Info"
        f = {}

        ret = m.call("platforminfo")
        if m.checkResponse(ret) then
            f[field] = ret.data
        else
            f[field] = {}
        end if

        constants.addFields(f)
    end if

    return constants.getField(field)
end function

function RokuMfgPqInfo()
    field = "pqparameters"
    constants = getGlobalAA().mfg_constants

    if not constants.hasField(field) then
        print "Init PQ Data"
        f = {}
        f[field] = {}
        json_predates_schema = Invalid
        if m.util.isTvPlatform() then
            key = "pq/schema/settings"
            ret = m.call("custominfo", {
                action: "get",
                data: key
            })

            if m.checkResponse(ret) then
                json_predates_schema = false
            else
                key = "pq/parameters"
                ret = m.call("custominfo", {
                    action: "get",
                    data: key
                })

                if m.checkResponse(ret) then
                    json_predates_schema = true
                end if
            end if

            if json_predates_schema <> Invalid then
                f[field] = ret.data[key]

                ' Also populate useful data
                params = ret.data[key]
                pqdata = {}
                ctwb = []
                for each entry in params
                    if params[entry].uiid = "color_temp" then
                        pqdata.color_temp = params[entry].selector
                    else if params[entry].uiid = "aspect" then
                        pqdata.aspect = params[entry].selector
                    else if params[entry].uiid = "dynamic_contrast" then
                        pqdata.dynamic_contrast = params[entry].selector
                    else if left(params[entry].uiid, 6) = "ct_wb_" then
                        if json_predates_schema then
                            pqname = params[entry].pqname
                        else
                            pqname = entry
                        end if
                        ctwb.push(pqname)
                    end if
                end for

                pqdata.ct_wb = ctwb
                RokuMfgState().set("PqKeyValues", pqdata)
            end if
        end if

        constants.addFields(f)
    end if

    return constants.getField(field)
end function


function RokuMfgAqInfo()
    field = "aqparameters"
    constants = getGlobalAA().mfg_constants

    if not constants.hasField(field) then
        print "Init AQ Data"
        f = {}
        f[field] = {}
        if m.util.isTvPlatform() then
            key = "aq/parameters"
            ret = m.call("custominfo", {
                action: "get",
                data: key
            })

            if m.checkResponse(ret) then
                f[field] = ret.data[key]
            end if
        end if

        constants.addFields(f)
    end if

    return constants.getField(field)
end function

function RokuMfgCallApi(command="" as String, payload={} as Object) as Object
#if debug
    timer = createObject("roTimespan")
#end if
    ' Use the ContentNode directly as opposed to calling m.sysInfo()
    ' m.sysInfo() takes significantly longer because it has to deep-copy the data
    constants = getGlobalAA().mfg_constants
    payload["header_"] = {
        is_secure: constants.sysinfo.issecure,
        is_manufacturing: constants.sysinfo.ismanufacturing,
        device_type: constants.sysinfo.devicetype
    }

    if RokuMfgIsString(payload.action) then
        payload.action = lcase(payload.action)
    end if

    libApis = getGlobalAA().mfg_lib_apis
    cmd = lcase(command)
    if invalid <> libApis[command] then
        ' Library handles the command
        ret = libApis[command](m.impl, payload)
    else
        ret = RokuMfgCallExternal(m.impl, cmd, payload)
    end if

    ret.header_.command = command

#if debug
    print command + " took " + stri(timer.totalMilliseconds()) + "ms"
#end if

    return ret
end function
