'*********************************************************************
'** (c) 2018-2019 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************
' Roku_MFG_State_Tuner.brs
' Helpers for maintaining tuner state

'
' Data state object specific to Roku MFG Library tuner state. Intended to be
' used only by Roku MFG Library internal functions (i.e., not exposed to
' individual partner apps). Uses a separate data store module for separation.
'
' Provides the following public member functions:
' @see .get RokuMfgState_TunerGet()
' @see .set RokuMfgState_TunerSet()
' @see .lock RokuMfgState_TunerLock()
' @see .unlock RokuMfgState_TunerUnlock()
' @see .refresh RokuMfgState_TunerCacheRefresh()
' @see .invalidate RokuMfgState_TunerInvalidate()
'
' @return store a new roAssociativeArray wrapping tuner store functionality
function RokuMfgState_TunerModuleInit()
    o = {
        data: RokuMfgState_StoreModuleInit(),

        get: RokuMfgState_TunerGet,
        set: RokuMfgState_TunerSet,
        lock: RokuMfgState_TunerLock,
        unlock: RokuMfgState_TunerUnlock,
        refresh: RokuMfgState_TunerCacheRefresh,
        invalidate: RokuMfgState_TunerInvalidate
    }

    o.unlock()
    return o
end function

'
' Get key from the tuner data store.
'
' @param key The key to access in the data store
' @return obj Dynamic value obtained from store. Invalid if key is not set
function RokuMfgState_TunerGet(key as String) as Dynamic
    return m.data.get(key)
end function

'
' Set key in the tuner data store. Adds new key or replaces existing key.
'
' @param key The key to set in the data store
' @param value Dynamic value to store
' @return true is set is successful, false otherwise
function RokuMfgState_TunerSet(key as String, value as Dynamic) as Boolean
    return m.data.set(key, value)
end function

'
' Attempt to obtain lock to prevent recursive calls to modify tuner cache
'
' @return true if lock was obtained, false otherwise
function RokuMfgState_TunerLock() as Boolean
    if not m.get("CacheLock") then
        return m.set("CacheLock", true)
    end if

    return false
end function

'
' Release lock on tuner cache. Must be called manually after calling .lock()
sub RokuMfgState_TunerUnlock()
    m.set("CacheLock", false)
end sub

'
' Refresh the tuner cache state. Attempts to get updated local copy of channel
' list from RokuOS. If successful, also updates the current channel from RokuOS.
' If getting the current channel fails (e.g., refreshing cache due to channel
' DB reload), tune to the first channel in the list.
sub RokuMfgState_TunerCacheRefresh()
    if not m.lock() then
        return
    end if

    allChannels = RokuMfg().call("tuner", {action: "get"})
    if not RokuMfgCheckResponse(allChannels) then
        m.unlock()
        return
    end if

    for idx = 0 to allChannels.data.count() - 1
        allChannels.data[idx].index = idx
    end for

    m.data.set("ChannelList", allChannels.data)
    ret = RokuMfg().call("tuner", {
        action: "get",
        data: "current"
    })

    match = false
    if RokuMfgCheckResponse(ret) then
        for each channel in allChannels.data
            match = true
            for each key in ret.data
                if channel[key] <> ret.data[key]
                    match = false
                    exit for
                end if
            end for

            if match then
                m.data.set("CurrentChannel", channel)
                exit for
            end if
        end for
    else if allChannels.data.count() > 1 then
        ' If we failed to get the current channel, tune to the first channel
        ' in the list because this is likely caused by a DB change.
        print "INFO: tuning to first channel in list"
        RokuMfgCheckResponse(RokuMfg().call("tuner", {
            action: "set",
            data: allChannels.data[0]
        }))
    end if

    m.unlock()
end sub

'
' Invalidate any cached tuner data (e.g., ChannelList and CurrentChannel).
' Useful when the underlying tuner database has been reloaded to ensure the
' next tuner access repopulates the cache with updated information
'
' @param refresh whether cache should be repopulated immediately. Default is false (don't refresh)
sub RokuMfgState_TunerInvalidate(refresh=false as Boolean)
    if not m.lock() then
        print "WARNING: failed to invalidate tuner cache"
        return
    end if

    m.data.unset("ChannelList")
    m.data.unset("CurrentChannel")
    m.unlock()

    if refresh then
        m.refresh()
    end if
end sub
