'*********************************************************************
'** (c) 2019 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************

'
' Construct and initialize a Screen module object
' Provides the following functions:
' @see .enableAlpha RokuMfgScreenEnableAlphaOnScreen,
' @see .clear RokuMfgScreenClear,
' @see .fill RokuMfgScreenfill,
' @see .swapBuffers RokuMfgScreenSwapBuffers,
' @see .drawItem RokuMfgScreenDrawItem,
' @see .drawBase RokuMfgScreenDrawBaseLayer,
' @see .clearItems RokuMfgScreenClearItems,
' @see .addOSD RokuMfgScreenAddOSD,
' @see .deleteOSD RokuMfgScreenDeleteOSD,
' @see .prepareOSDs RokuMfgScreenPrepareOSDs,
' @see .updateOSDTimers RokuMfgScreenUpdateOSDTimers,
'
' @return o a new roAssociativeArray wrapping Screen functionality
function RokuMfgScreenModule(params=invalid as Object) as Object
    print "Initializing Screen Module"

    'Pick the best display size for the screen.
    devInfo = createObject("roDeviceInfo")
    display = invalid
    for each res in devInfo.GetSupportedGraphicsResolutions()
        if RokuMfgIsInvalid(display) or display.width < res.width then
            display = res
        end if
    end for
    if RokuMfgIsInvalid(display) then
        print "ERROR: No supported display resolutions found!"
        stop
    end if

    screen = createObject("roScreen", true, display.width, display.height)

    o = {
        impl: screen,

        'Dimensions, used by layout-generating code.
        width: screen.getWidth(),
        height: screen.getHeight(),
        visibleTop: 30,
        visibleLeft: 30,

        'Layouts for each layer get placed in these arrays.
        mainItems : p_createLayer("main"),
        midItems  : p_createLayer("mid"),
        menuItems : p_createLayer("menu"),
        osdItems  : p_createLayer("osd"),
        frontItems: p_createLayer("front"),

        osds: {},

        alphaEnabled: false,
        enableAlpha: RokuMfgScreenEnableAlphaOnScreen,
        clear: RokuMfgScreenClear,
        fill: RokuMfgScreenfill,
        swapBuffers: RokuMfgScreenSwapBuffers,
        drawItem: RokuMfgScreenDrawItem,
        drawBase: RokuMfgScreenDrawBaseLayer,
        clearItems: RokuMfgScreenClearItems,

        addOSD: RokuMfgScreenAddOSD,
        deleteOSD: RokuMfgScreenDeleteOSD,
        prepareOSDs: RokuMfgScreenPrepareOSDs,
        updateOSDTimers: RokuMfgScreenUpdateOSDTimers
    }

    if RokuMfgIsAA(params) then
        RokuMfgMerge(o, params)
    end if

    'Calculate visible screen dimensions.
    o.visibleWidth  = o.width - (2 * o.visibleLeft)
    o.visibleHeight = o.height - (2 * o.visibleTop)
    o.visibleRight  = o.visibleLeft + o.visibleWidth
    o.visibleBottom = o.visibleTop + o.visibleHeight

    'Since AAs can't enforce order without weirdness, we'll use a separate array.
    o.layers = [o.mainItems, o.midItems, o.menuItems, o.osdItems, o.frontItems]

    return o
end function

function p_createLayer(name as String) as Object
    o = {
        name: name,
        enable: true,
        items: [],
        append: p_appendToLayer,
        push: p_addItemToLayer,
        clear: p_clearLayer
    }
    return o
end function

sub p_appendToLayer(items as Object)
    m.items.append(items)
end sub

sub p_addItemToLayer(item as Object)
    m.items.push(item)
end sub

sub p_clearLayer()
    m.items.clear()
end sub

'
' Used to set alpha or not for displaying content on screen. Should only be called
' as enableAlpha from the RokuMfgScreenModule() object.
'
' @param enable true-enable alpha, false-disable alpha
sub RokuMfgScreenEnableAlphaOnScreen(enable as Boolean)
    m.impl.setAlphaEnable(enable)
    m.alphaEnabled = enable
end sub

'
' Used to clear the whole screen. Should only be called as clear from the
' RokuMfgScreenModule() object.
sub RokuMfgScreenClear()
    m.impl.clear(&h00000000)
end sub

'
' Used to fill screen with provided color . Should only be called as fill from
' the RokuMfgScreenModule() object.
'
' @param color the 32 bits color data, the format is RGBA.
sub RokuMfgScreenfill(color as Integer)
    m.impl.clear(color)
end sub

'
' This function complete all queued drawing operations on the back buffer (draw
' surface).The new back buffer should be assumed to be in a garbage state after
' this call is complete, which means you will need to re-render the entire frame
' before a subsequent call to SwapBuffers. This call will not return until the
' back buffer is ready to be drawn on to. Depending on the implementation, it may
' take up to a single video frame period for the new front buffer to become visible.
' Should only be called as swapBuffers from the RokuMfgScreenModule() object.
sub RokuMfgScreenSwapBuffers()
    m.impl.swapBuffers()
end sub

'
' Used to draw out provided object(support different types) on screen. Should only
' be called as drawItem from the RokuMfgScreenModule() object.
'
' @param item the provided object to be shown on screen, support "image","rect",
'     "text" and "box" types
' @param ctx the offset to the current location of provided object which will be
'      shown
sub RokuMfgScreenDrawItem(item as Object, ctx as Object)
    if item.type = "image" then
        if false = m.alphaEnabled then
            m.impl.setAlphaEnable(true)
            m.impl.drawScaledObject(item.x + ctx.x, item.y + ctx.y, item.scaleX, item.scaleY, item.source)
            m.impl.setAlphaEnable(false)
        else
            m.impl.drawScaledObject(item.x + ctx.x, item.y + ctx.y, item.scaleX, item.scaleY, item.source)
        end if
    else if "rect" = item.type then
        m.impl.drawRect(item.x + ctx.x, item.y + ctx.y, item.w, item.h, item.color)
    else if "text" = item.type then
        m.impl.drawText(item.text, item.x + ctx.x, item.y + ctx.y, item.color, item.font)
    else if "box" = item.type then
        m.impl.drawRect(item.x + ctx.x, item.y + ctx.y, item.getWidth(), item.getHeight(), item.color)
        for each subitem in item.content
            m.drawItem(subitem, {x: ctx.x + item.x + item.padding, y: ctx.y + item.y + item.padding})
        end for
    end if
end sub

'
' Used to draw out all content on all logic layers on screen. Should only be
' called as drawBase from the RokuMfgScreenModule() object.
sub RokuMfgScreenDrawBaseLayer()
   'Set opaque background
    m.impl.clear(&h00000000)

    for each layer in m.layers
        if layer.enable then
            for each item in layer.items
                m.drawItem(item, {x: 0, y: 0})
            end for
        end if
    end for

    m.swapBuffers()
end sub

'
' Used to clear all object items on all logic layers of screen module. Should
' only be called as clearItems from the RokuMfgScreenModule() object.
sub RokuMfgScreenClearItems()
    m.mainItems.clear()
    m.midItems.clear()
    m.menuItems.clear()
    m.osdItems.clear()
    m.frontItems.clear()
end sub

sub p_deleteOSD()
    if RokuMfgHasInterface(m.onDelete, "ifFunction")
        m.onDelete()
    end if

    'not same with m.osds[key].delete(), it will call osds array child API "delete" to delete the OSD.
    m.screen.osds.delete(m.name)
end sub

'
' Used to add given osd object to screen module osd stack "osd{}" . Should only
' be called as addOSD from the RokuMfgScreenModule() object.
'
' @param ms the extra timeout value (which osd displaying will disappear) to be
'      added to given osd object
' @param osd the given osd object
'
' @return o a new osd object which is merged with defauts settings.
function RokuMfgScreenAddOSD(ms as Integer, osd as Object) as Object
    defaults = {
        name: invalid,
        timeout: 3000,
        display: {
            type: "text",
            info: {
                text: "OSD Example",
                textAttrs: {
                    color: "white"
                },
                targetRect: {x: 200, y: 200}
            }
        }
        screen: m,
        delete: p_deleteOSD,
        onDelete: invalid
    }

    o = RokuMfgDeepMerge(defaults, osd)

    if RokuMfgIsInt(o.timeout) then
        o.timeout = o.timeout + ms
    end if

    ' Generate an anonymous name in case one isn't specified.
    if RokuMfgIsInvalid(o.name) then
        i = 0
        while true
            name = "anonymous" + RokuMfgStrCast(i)
            if RokuMfgIsInvalid(m.osds[name]) then
                o.name = name
                exit while
            end if
        end while
    end if

    m.osds[o.name] = o
    return o
end function

'
' Used to delete one defined "osd" object from osd stack "osd{}" . Should only
' be called as deleteOSD from the RokuMfgScreenModule() object.
'
' @param id the "id" of "osd" object to be deleted
sub RokuMfgScreenDeleteOSD(id as string)
    if RokuMfgIsAA(m.osds[id]) then
        m.osds.delete(id)
    end if
end sub

'
' Used to add all osd objects in osd stack "osd{}" to "osd" logic layer of screen
' module, Should only be called as prepareOSDs from the RokuMfgScreenModule() object.
sub RokuMfgScreenPrepareOSDs()
    m.osdItems.clear()

    for each key in m.osds
        osd = m.osds[key]

        if RokuMfgIsAA(osd) then
            if RokuMfgIsAA(osd.display) then
                m.osdItems.push(osd.display)
            else if RokuMfgIsArray(osd.display) then
                m.osdItems.append(osd.display)
            end if
        end if
    end for
end sub

'
' Used to update the timeout time of all osd objects in osd stack "osd{}", new
' timeout time should be value of "old timeout time" subject the provided consumed
' time "ms" . Should only be called as updateOSDTimers from the RokuMfgScreenModule()
' object.
'
' @param ms the consumed time out time to be subject
' @param nextTimeout the supposed timeout time for the next osd object which will disappear
'
' @return nextTimeout the timeout time for the next osd object which will disappear
function RokuMfgScreenUpdateOSDTimers(ms as Integer, nextTimeout) as Integer
    for each key in m.osds
        osd = m.osds[key]

        if RokuMfgIsAA(osd) and RokuMfgIsInt(osd.timeout) then
            osd.timeout = osd.timeout - ms

            if 0 >= osd.timeout then
                'It will call the deleteOSD function by default if outside parameter
                'does not register the delete function when addOSD() is called.
                m.osds[key].delete()
                osd = invalid
            else
                if nextTimeout = 0 or nextTimeout > osd.timeout then
                    nextTimeout = osd.timeout
                end if
            end if
        end if
    end for

    return nextTimeout
end function