'*********************************************************************
'** (c) 2019 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************

'
' Construct and initialize a KeyFilter module object
' Provides the following functions:
' @see .install RokuMfgKeyFilterInstall,
' @see .remove RokuMfgKeyFilterRemove,
' @see .removeUntil RokuMfgKeyFilterRemoveUntil,
' @see .getHoldDuration RokuMfgKeyFilterGetKeyHoldDuration,
' @see .getKeyType RokuMfgKeyFilterGetKeyType,
' @see .handle RokuMfgKeyFilterHandleKeyEvent,
'
' @return keyFilter a new roAssociativeArray wrapping keyFilter functionality
function RokuMfgKeyFilterModule(params=invalid as Object) as Object
    print "Initializing KeyFilter Module"
    keyFilter = {
        install: RokuMfgKeyFilterInstall,
        remove: RokuMfgKeyFilterRemove,
        removeUntil: RokuMfgKeyFilterRemoveUntil,
        getHoldDuration: RokuMfgKeyFilterGetKeyHoldDuration,
        getKeyType: RokuMfgKeyFilterGetKeyType,
        handle: RokuMfgKeyFilterHandleKeyEvent,

        filters: [],
        lastKeyEvent: {},
        pressTimer: createObject("roTimespan"),
        isPressed: false
    }

    if RokuMfgIsAA(params) then
        RokuMfgMerge(keyFilter, params)
    end if

    return keyFilter
end function

'
' Used to install a key filter handle function to Key filter module. Should only
' be called as install from the RokuMfgKeyFilterModule() object.
'
' @param filter key filter handle function to be installed
' @param context the parameter to be given when key filter handle function is called
'
' @return true if install was successful, false otherwise
function RokuMfgKeyFilterInstall(filter as Function, context as Object) as Boolean
    ' Don't allow duplicate keyfilters?
    if false = RokuMfgIsInvalid(filter) and false = RokuMfgIsInvalid(context) then
        m.filters.unshift({func: filter, context: context})
    else
        return false
    end if

    return true
end function

'
' Used to remove a key filter handle function from Key filter module. Should only
' be called as remove from the RokuMfgKeyFilterModule() object.
'
' @param filter key filter handle function to be removed
'
' @return true if remove was successful, false otherwise
function RokuMfgKeyFilterRemove(filter) as Boolean
    for i = 0 to m.filters.count() - 1
        if filter = m.filters[i].func then
            m.filters.delete(i)
            return true
        end if
    end for

    return false
end function

'
' Removes all key filters from the top until the filter passed in is
' detected.  Doesn't remove any filters if filter isn't currently
' in the filter stack. Should only be called as removeUntil from the
' RokuMfgKeyFilterModule() object.
'
' @param filter the end key filter handle function when key filter is
'               removed from top
'
' @return true if remove was successful, false otherwise
function RokuMfgKeyFilterRemoveUntil(filter) as Boolean
    ' First check to see if the filter exists.
    for each f in m.filters
        if filter = f.func then
            while filter <> m.filters[0].func
                m.filters.shift()
            end while

            m.filters.shift()
            return true
        end if
    end for

    return false
end function

'
' Used to get key hold duration time. Should only  be called as getHoldDuration
' from the RokuMfgKeyFilterModule() object.
'
' @return the key hold duration time
function RokuMfgKeyFilterGetKeyHoldDuration() as Integer
    if m.isPressed then
        return m.pressTimer.totalMilliseconds()
    end if

    return 0
end function

'
' Used to get key type (TV KEY PAD or REMOTE) for the last pressed key event.
' Should only be called as getKeyType from the RokuMfgKeyFilterModule() object.
'
' @return "PANELBUTTON", "REMOTE", or "UNKNOWN"
function RokuMfgKeyFilterGetKeyType() as String
    if false = RokuMfgIsInvalid(m.lastKeyEvent) and 2 = m.lastKeyEvent.rawKeyType then
        return "PANELBUTTON"
    else if false = RokuMfgIsInvalid(m.lastKeyEvent) and 0 = m.lastKeyEvent.rawKeyType then
        return "REMOTE"
    end if

    return "UNKNOWN"
end function

'
' Keypress event handle function, this function will deal with all installed
' key filter handle functions one by one, the last installed key filter function
' will be delt firstly, if one key press event is delt, then return directly
' Should only be called as handle from the RokuMfgKeyFilterModule() object.
'
' @param event the keypress event to handle.
'
' @return true if keypress event is delt with successfully, false otherwise
function RokuMfgKeyFilterHandleKeyEvent(event as Object) as Boolean
    if RokuMfgIsInvalid(event) or RokuMfgIsInvalid(event.rawKeyPressed) or RokuMfgIsInvalid(event.rawKeyValue) then
        print "KeyfilterHandle: Input event is not valid"
        return false
    end if

    pressed = event.rawKeyPressed
    keycode = event.rawKeyValue

    m.lastKeyEvent = event

    if false = pressed then
        m.isPressed = false
        return true
    end if

    if not m.isPressed then
        m.isPressed = true
        m.pressTimer.mark()
    end if

    for each filter in m.filters
        keycode = filter.func(filter.context, keycode)
        if invalid = keycode then
            print "KeyfilterHandle: keyevent handled in: ", filter.func
            return true
        end if
    end for

    print "KeyfilterHandle: Key code " + RokuMfgStrCast(keycode) + " was not handled."
    return false
end function