'*********************************************************************
'** (c) 2019 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************

'
' Construct a I2C module object
' Provides the following functions:
' @see .getPort RokuMfgI2CGetPort
' @see .setPort RokuMfgI2CSetPort
' @see .getDevAddress RokuMfgI2CGetDevAddress
' @see .setDevAddress RokuMfgI2CSetDevAddress
' @see .read RokuMfgI2CRead
' @see .write RokuMfgI2CWrite
'
' @param params roAssociativeArray of parameters to merge into the returned object.
'     Also allows overriding default functionality and adding features not
'     provided by the default implementation.
'     Default is invalid.
'
' @return i2c a new roAssociativeArray wrapping I2C functionality
function RokuMfgI2CModule(params=invalid as Object) as Object
    print "Initializing I2C Module"
    i2c = {
        port: 3,
        devaddress: 0

        getPort: RokuMfgI2CGetPort,
        setPort: RokuMfgI2CSetPort,
        getDevAddress: RokuMfgI2CGetDevAddress,
        setDevAddress: RokuMfgI2CSetDevAddress,
        read: RokuMfgI2CRead,
        write: RokuMfgI2CWrite
    }

    if RokuMfgIsAA(params) then
        RokuMfgMerge(i2c, params)
    end if

    'Set default port of I2C device.
    i2c.port = p_getI2CDefaultPort()

    return i2c
end function

function p_getI2CDefaultPort() as Integer
    'Try to read the port number from customer package settings firstly.
    key = "board/i2c/amp/bus"
    ret = RokuMfg().call("custominfo", {
        action: "get",
        data: key
    })

    if RokuMfgCheckResponse(ret) then
        port = RokuMfgIntCast(ret.data[key])
    else
        port_table = {
            "MStar-T10": 3,
            "MStar-T14": 6,
            "MStar-T22": 3,
            "RTD-2873": 1
        }

        chip = RokuMfg().sysinfo().chiptype
        if not RokuMfgIsInvalid(port_table[chip]) then
            port = port_table[chip]
        else
            print "Unknown chip type " + chip
            port = 0
        end if
    end if

    return port
end function

'
' Used to get the I2C bus port number for current platform. Should only be called
' as getPort from the RokuMfgI2CModule() object.
'
' @return the I2C bus port number
function RokuMfgI2CGetPort() as Integer
    return m.port
end function

'
' Used to change the I2C bus port number for I2C operation. Should only be called
' as setPort from the RokuMfgI2CModule() object.
'
' @param port the I2C bus port number to change.
'
' @return always return true
function RokuMfgI2CSetPort(port as Integer) as Boolean
    m.port = port
    return true
end function

'
' Used to get the device address on the i2c bus for target i2c device. Should
' only be called as getDevAddress from the RokuMfgI2CModule() object.
'
' @return the device address on the i2c bus
function RokuMfgI2CGetDevAddress() as Integer
    return m.devaddress
end function

'
' Used to change the device address on the i2c bus for target i2c device. Should
' only be called as setDevAddress from the RokuMfgI2CModule() object.
'
' @param devaddress the device address on the i2c bus for target i2c device.
'
' @return always return true
function RokuMfgI2CSetDevAddress(devaddress as Integer) as Boolean
    m.devaddress = devaddress
    return true
end function

'
' Used to read some data from target i2c device. Should only be called as read
' from the RokuMfgI2CModule() object.
'
' @param numbytes Number of bytes to read from the sink device.
' @param subaddress (optional) the slave address to access on the sink device.
'
' @return the readback data from target i2c device
function RokuMfgI2CRead(numbytes as Integer, subaddress=invalid) as Object
    pl = {
        action: "read",
        port: m.port,
        devaddress: m.devaddress,
        numbytes: numbytes
    }

    if not RokuMfgIsInvalid(subaddress) then
        pl.subaddress = subaddress
    end if

    ret = RokuMfg().call("i2c", pl)
    if RokuMfgCheckResponse(ret) then
        ba = createObject("roByteArray")
        ba.fromBase64String(ret.data)
        return ba
    end if

    print "RokuMfgI2CRead: read failed! devaddress = " + RokuMfgStrCast(devaddress) + "subaddress = " + RokuMfgStrCast(subaddress)
    return []
end function

'
' Used to write some data to target i2c device. Should only be called as write
' from the RokuMfgI2CModule() object.
'
' @param data the array of data to write. allow "roByteArray" or "Int" type
'     data format
' @param subaddress (optional) The device subaddress to write to.
'
' @return true if the data is written into i2c device successfully; Otherwise
'         return false.
function RokuMfgI2CWrite(data as Object, subaddress=invalid) as Boolean
    if RokuMfgIsInvalid(data) then
        print "RokuMfgI2CWrite: No input data to write!"
        return false
    end if

    if not RokuMfgHasInterface(data, "ifByteArray") then
        ba = createObject("roByteArray")
        if RokuMfgIsArray(data) then
            for each el in data
                ba.push(el)
            end for
        else
            ' This is a legacy case that allows us to pass in data as an
            ' integer truncated to a byte (e.g., &0hff)
            ba.push(data)
        end if
    else
        ba = data
    end if

    pl = {
        action: "write",
        port: m.port,
        devaddress: m.devaddress,
        data: ba.toBase64String()
    }

    if not RokuMfgIsInvalid(subaddress) then
        pl.subaddress = subaddress
    end if

    ret = RokuMfg().call("i2c", pl)
    if RokuMfgCheckResponse(ret) then
        return true
    end if

    print "RokuMfgI2CWrite: write failed!"
    return false
end function
