'*********************************************************************
'** (c) 2019 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************

'
' Construct and initialize a "range" type "defined data" object
' Provides the following functions:
' @see .update RokuMfgDataUpdateRange
' @see .set RokuMfgDataSetRange
' @see .get RokuMfgDataGetRangeValue,
' @see .getValue RokuMfgDataGetRangeValue,
' @see .within RokuMfgDataIsValueWithinRange
' @see .refresh RokuMfgDataRefreshRangeValue
'
' Used to construct and initialize a "range" type "defined data" object. Should
' only be called as constructRange from the RokuMfgDataModule() object.
'
' @param params roAssociativeArray of parameters to merge into the returned object
'     Supported keys:
'     * id: the "id" of defined "data" object
'     * type: the data type of defined "data" object, support "Float" and "Int" value
'     * onInit: the callback function triggered when constructing a data object
'     * onRfresh: the callback function triggered when calling .get() to obtain an un-cached data object
'     * onUpdate: the callback function triggered when calling .set() to modify a data object
'     Also allows overriding default functionality and adding features not
'     provided by the default implementation.
'
' @return the defined "data" object
function RokuMfgDataConstructRange(params as Object) as Object
    o = {
        class_: "Range",
        type: "Int",
        range: [0, 100],
        value: 0,
        cached: true,
        update: RokuMfgDataUpdateRange,
        set: RokuMfgDataSetRange,
        get: RokuMfgDataGetRangeValue,          ' for uniformity in the API
        getValue: RokuMfgDataGetRangeValue,      ' for uniformity in the API
        within: RokuMfgDataIsValueWithinRange,
        refresh: RokuMfgDataRefreshRangeValue,

        onInit: invalid,
        onRefresh: invalid,
        onUpdate: invalid
    }

    if RokuMfgIsAA(params) then
        RokuMfgMerge(o, params)
    end if

    if "Float" = o.type then
        o.set = RokuMfgDataSetRangeFloat
        o.get = RokuMfgDataGetRangeValueFloat          ' for uniformity in the API
        o.getValue = RokuMfgDataGetRangeValueFloat     ' for uniformity in the API
        o.update = RokuMfgDataUpdateRangeFloat
        o.within = RokuMfgDataIsValueWithinRangeFloat
    end if

    ' Add to data store if ID isn't invalid
    if not RokuMfgIsInvalid(o.id) then
        o.app = m.app
        o.data = m
        m.add(o.id, o)
    end if

    if RokuMfgHasInterface(o.onInit, "ifFunction") then
        o.onInit()
    end if

    return o
end function

'
' Used to set a value for "range" type "defined data" object. Should only be
' called as set from the RokuMfgDataConstructRange() object.
'
' @param value the value to be set
'
' @return the read back value of "range" type "defined data" object after it has
'         been set.
function RokuMfgDataSetRange(value as Integer) as Integer
    ' Fix range
    if value < m.range[0] then
        value = m.range[0]
    else if value > m.range[1] then
        value = m.range[1]
    end if

    m.value = value

    if RokuMfgHasInterface(m.onUpdate, "ifFunction") then
        m.onUpdate()
    end if

    return m.get()
end function

function RokuMfgDataSetRangeFloat(value as Float) as Float
    ' Fix range
    if value < m.range[0] then
        value = m.range[0]
    else if value > m.range[1] then
        value = m.range[1]
    end if

    m.value = value

    if RokuMfgHasInterface(m.onUpdate, "ifFunction") then
        m.onUpdate()
    end if

    return m.get()
end function

'
' Used to increase the value for "range" type "defined data" object. Should
' only be called as update from the RokuMfgDataConstructRange() object.
'
' @param increment the incremental value to be added to current value for "range"
'     type "defined data" object
'
' @return the read back value of "range" type "defined data" object after it has
'         been set.
function RokuMfgDataUpdateRange(increment as Integer) as Integer
    newValue = m.value + increment
    return m.set(newValue)
end function

function RokuMfgDataUpdateRangeFloat(increment as Float) as Float
    newValue = m.value + increment
    return m.set(newValue)
end function
'
' Used to get the current value for "range" type "defined data" object. Should
' only be called as get or getValue from the RokuMfgDataConstructRange() object.
'
' @return the current value for "range" type "defined data" object
function RokuMfgDataGetRangeValue() as Integer
    if not m.cached then
        m.refresh()
    end if
    return m.value
end function

function RokuMfgDataGetRangeValueFloat() as Float
    if not m.cached then
        m.refresh()
    end if
    return m.value
end function

'
' Used to check whether the provided value is in the allowed value range for
' "range" type "defined data" object. Should only be called as within from the
'  RokuMfgDataConstructRange() object.
'
' @value the given value to be check
'
' @return true if value is in the range, false otherwise
function RokuMfgDataIsValueWithinRange(value as Integer) as Boolean
    if value >= m.range[0] and value <= m.range[1] then
        return true
    end if

    return false
end function

function RokuMfgDataIsValueWithinRangeFloat(value as Float) as Boolean
    if value >= m.range[0] and value <= m.range[1] then
        return true
    end if

    return false
end function

'
' Used to update the value to platform for "range" type "defined data" object.
' Should only be called as refresh from the RokuMfgDataConstructRange() object.
sub RokuMfgDataRefreshRangeValue()
    if RokuMfgHasInterface(m.onRefresh, "ifFunction") then
        m.onRefresh()
    end if
end sub