'*********************************************************************
'** (c) 2018-2019 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************
' Roku_MFG_Legacy_STB.brs
' Wrappers for the legacy "stb" APIs

' @api internal Supports4kHDMITX
' Determine if the device is a 4K HDMI source.
' This API is deprecated -- use #GetSinkEdid instead.
'
' @return
' Integer valid - 0 : Does not support 4K HDMI TX; 1 : Supports 4K HDMI TX
' String error - Description of error condition.
function RokuMfgLegacy_Supports4kHDMITX() as Object
    edid = RokuMfgLegacy_GetSinkEdid()
    if 1 = edid.valid then
        if 1 = edid.Supports4k then
            return RokuMfgLegacySuccess()
        else
            return RokuMfgLegacyError("does not support 4ktx")
        end if
    end if

    return RokuMfgLegacyError(edid.error)
end function

' @api internal GetMonitorNameHDMITX
' Get information about the HDMI sink (monitor) connected to this device.
' This function is deprecated -- use #GetSinkEdid instead.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String name - The name of the sink.
' String Summary - Summary info, if available and supported.
function RokuMfgLegacy_GetMonitorNameHDMITX() as Object
    edid = RokuMfgLegacy_GetSinkEdid()
    if 0 = edid.valid then
        return RokuMfgLegacyError(edid.error)
    end if

    if RokuMfgIsChipType("MStar-3Z") then
        if "" <> edid.Summary then
            return RokuMfgLegacySuccess({
                summary: edid.Summary
            })
        else
            return RokuMfgLegacyError("no summary info")
        end if
    else
        if "" <> edid.MonitorName then
            return RokuMfgLegacySuccess({
                name: edid.MonitorName
            })
        else
            return RokuMfgLegacyError("no monitor name")
        end if
    end if

    return RokuMfgLegacyError("unhandled case")
end function

' @api internal SetFan
' Set the internal fan's to a particular mode.
' This API is currently only supported on the Dallas and Richmond projects.
'
' @args
' String setting - One of: on, off, pulse
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetFan(setting as String) as Object
    pl = {action: "set"}
    regex = createObject("roRegex", "[0-9]+", "i")
    if regex.isMatch(setting) then
        pl.data = val(setting, 10)
    else
        pl.data = setting
    end if

    ret = RokuMfg().call("fan", pl)
    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api internal GetFanRpm
' Get the RPM of the fan from the tachometer.
' This code internally calls fanrpm.
' This API is currently only supported on the Dallas project.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String rpm - The string returned from fanrpm
function RokuMfgLegacy_GetFanRpm() as Object
    ret = RokuMfg().call("fan", {action: "get"})
    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            rpm: ret.data
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api internal RunLitepointCmd
' Run a litepoint test command.
'
' @args
' roAssociativeArray data {
'     String command - The name of the command to run.  One of, wl, simple_app, rtwpriv, rtlbtmp
'     roArray args - A list of arguments to provide to the command being run.  This must always be present, even if empty.
' }
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - In this case, only exists when an exception is thrown.
' String output - The output from the command.
function RokuMfgLegacy_RunLitepointCmd(data as Object) as Object
    if not RokuMfgIsString(data.command) then
        return RokuMfgLegacyError("expected data to contain command string")
    else if not RokuMfgIsArray(data.args) then
        return RokuMfgLegacyError("expected data to contain args array")
    end if

    ret = RokuMfg().call("litepoint", {
        action: data.command,
        data: data.args
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({output: ret.data})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api internal GetHardwareCapabilities
' Get a list of hardware capabilities for the device.
' For each capability listed, a 1 means the device supports that feature, and a 0
' means the device does not support that feature.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer 4k - Device supports 4K playback.
' Integer 1080p - Device supports 1080p playback.
' Integer ethernet - Device supports ethernet (wired) connections.
' Integer bluetooth - Device supports bluetooth.
' Integer sdcard - Device has at least one SD card slot.
' Integer usb - Device has at least one USB port.
function RokuMfgLegacy_GetHardwareCapabilities() as Object
    capabilities = getGlobalAA().mfg_constants.sysinfo.capabilities
    data = {}
    keys = ["4k", "1080p", "ethernet", "bluetooth", "sdcard", "usb"]
    for each key in keys
        ' API expects return values as Integers
        if not RokuMfgIsInvalid(capabilities[key]) and true = capabilities[key] then
            data[key] = 1
        else
            data[key] = 0
        end if
    end for

    return RokuMfgLegacySuccess(data)
end function

' @api internal GetHdmiOutputMode
' Get the current HDMI video output mode.
' It will provide the resolution and whether the signal is interlaced or progressive.
'
' Examples: 480i, 1080p, 2160p30
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String mode - HDMI video output mode.
function RokuMfgLegacy_GetHdmiOutputMode() as Object
    ret = RokuMfg().call("video", {
        action: "get",
        component: "mode"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({mode: ret.data.video_mode})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api internal SetHdmiOutputMode
' Set the current HDMI video output mode.
' Provide the resolution and whether the signal is interlaced or progressive, and can optionally
' provide the framerate.
'
' Examples: 480i, 1080p, 2160p30
'
' @args
' String mode_str - The requested HDMI video output mode.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetHdmiOutputMode(mode_str as String) as Object
    ret = RokuMfg().call("video", {
        action: "set",
        component: "mode",
        data: mode_str
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api internal GetSinkEdid
' Get EDID information from the connected HDMI sink.
'
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String Manufacturer - The sink manufacturer's name.
' String MonitorName - The sink's name.
' Integer ProductCode - The sink's product code.
' Integer SerialNumber - The sink's serial number.
' Integer ManufactureWeek - The sink's week of manufacture.
' Integer ManufactureYear - The sink's year of manufacture.
' String Summary -  Summary of all EDID data.
' roArray SupportedModes -  All supported output modes as strings.
' Integer Supports4k -  Does the sink support 4k.
' Integer Supports4k30 -  Does the sink support 4k30.
' Integer Supports4k60 -  Does the sink support 4k60.
function RokuMfgLegacy_GetSinkEdid() as Object
    ret = RokuMfg().call("hdmi", {
        action: "get",
        component: "txedid"
    })

    if RokuMfgCheckResponse(ret) then
        keys = []
        return RokuMfgLegacySuccess(ret.data)
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function
