'*********************************************************************
'** (c) 2018-2019 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************
' Roku_MFG_Legacy_Raw.brs
' Wrappers for the legacy "raw" APIs

' @api EnableRawKeys
' Enable raw key handling.
' By default, the Roku OS ignores IR keycodes that do not match Roku's or OEM's
' requested key set. Additionally, other keys are handled by the system, rather
' than being passed down to the BrightScript application (e.g. the Power key).
'
' This API changes both behaviors, and instead passes every valid key to the
' BrightScript application for custom handling.
function RokuMfgLegacy_EnableRawKeys()
    ret = RokuMfg().call("rc_ir", {
        action: "raw",
        enable: true
    })

    ' Original implementation returned void.
    ' Check the response but don'd do anything with it.
    RokuMfgCheckResponse(ret)
end function

' @api DisableRawKeys
' Disable raw key handling.
' By default, the Roku OS ignores IR keycodes that do not match Roku's or OEM's
' requested key set. Additionally, other keys are handled by the system, rather
' than being passed down to the BrightScript application (e.g. the Power key).
'
' If the BrightScript application has already enabled raw key handling via
' #EnableRawKeys, then this API returns key handling back to system defaults.
function RokuMfgLegacy_DisableRawKeys()
    ret = RokuMfg().call("rc_ir", {
        action: "raw",
        enable: false
    })

    ' Original implementation returned void.
    ' Check the response but don'd do anything with it.
    RokuMfgCheckResponse(ret)
end function

' @api EnableKeyReleaseEvents
' Allow BrightScript apps to respond to key release events.
' The default behavior is to only pass key press events to BrightScript, to reduce
' complexity. However, some manufacturing requirements necessitate knowing when
' a key is released, to distinguish between a long key press and a short key press.
'
' @args
' Integer enable - 0 : disable key release events; 1 : enable key release events
function RokuMfgLegacy_EnableKeyReleaseEvents(enable as Integer)
    ret = RokuMfg().call("rc_ir", {
        action: "release",
        enable: RokuMfgBoolCast(enable)
    })

    ' Original implementation returned void.
    ' Check the response but don'd do anything with it.
    RokuMfgCheckResponse(ret)
end function

' @api EnableRawKeyLedEvent
' Control whether the TV's LED blinks when a rawkey is handled.
' The default behavior is to blink for every rawkey.
' Use this API to help keep the LED off (e.g., to avoid interference with IR sensor).
'
' @args
' Integer enable - 0 : disable led response; 1 : enable led response
function RokuMfgLegacy_EnableRawKeyLedEvent(enable as Integer)
    ret = RokuMfg().call("rc_ir", {
        action: "led",
        enable: RokuMfgBoolCast(enable)
    })

    ' Original implementation returned void.
    ' Check the response but don'd do anything with it.
    RokuMfgCheckResponse(ret)
end function

