'*********************************************************************
'** (c) 2018-2019 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************
' Roku_MFG_Legacy_PQ.brs
' Wrappers for the legacy "pq" APIs
Library "state/Roku_MFG_State.brs"

' @api SavePQOnExit
' Enable/Disable PQ saving on application exit.
' When the application starts, save on exit is disabled by default.
'
' @args
' Integer enable - 0 : do not save PQ on exit; 1 : save PQ on exit.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SavePQOnExit(enable as Integer) as Object
    ret = RokuMfg().call("pq", {
        action: "saveonexit",
        data: RokuMfgBoolCast(enable)
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SavePQSettings
' Save current PQ database to persistent storage.
' THIS FUNCTION IS DEPRECATED.  It duplicates functionality with SavePQDBChanges().
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SavePQSettings() as Object
    return RokuMfgLegacyError("This API is deprecated. Use SavePQDBChanges.")
end function

' @api SavePQDBChanges
' Save all PQ database changes to persistent storage.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SavePQDBChanges() as Object
    ret = RokuMfg().call("pq", {
        action: "save"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetPQDBEntry
' Get the specified PQ database parameter.
'
' @args
' String name - The name of the PQ database entry to be retrieved.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer value - The value of the specified PQ parameter.
function RokuMfgLegacy_GetPQDBEntry(name as String) as Object
    ret = RokuMfg().call("pq", {
        action: "get",
        component: "setting",
        data: name
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            value: val(ret.data[name], 10)
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetPQDBEntry
' Set the specified parameter in the PQ database.
'
' @args
' roAssociativeArray data {
'     String name - The name of the PQ database entry to modify.
'     String type - The type of entry, one of: int, string
'     Dynamic value - The value to be set.
' }
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetPQDBEntry(data as Object) as Object
    mfg = RokuMfg()
    if not mfg.util.isString(data.name) then
        return RokuMfgLegacyError("expected name as string")
    else if mfg.util.isInvalid(data.value) then
        return RokuMfgLegacyError("expected valid data value")
    end if

    pl = {}
    pl[data.name] = mfg.util.strCast(data.value)
    ret = mfg.call("pq", {
        action: "set",
        component: "setting",
        data: pl
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api ApplyPQUpdateEverywhere
' Apply the current value of a PQ parameter to all PQ database entries in the specified layer.
'
' Applies a single parameter value to all PQDB entries at the specified layer (e.g., all GeekSquad
' layer "Normal" presets, or all GeekSquad layer Warm color temperatures. This function only works
' on the GeekSquad and InputType layers (Global only has one entry for any object, the user levels
' don't make sense for this function)
'
' @args
' String parameter - The parameter to apply everywhere.
' Integer layer - Base layer to apply to. 2 : Geek Squad; 3 : Input type
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_ApplyPQUpdateEverywhere(parameter as String, layer as Integer) as Object
    layer_map = {
        "2": "persistent",
        "3": "input"
    }

    layer_str = RokuMfgStrCast(layer)
    if RokuMfgIsInvalid(layer_map[layer_str]) then
        return RokuMfgLegacyError("invalid layer")
    end if

    ret = RokuMfg().call("pq", {
        action: "apply",
        component: "everywhere",
        layer: layer_map[layer_str],
        data: parameter
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api ResetPQDBLayer
' Reset an entire layer of the PQ database.
' This, in particular, allows lower layers to "show through" when previously, the higher layers have
' already overridden those particular settings.  Use with caution, as it includes wiping user
' preferences.
'
' @args
' Integer layer - The layer to reset.  1 : user; 2 : Geek Squad; 3 : Input Type; 4 : Global
' Integer curOnly - 0 : Apply changes to all layers above selected; 1 : only clear the specified layer.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_ResetPQDBLayer(layer as Integer, curOnly as Integer) as Object
    layer_map = {
        "1": "user",
        "2": "persistent",
        "3": "input",
        "4": "global"
    }

    layer_str = RokuMfgStrCast(layer)
    if RokuMfgIsInvalid(layer_map[layer_str]) then
        return RokuMfgLegacyError("unsupported layer")
    end if

    ret = RokuMfg().call("pq", {
        action: "reset",
        data: layer_map[layer_str],
        global: not RokuMfgBoolCast(curOnly)
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetAllColorTemps
' Get all available color temperature options.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' roArray temps - A list containing available color temperatures.
function RokuMfgLegacy_GetAllColorTemps() as Object
    temps = []
    for each el in RokuMfgState().get("PqKeyValues").color_temp
        temps.push(el.pqvalue)
    end for

    return RokuMfgLegacySuccess({temps: temps})
end function

' @api GetColorTemp
' Get the current color temperature setting.
'
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String temp - The color temperature currently in use, one of: normal, cool, warm.
function RokuMfgLegacy_GetColorTemp() as Object
    key = "ColorTemperature"
    ret = RokuMfg().call("pq", {
        action: "get",
        component: "setting",
        data: key
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({temp: ret.data[key]})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetColorTemp
' Set the current color temperature.
'
' @args
' String tempstr - The color temperature to change to, one of: normal, cool, warm.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetColorTemp(tempstr as String) as Object
    mfg = RokuMfg()
    key = "ColorTemperature"
    pl = {}
    pl[key] = mfg.util.capitalize(tempstr)
    ret = mfg.call("pq", {
        action: "set",
        component: "setting",
        data: pl
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetWhiteBalance
' Get the white balance level for the current source.
'
' @args
' String colorstr - The white balance color, one of: red, green, blue.
' String paramstr - The white balance paramter, one of: offset, gain.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer value - The current white balance parameter value.
function RokuMfgLegacy_GetWhiteBalance(colorstr as String, paramstr as String) as Object
    mfg = RokuMfg()
    key = "WB_" + mfg.util.capitalize(colorstr) + "_" + mfg.util.capitalize(paramstr)
    ret = mfg.call("pq", {
        action: "get",
        component: "setting",
        data: key
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({value: val(ret.data[key], 10)})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetWhiteBalance
' Set the current white balance value for the current source.
'
' @args
' String colorstr - The white balance color to adjust, one of: red, green, blue.
' String paramstr - The white balance parameter to adjust, one of: offset, gain.
' Integer value - The value to set.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetWhiteBalance(colorstr as String, paramstr as String, value as Integer) as Object
    mfg = RokuMfg()
    key = "WB_" + mfg.util.capitalize(colorstr) + "_" + mfg.util.capitalize(paramstr)
    pl = {}
    pl[key] = value
    ret = mfg.call("pq", {
        action: "set",
        component: "setting",
        data: pl
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetTempModeWhiteBalance
' Get a white balance parameter for a given color temperature, on the current source.
'
' @args
' String tempmode - The white balance color temperature, one of: normal, cool, warm.
' String colorstr - The white balance color, one of: red, green, blue.
' String paramstr - The white balance parameter, one of: offset, gain.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer value - The value of the white balance parameter
function RokuMfgLegacy_GetTempModeWhiteBalance(tempmode as String, colorstr as String, paramstr as String) as Object
    saved = RokuMfgLegacy_GetColorTemp()
    if 1 <> saved.valid then
        return RokuMfgLegacyError("failed to get current color temp")
    end if

    ret = RokuMfgLegacy_SetColorTemp(tempmode)
    if 1 <> ret.valid then
        return RokuMfgLegacyError("failed to set temp to " + tempmode)
    end if

    mfg = RokuMfg()
    key = "WB_" + mfg.util.capitalize(colorstr) + "_" + mfg.util.capitalize(paramstr)
    ret = RokuMfg().call("pq", {
        action: "get",
        component: "setting",
        data: key
    })

    RokuMfgLegacy_SetColorTemp(saved.temp)
    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({value: val(ret.data[key], 10)})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetTempModeWhiteBalance
' Set the white balance parameter for a given color temperature, on the current source.
'
' @args
' String tempmode - The white balance color temperature, one of: normal, cool, warm.
' String colorstr - The white balance color, one of: red, green, blue.
' String paramstr - The white balance parameter, one of: offset, gain.
' Integer value - The value to be set.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetTempModeWhiteBalance(tempmode as String, colorstr as String, paramstr  as String, value as Integer) as Object
    saved = RokuMfgLegacy_GetColorTemp()
    if 1 <> saved.valid then
        return RokuMfgLegacyError("failed to get current color temp")
    end if

    ret = RokuMfgLegacy_SetColorTemp(tempmode)
    if 1 <> ret.valid then
        return RokuMfgLegacyError("failed to set temp to " + tempmode)
    end if

    mfg = RokuMfg()
    key = "WB_" + mfg.util.capitalize(colorstr) + "_" + mfg.util.capitalize(paramstr)
    pl = {}
    pl[key] = value
    ret = RokuMfg().call("pq", {
        action: "set",
        component: "setting",
        data: pl
    })

    RokuMfgLegacy_SetColorTemp(saved.temp)
    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetDefaultWhiteBalance
' Set the white balance parameter for all available color temperatures.
'
' @args
' String colorstr - The white balance color, one of: red, green, blue.
' String paramstr - The white balance parameter, one of: offset, gain.
' Integer value - The value to set.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetDefaultWhiteBalance(colorstr as String, paramstr as String, value as Integer) as Object
' tfranklin: needs ported
    return RokuMfgLegacyError("needs ported")
end function

' @api GetColorTempValues
' Get all white balance parameter values for the specified color temperature, on the current source.
'
' @args
' String tempmode - The color temperature, one of: normal, cool, warm.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' roAssociativeArray values {
'     Integer redgain - The red gain value.
'     Integer bluegain - The blue gain value.
'     Integer greengain - The green gain value.
'     Integer redoffset - The red offset value.
'     Integer blueoffset - The blue offset value.
'     Integer greenoffset - The green offset value.
' }
function RokuMfgLegacy_GetColorTempValues(tempmode as String) as Object
    saved = RokuMfgLegacy_GetColorTemp()
    if 1 <> saved.valid then
        return RokuMfgLegacyError("failed to get current color temp")
    end if

    ret = RokuMfgLegacy_SetColorTemp(tempmode)
    if 1 <> ret.valid then
        return RokuMfgLegacyError("failed to set temp to " + tempmode)
    end if

    ret = RokuMfg().call("pq", {
        action: "get",
        component: "setting",
        data: RokuMfgState().get("PqKeyValues").ct_wb
    })

    RokuMfgLegacy_SetColorTemp(saved.temp)
    if RokuMfgCheckResponse(ret) then
        values = {}
        for each el in ret.data
            tok = el.split("_")
            values[lcase(tok[1]) + lcase(tok[2])] = val(ret.data[el], 10)
        end for

        return RokuMfgLegacySuccess({values: values})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api Get3DNR
' Get the TV's 3D noise reduction state.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer valueNR - The noise reduction state.
' Integer valueMpegNR - The MPEG noise reduction state.
function RokuMfgLegacy_Get3DNR() as Object
    ret = RokuMfg().call("pq", {
        action: "get",
        component: "setting",
        data: ["NoiseReduction", "MpegNR"]
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            valueNR: val(ret.data["NoiseReduction"], 10),
            valueMpegNR: val(ret.data["MpegNR"], 10)
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api Set3DNR
' Set the TV's 3D noise reduction state.
'
' @args
' Integer setting - The 3D noise reduction stae.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_Set3DNR(setting as Integer) as Object
    ret = RokuMfg().call("pq", {
        action: "set",
        component: "setting",
        data: {
            "NoiseReduction": setting,
            "MpegNR": setting
        }
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SwitchPictureMode
' Set the TV picture mode.
' THIS API IS DEPRECATED AS OF JANUARY 2016.
'
' @args
' String preset - The picture mode to switch to.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SwitchPictureMode(preset as String) as Object
    return RokuMfgLegacyError("deprecated")
end function

' @api GetPicturePresetMode
' Get the current picture preset mode.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String value - The current picture preset mode.
function RokuMfgLegacy_GetPicturePresetMode() as Object
    key = "Picture"
    ret = RokuMfg().call("pq", {
        action: "get",
        component: "setting",
        data: key
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({value: ret.data[key]})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetPicturePresetMode
' Set the picture preset mode.
'
' @args
' String preset - The picture preset mode to switch to.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetPicturePresetMode(preset as String) as Object
    key = "Picture"
    pl = {}
    pl[key] = preset
    ret = RokuMfg().call("pq", {
        action: "set",
        component: "setting",
        data: pl
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetPictModeDisplayParameter
' Get a particular parameter from the specified picture mode.
'
' @args
' String pictmodestr - The picture mode to retrieve parameter value from.
' String paramstr - The video parameter to retrieve, one of: brightness, contrast, saturation, hue, sharpness, backlight.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer value - The retrieved paramter value.
function RokuMfgLegacy_GetPictModeDisplayParameter(pictmodestr as String, paramstr as String) as Object
    saved = RokuMfgLegacy_GetPicturePresetMode()
    if 1 <> saved.valid then
        return RokuMfgLegacyError("failed to get current picture mode")
    end if

    ret = RokuMfgLegacy_SetPicturePresetMode(pictmodestr)
    if 1 <> ret.valid then
        return RokuMfgLegacyError("failed to set picture to " + pictmodestr)
    end if

    if lcase(paramstr) = "backlight" then
        paramstr = "Backlight_Level"
    else 
        paramstr = RokuMfgCapitalize(paramstr)
    end if

    ret = RokuMfg().call("pq", {
        action: "get",
        component: "setting",
        data: paramstr
    })

    RokuMfgLegacy_SetPicturePresetMode(saved.value)
    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({value: val(ret.data[paramstr], 10)})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetPictModeDisplayParameter
' Set the specified video parameter for a given picture mode.
'
' @args
' String pictmodestr - The picture mode to adjust.
' String paramstr - The video parameter to adjust, one of: brightness, contrast, saturation, hue, sharpness, backlight.
' Integer value - The value to be set.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetPictModeDisplayParameter(pictmodestr as String, paramstr as String, value as Integer) as Object
    saved = RokuMfgLegacy_GetPicturePresetMode()
    if 1 <> saved.valid then
        return RokuMfgLegacyError("failed to get current picture mode")
    end if

    ret = RokuMfgLegacy_SetPicturePresetMode(pictmodestr)
    if 1 <> ret.valid then
        return RokuMfgLegacyError("failed to set picture to " + pictmodestr)
    end if

    if lcase(paramstr) = "backlight" then
        paramstr = "Backlight_Level"
    else 
        paramstr = RokuMfgCapitalize(paramstr)
    end if

    pl = {}
    pl[paramstr] = value
    ret = RokuMfg().call("pq", {
        action: "set",
        component: "setting",
        data: pl
    })

    RokuMfgLegacy_SetPicturePresetMode(saved.value)
    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetDisplayParameterValue
' Get a display parameter value for the current picture mode.
'
' @args
' String paramstr - The video parameter to retrieve, one of: brightness, contrast, saturation, hue, sharpness, backlight.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer value - The current value of specified video parameter.
function RokuMfgLegacy_GetDisplayParameterValue(paramstr as String) as Object
    if lcase(paramstr) = "backlight" then
        paramstr = "Backlight_Level"
    else 
        paramstr = RokuMfgCapitalize(paramstr)
    end if

    ret = RokuMfg().call("pq", {
        action: "get",
        component: "setting",
        data: paramstr
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({value: val(ret.data[paramstr], 10)})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetDisplayParameterValue
' Set the specified video parameter for the current picture mode.
'
' @args
' String paramstr - The video parameter to adjust, one of: brightness, contrast, saturation, hue, sharpness, backlight.
' Integer value - The value to be set.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetDisplayParameterValue(paramstr as String, value as Integer) as Object
    if lcase(paramstr) = "backlight" then
        paramstr = "Backlight_Level"
    else 
        paramstr = RokuMfgCapitalize(paramstr)
    end if

    pl = {}
    pl[paramstr] = value
    ret = RokuMfg().call("pq", {
        action: "set",
        component: "setting",
        data: pl
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetDisplayAspectRatio
' Get the TV's current display aspect ratio.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String ratio - The TV's current aspect ratio, one of: auto, direct, normal, stretch, zoom.
function RokuMfgLegacy_GetDisplayAspectRatio() as Object
    key = "AspectRatioOverride"
    ret = RokuMfg().call("pq", {
        action: "get",
        component: "setting",
        data: key
    })

    if RokuMfgCheckResponse(ret) then
        for each el in RokuMfgState().get("PqKeyValues").aspect
            if ret.data[key] = el.pqvalue then
                return RokuMfgLegacySuccess({ratio: el.name})
            end if
        end for

        ret.header_.description = "failed to get aspect ratio"
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetDisplayAspectRatio
' Set the TV's display aspect ratio.
'
' @args
' String mode - The aspect ratio, one of: auto, direct, normal, stretch, zoom.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetDisplayAspectRatio(mode as String) as Object
    ratio = invalid
    for each el in RokuMfgState().get("PqKeyValues").aspect
        if lcase(mode) = lcase(el.name) then
            ratio = el.pqvalue
            exit for
        end if
    end for

    if RokuMfgIsInvalid(ratio) then
        return RokuMfgLegacyError("unsupported aspect ratio: " + mode)
    end if

    ret = RokuMfg().call("pq", {
        action: "set",
        component: "setting",
        data: {"AspectRatioOverride": ratio}
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetVideoCurve
' Get the specified video curve.
'
' A curve is defined by 8 points, each containing:
' <ul><li>The reference point corresponding to a UI value.</li>
' <li>The hardware value to actually use.</li></ul>
'
' @args
' String curve_sel - The name of the video curve to retrieve, one of: brightness, contrast, saturation, hue, sharpness.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer count - The number of reference points for the curve,  used to access the two sets of return values below.
' Integer refPt[N] - The Nth UI value as a reference point (x-axis) in the video curve calculation.
' Integer value[N] - The Nth internal (hardware) value (y-axis) to be used in the video curve calculation.
function RokuMfgLegacy_GetVideoCurve(curve_sel as String) as Object
    key = RokuMfgCapitalize(curve_sel) + "_Curve"
    ret = RokuMfg().call("pq", {
        action: "get",
        component: "setting",
        data: key
    })

    if RokuMfgCheckResponse(ret) then
        if not RokuMfgIsString(ret.data[key]) or 0 = len(ret.data[key]) then
            return RokuMfgLegacyError("failed to get curve")
        end if

        data = {}
        tok = ret.data[key].split(",")
        for idx = 0 to tok.count() - 1 step 2
            if "" = tok[idx] then
                ' Curve string can end in ',', so handle the trailing empty string
                exit for
            end if
            n = RokuMfgStrCast(idx / 2)
            data["refPt" + n] = val(tok[idx], 10)
            data["value" + n] = val(tok[idx + 1], 10)
        end for
        data.count = idx / 2

        return RokuMfgLegacySuccess(data)
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetVideoCurve
' Set the specified video curve.
'
' A curve is defined by 8 points, each containing:
' <ul><li>The reference point corresponding to a UI value.</li>
' <li>The hardware value to actually use.</li></ul>
'
' @args
' roAssociativeArray data {
'     String curve - The name of the video curve to set, one of: brightness, contrast, saturation, hue, sharpness.
'     Integer refPt[N] - The Nth UI value as a reference point (x-axis) in the video curve calculation.
'     Integer value[N] - The Nth internal (hardware) value (y-axis) to be used in the video curve calculation.
' }
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetVideoCurve(data as Object) as Object
    if RokuMfgIsInvalid(data.curve) then
        return RokuMfgLegacyError("expected data.curve as string")
    else if RokuMfgIsInvalid(data.count) then
        return RokuMfgLegacyError("expected data.count as integer")
    end if

    ' FIXME: should not be hardcoded
    curve = ""
    if 5 = data.count then
        for i = 0 to 4 step 1
            keystr = RokuMfgStrCast(i)
            refkey = "refPt" + keystr
            valkey = "value" + keystr
            if RokuMfgIsInvalid(data[refkey]) or RokuMfgIsInvalid(data[valkey]) then
                return RokuMfgLegacyError("missing point or value for index " + keystr)
            end if

            curve = curve + RokuMfgStrCast(data[refkey]) + "," + RokuMfgStrCast(data[valkey]) + ","
        end for

        curve = left(curve, len(curve) - 1)
    end if

    if "" <> curve then
        pl = {}
        pl[RokuMfgCapitalize(data.curve) + "_Curve"] = curve
        ret = RokuMfg().call("pq", {
            action: "set",
            component: "setting",
            data: pl
        })

        if not RokuMfgCheckResponse(ret) then
            return RokuMfgLegacyError(ret.header_.description)
        end if
    end if

    return RokuMfgLegacySuccess()
end function

' @api GetEditGlobalVideoPresets
' Get the current preset editing layer for PQ.
'
' This value determines how most PQ APIs apply their changes.
' <ul><li><b>Global layer</b> will affect all inputs on all picture modes.</li>
' <li><b>Input Type layer</b> will affect only a certain input (e.g. HDMI).</li>
' <li><b>Geek Squad layer</b> affects inputs per picture mode.  This layer is intended to be used to persist
' PQ settings on a per-TV basis.</li></ul>
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer value - The layer set as the default edit layer. 0 : input-type layer; 1 : global layer; 2 : geek squad layer
function RokuMfgLegacy_GetEditGlobalVideoPresets() as Object
    ret = RokuMfg().call("pq", {
        action: "get",
        component: "layer"
    })

    if RokuMfgCheckResponse(ret) then
        layer_map = {
            input: 0,
            global: 1,
            persistent: 2
        }

        value = layer_map[ret.data]
        if not RokuMfgIsInvalid(value) then
            return RokuMfgLegacySuccess({value: value})
        end if

        ret.header_.description = "failed to get edit level"
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetEditGlobalVideoPresets
' Choose the preset editing layer for PQ.
'
' This value determines how most PQ APIs apply their changes.
' <ul><li><b>Global layer</b> will affect all inputs on all picture modes.</li>
' <li><b>Input Type layer</b> will affect only a certain input (e.g. HDMI).</li>
' <li><b>Geek Squad layer</b> affects inputs per picture mode.  This layer is intended to be used to persist
' PQ settings on a per-TV basis.</li></ul>
'
' @args
' Integer layer - The layer set as the default edit layer. 0 : input-type layer; 1 : global layer; 2 : geek squad layer
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetEditGlobalVideoPresets(layer as Integer) as Object
    layer_map = {
        "0": "input",
        "1": "global",
        "2": "persistent"
    }

    level = layer_map[RokuMfgStrCast(layer)]
    if RokuMfgIsInvalid(level) then
        return RokuMfgLegacyError("invalid layer")
    end if

    ret = RokuMfg().call("pq", {
        action: "set",
        component: "layer",
        data: level
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetDynamicBacklightControl
' Get the dynamic backlight control mode and reference points.
'
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer enable - 0 : dynamic backlight disabled; 2: dynamic backlight Low; 3 : dynamic backlight High
' Integer level - The reference point for the current backlight level.
function RokuMfgLegacy_GetDynamicBacklightControl() as Object
    key_dynamic = "Backlight_Auto_Mode"
    key_level = "Backlight_Level"
    ret = RokuMfg().call("pq", {
        action: "get",
        component: "setting",
        data: [key_dynamic, key_level]
    })

    if RokuMfgCheckResponse(ret) then
        data = {enable: val(ret.data[key_dynamic], 10)}
        if "0" = ret.data[key_dynamic] then
            data.level = val(ret.data[key_level], 10)
        end if

        return RokuMfgLegacySuccess(data)
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetDynamicBacklightControl
' Enable/disable backlight control mode and set the backlight level.
'
' @args
' Integer enable - 0 : Disable dynamic backlight; 2 : Enable dynamic backlight (Low); 3 : Enable dynamic backlight (High).
' Integer level - The level of backlight to use. Has no effect if dynamic backlight mode is enabled.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetDynamicBacklightControl(enable as Integer, level as Integer) as Object
    key_dynamic = "Backlight_Auto_Mode"
    key_level = "Backlight_Level"
    pl = {}
    pl[key_dynamic] = enable
    if 0 = enable then
        pl[key_level] = level
    end if

    ret = RokuMfg().call("pq", {
        action: "set",
        component: "setting",
        data: pl
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetAllBacklightControls
' Get all backlight control parameters.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer Auto_Mode - 0 : Disable dynamic backlight; 2 : Enable dynamic backlight (Low); 3 : Enable dynamic backlight (High).
' Integer LowerLimit - The minimum drive level for dynamic backlight control.
' Integer UpperLimit - The maximum drive level for dynamic backlight control.
' Integer Level - The current backlight level.
' Integer AplPointStart - The minimum backlight value for average panel luminance (range: 0-255)
' Integer AplPointEnd - The maximum backlight value for average panel luminance (range: 0-255)
' Integer ContrastLimit - The limit for contrast compensation (range: 0-255)
function RokuMfgLegacy_GetAllBacklightControls() as Object
    prefix = "Backlight_"
    keys_external = ["Auto_Mode", "LowerLimit", "UpperLimit", "Level",
        "AplPointStart", "AplPointEnd", "ContrastLimit"]

    keys_internal = []
    for each key in keys_external
        keys_internal.push(prefix + key)
    end for

    ret = RokuMfg().call("pq", {
        action: "get",
        component: "setting",
        data: keys_internal
    })

    if RokuMfgCheckResponse(ret) then
        data = {}
        for each key in keys_external
            data[key] = val(ret.data[prefix + key], 10)
        end for

        return RokuMfgLegacySuccess(data)
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetAllBacklightControls
' Set all backlight control parameters.
'
' @args
' roAssociativeArray data {
'     Integer Auto_Mode - 0 : Disable dynamic backlight; 2 : Enable dynamic backlight (Low); 3 : Enable dynamic backlight (High).
'     Integer LowerLimit - The minimum drive level for dynamic backlight control.
'     Integer UpperLimit - The maximum drive level for dynamic backlight control.
'     Integer Level - The current backlight level.
'     Integer AplPointStart - The minimum backlight value for average panel luminance (range: 0-255)
'     Integer AplPointEnd - The maximum backlight value for average panel luminance (range: 0-255)
'     Integer ContrastLimit - The limit for contrast compensation (range: 0-255)
' }
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetAllBacklightControls(data as Object) as Object
    keys_external = ["Auto_Mode", "LowerLimit", "UpperLimit", "Level",
        "AplPointStart", "AplPointEnd", "ContrastLimit"]

    keys_internal = {}
    for each key in keys_external
        if not RokuMfgIsInvalid(data[key]) then
            keys_internal["Backlight_" + key] = data[key]
        end if
    end for

    if 0 = keys_internal.count() then
        return RokuMfgLegacyError("expected data to contain at least one of: " + RokuMfgStrCast(keys_external))
    end if

    ret = RokuMfg().call("pq", {
        action: "set",
        component: "setting",
        data: keys_internal
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetLocalDimmingControls
' Get local dimming parameters.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer Mode - The local dimming mode. 0 : off; 1 : low; 2 : medium; 3 : high
' Integer Low_LowLimit - The LowLimit value for local dimming's "Low" mode.
' Integer Med_LowLimit - The LowLimit value for local dimming's "Medium" mode.
' Integer High_LowLimit - The LowLimit value for local dimming's "High" mode.
function RokuMfgLegacy_GetLocalDimmingControls() as Object
    prefix = "LocalDimming_"
    keys_external = ["Mode", "Low_LowLimit", "Med_LowLimit", "High_LowLimit"]
    keys_internal = []
    for each key in keys_external
        keys_internal.push(prefix + key)
    end for

    ret = RokuMfg().call("pq", {
        action: "get",
        component: "setting",
        data: keys_internal
    })

    if RokuMfgCheckResponse(ret) then
        data = {}
        for each key in keys_external
            data[key] = val(ret.data[prefix + key], 10)
        end for

        return RokuMfgLegacySuccess(data)
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetLocalDimmingControls
' Set local dimming parameters.
'
' @args
' roAssociativeArray data {
'     Integer Mode - The local dimming mode. 0 : off; 1 : low; 2 : medium; 3 : high
'     Integer Low_LowLimit - The LowLimit value for local dimming's "Low" mode.
'     Integer Med_LowLimit - The LowLimit value for local dimming's "Medium" mode.
'     Integer High_LowLimit - The LowLimit value for local dimming's "High" mode.
' }
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetLocalDimmingControls(data as Object) as Object
    keys_external = ["Mode", "Low_LowLimit", "Med_LowLimit", "High_LowLimit"]
    keys_internal = {}
    for each key in keys_external
        if not RokuMfgIsInvalid(data[key]) then
            keys_internal["LocalDimming_" + key] = data[key]
        end if
    end for

    if 0 = keys_internal.count() then
        return RokuMfgLegacyError("expected data to contain at least one of: " + RokuMfgStrCast(keys_external))
    end if

    ret = RokuMfg().call("pq", {
        action: "set",
        component: "setting",
        data: keys_internal
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api ForceGammaWBOnGraphics
' Enable/Disable gamma and white balance on graphics plane.
'
' Normally, the graphics plane is unaffected by certain white balance and in some circumstances, gamma curves
' other than normal are suppressed. When enabled, this API will call underlying vendor-specific APIs to override
' this behavior.
'
' @args
' Integer enable - 0 : disable; 1 : enable
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_ForceGammaWBOnGraphics(enable as Integer) as Object
    ret = RokuMfg().call("gamma", {
        action: "enable",
        component: "force",
        data: RokuMfgBoolCast(enable)
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetMixDimmingControls
' Get mixed dimming control parameters.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer ADim_3DMax - Needs documentation.
' Integer ADim_Max - Needs documentation.
' Integer ADim_Min - Needs documentation.
' Integer PDim_Max - Needs documentation.
' Integer PDim_Min - Needs documentation.
' Integer PDim_Start - Needs documentation.
' Integer BL0 - Needs documentation.
' Integer BL2 - Needs documentation.
' Integer BL3 - Needs documentation.
' Integer X1 - Needs documentation.
' Integer X2 - Needs documentation.
function RokuMfgLegacy_GetMixDimmingControls() as Object
    prefix = "MixDimming_"
    keys_external = ["ADim_3DMax", "ADim_Max", "ADim_Min", "PDim_Max",
        "PDim_Min", "PDim_Start", "BL0", "BL2", "BL3", "X1", "X2"]

    keys_internal = []
    for each key in keys_external
        keys_internal.push(prefix + key)
    end for

    ret = RokuMfg().call("pq", {
        action: "get",
        component: "setting",
        data: keys_internal
    })

    if RokuMfgCheckResponse(ret) then
        data = {}
        for each key in keys_internal
            data[key] = ret.data[prefix + key]
        end for

        return RokuMfgLegacySuccess(data)
    end if

    return RokuMfgLegacyError(ret.header_.description) 
end function

' @api SetMixDimmingControls
' Set the mixed dimming controls.
'
' @args
' roAssociativeArray data {
'     Integer ADim_3DMax - Needs documentation.
'     Integer ADim_Max - Needs documentation.
'     Integer ADim_Min - Needs documentation.
'     Integer PDim_Max - Needs documentation.
'     Integer PDim_Min - Needs documentation.
'     Integer PDim_Start - Needs documentation.
'     Integer BL0 - Needs documentation.
'     Integer BL2 - Needs documentation.
'     Integer BL3 - Needs documentation.
'     Integer X1 - Needs documentation.
'     Integer X2 - Needs documentation.
' }
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetMixDimmingControls(data as Object) as Object
    keys_external = ["ADim_3DMax", "ADim_Max", "ADim_Min", "PDim_Max",
        "PDim_Min", "PDim_Start", "BL0", "BL2", "BL3", "X1", "X2"]
    keys_internal = {}
    for each key in keys_external
        if not RokuMfgIsInvalid(data[key]) then
            keys_internal["LocalDimming_" + key] = data[key]
        end if
    end for

    if 0 = keys_internal.count() then
        return RokuMfgLegacyError("expected data to contain at least one of: " + RokuMfgStrCast(keys_external))
    end if

    ret = RokuMfg().call("pq", {
        action: "set",
        component: "setting",
        data: keys_internal
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetEPSAutoColorSpaceControls
' Get control parameters for automatic color space.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer CS_Red_Red - Needs documentation.
' Integer CS_Red_Green - Needs documentation.
' Integer CS_Red_Blue - Needs documentation.
' Integer CS_Green_Red - Needs documentation.
' Integer CS_Green_Green - Needs documentation.
' Integer CS_Green_Blue - Needs documentation.
' Integer CS_Blue_Red - Needs documentation.
' Integer CS_Blue_Green - Needs documentation.
' Integer CS_Blue_Blue - Needs documentation.
' Integer CS_Yellow_Red - Needs documentation.
' Integer CS_Yellow_Green - Needs documentation.
' Integer CS_Yellow_Blue - Needs documentation.
' Integer CS_Cyan_Red - Needs documentation.
' Integer CS_Cyan_Green - Needs documentation.
' Integer CS_Cyan_Blue - Needs documentation.
' Integer CS_Magenta_Red - Needs documentation.
' Integer CS_Magenta_Green - Needs documentation.
' Integer CS_Magenta_Blue - Needs documentation.
function RokuMfgLegacy_GetEPSAutoColorSpaceControls() as Object
' tfranklin: needs ported
    return RokuMfgLegacyError("needs ported")
end function

' @api SetEPSAutoColorSpaceControls
' Set control parameters for automatic color space.
'
' @args
' roAssociativeArray data {
'     Integer CS_Red_Red - Needs documentation.
'     Integer CS_Red_Green - Needs documentation.
'     Integer CS_Red_Blue - Needs documentation.
'     Integer CS_Green_Red - Needs documentation.
'     Integer CS_Green_Green - Needs documentation.
'     Integer CS_Green_Blue - Needs documentation.
'     Integer CS_Blue_Red - Needs documentation.
'     Integer CS_Blue_Green - Needs documentation.
'     Integer CS_Blue_Blue - Needs documentation.
'     Integer CS_Yellow_Red - Needs documentation.
'     Integer CS_Yellow_Green - Needs documentation.
'     Integer CS_Yellow_Blue - Needs documentation.
'     Integer CS_Cyan_Red - Needs documentation.
'     Integer CS_Cyan_Green - Needs documentation.
'     Integer CS_Cyan_Blue - Needs documentation.
'     Integer CS_Magenta_Red - Needs documentation.
'     Integer CS_Magenta_Green - Needs documentation.
'     Integer CS_Magenta_Blue - Needs documentation.
'}
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetEPSAutoColorSpaceControls(data as Object) as Object
' tfranklin: needs ported
    return RokuMfgLegacyError("needs ported")
end function

' @api GetMEMCControls
' Get MEMC parameters.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer Mode - The current MEMC "smooth action" mode. 0 : off; 1 : low; 2 : medium; 3 : high
function RokuMfgLegacy_GetMEMCControls() as Object
    key = "MEMC_Mode"
    ret = RokuMfg().call("pq", {
        action: "get",
        component: "setting",
        data: key
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            Mode: val(ret.data[key], 10)
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetMEMCControls
' Set MEMC parameters.
'
' @args
' roAssociativeArray data {
'     Integer Mode - The MEMC "smooth action" mode. 0 : off; 1 : low; 2 : medium; 3 : high
' }
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetMEMCControls(data as Object) as Object
    if not RokuMfgIsAA(data) or not RokuMfgIsInt(data.mode) then
        return RokuMfgLegacyError("expected input data as roAssociativeArray containing mode as integer")
    else if 0 > data.mode or 3 < data.mode then
        return RokuMfgLegacyError("mode out of range")
    end if

    ret = RokuMfg().call("pq", {
        action: "set",
        component: "setting",
        data: {"MEMC_Mode": data.mode}
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api EnableDithering
' Enables/Disables dithering on the system.
' Dithering must be disabled for certain factory operations, like VCOM adjustment.  Dithering is
' enabled by default on most systems.
'
' @args
' Integer enable - 0 : disable dithering; 1 : enable dithering
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_EnableDithering(enable as Integer) as Object
' tfranklin: needs ported
    return RokuMfgLegacyError("needs ported")
end function

' @api GetGammaTableData
' Retrieves the currently loaded gamma table data
'
' @args
' String temp - color temperature to retrieve (Cool, Normal, Warm, VeryCool)
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - error message if not valid
' Array Red[256] - Red gamma data for 256 points
' Array Green[256] - Green gamma data for 256 points
' Array Blue[256] - Blue gamma data for 256 points
function RokuMfgLegacy_GetGammaTableData(temp as String) as Object
    ret = RokuMfg().call("gamma", {
        component: "custom",
        action: "get",
        temperature: RokuMfgCapitalize(temp)
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            red: ret.data.red,
            green: ret.data.green,
            blue: ret.data.blue
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetGammaTableData
' Sets the hardware gamma table data without modifying the ODM gamma data in persistent config
'
' @args
' AssociativeArray containing:
' Array Red[256] - Red gamma data, 256 points
' Array Green[256] - Green gamma data, 256 points
' Array Blue[256] - Blue gamma data, 256 points
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - error message if not valid
function RokuMfgLegacy_SetGammaTableData(data as Object) as Object
    if not RokuMfgIsAA(data) or not RokuMfgIsArray(data.red) or not RokuMfgIsArray(data.green) or not RokuMfgIsArray(data.blue) then
        return RokuMfgLegacyError("expected data as associative array containing red, green, and blue arrays")
    end if

    ret = RokuMfg().call("gamma", {
        component: "hardware",
        action: "set",
        data: data
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api EnablePanelGamma(int enable)
' Sets the gamma to be applied to the entire panel
'
' @args
' Integer enable - 1 : enable gamma application, 0 : disable gamma application
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - error message if not valid
function RokuMfgLegacy_EnablePanelGamma(enable as Integer) as Object
    ret = RokuMfg().call("gamma", {
        action: "enable",
        component: "panel",
        data: RokuMfgBoolCast(enable)
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api EnableGamma(int enable)
' Sets the gamma to be applied to the VOP test pattern
'
' @args
' Integer enable - 1 : enable gamma application, 0 : disable gamma application
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - error message if not valid
function RokuMfgLegacy_EnableGamma(enable as Integer) as Object
    ret = RokuMfg().call("gamma", {
        action: "enable",
        component: "gamma",
        data: RokuMfgBoolCast(enable)
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api EnablePreGainOffset(int gain, int offset)
' Sets the pre-gain and/or offset to be applied to the VOP test pattern
'
' @args
' Integer gain - 1 : enable pre-gain, 0 : disable pre-gain
' Integer offset - 1 : enable pre-offset, 0 : disable pre-offset
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - error message if not valid
function RokuMfgLegacy_EnablePreGainOffset(gain as Integer, offset as Integer) as Object
    ret = RokuMfg().call("gamma", {
        action: "enable",
        component: "pregainoffset",
        gain: RokuMfgBoolCast(gain)
        offset: RokuMfgBoolCast(offset)
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetPreGammaGainOffset(char const * color, char const * which, int value)
' Sets the pre-gamma gain applied to the VOP test pattern
'
' @args
' String color - R/G/B/A for Red/Green/Blue/All
' String which - G/O for Gain/Offset
' Integer value - value to set the gain to
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - error message if not valid
function RokuMfgLegacy_SetPreGammaGainOffset(color as String, which as String, value as Integer) as Object
    color_map = {
        "R": "red",
        "G": "green",
        "B": "blue",
        "A": "all"
    }

    which_map = {
        "G": "gain",
        "O": "offset"
    }

    if RokuMfgIsInvalid(which_map[which]) then
        return RokuMfgLegacyError("invalid which paramter")
    else if RokuMfgIsInvalid(color_map[color]) then
        return RokuMfgLegacyError("invalid color paramter")
    end if

    ret = RokuMfg().call("gamma", {
        action: "set",
        component: "pregainoffset",
        color: color_map[color],
        field: which_map[which],
        data: value
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetDRMode
' Get the current Dynamic Range mode (HDR or SDR).
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String drmode - The current Dynamic Range mode (HDR or SDR)
function RokuMfgLegacy_GetDRMode() as Object
    ret = RokuMfg().call("video", {
        action: "get",
        component: "mode"
    })

    if RokuMfgCheckResponse(ret) then
        if "SDR" <> ret.data.dr_mode then
            ret.data.dr_mode = "HDR"
        end if

        return RokuMfgLegacySuccess({drmode: ret.data.dr_mode})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function
