'*********************************************************************
'** (c) 2018 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************
' Roku_MFG_Legacy_Networking.brs
' Wrappers for the legacy "networking" APIs
Library "state/Roku_MFG_State.brs"

' @api CheckNetworkStatus
' Get the system's network status.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String connection - Active connection, one of: wifi, wired
' Dynamic name - If connection is wifi, then this is a String with the access point's name.
' Dynamic strength - If connection is wifi, then this is an Integer with the access point's relative signal strength.
' Dynamic encryption - If connection is wifi, then this is a String with the access point's security mechanism, one of: WEP, WPA
' Dynamic cnx_status - If connection is wifi, then this is a String with the access point's connection status, one of: Disconnected, Reconnecting, Connected.
function RokuMfgLegacy_CheckNetworkStatus() as Object
    ret = RokuMfg().call("network", {
        component: "network",
        action: "check"
    })

    if RokuMfgCheckResponse(ret) then
        data = {}
        data.connection = ret.data.type
        if "wifi" = ret.data.type then
            data.name = ret.data.name
            data.strength = ret.data.strength
            data.encryption = ret.data.encryption
            data.cnx_status = ret.data.status
        end if

        return RokuMfgLegacySuccess(data)
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api CheckEthLinkStatus
' Check if a cable is plugged into the Ethernet port.
' Note that this one only applies to TVs with an Ethernet port.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String status - Link status, one of: active, inactive
function RokuMfgLegacy_CheckEthLinkStatus() as Object
    ret = RokuMfg().call("network", {
        component: "ethernet",
        action: "checklink"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            status: ret.data
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetMacAddressString
' Get the system's MAC address.
' Note that the MAC address will remain unchanged on some platforms until a reboot is issued.  In
' these cases, refer to the pcaddress value.
'
' @args
' String netstr - The network interface to retrieve the MAC address from, one of: wired, wifi
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String address - The MAC address
' String pcaddress - The MAC address in persistent storage.
function RokuMfgLegacy_GetMacAddressString(netstr as String) as Object
    ret = RokuMfg().call("network", {
        component: "mac",
        action: "get",
        data: netstr
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            address: ucase(ret.data.address),
            pcaddress: ucase(ret.data.pcaddress)
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetMacAddressString
' Set the system's MAC address.
' Note that the system requires a reboot before the new address will take effect.
'
' @args
' String netstr - The network interface to modify the MAC address for, one of: wired, wifi.
' String addrstr - The MAC address string, in the format XX:XX:XX:XX:XX:XX.
' Integer setWiredMacNow - If the interface is wired and this is set to 1, apply the new MAC immediately.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetMacAddressString(netstr as String, addrstr as String, setWiredMacNow as Integer) as Object
    ' setWiredMacNow is a required parameter in the function signature.
    ' Due to security policy, we do not have 'root' privilege necessary to
    ' apply the MAC address without a reboot, therefore we ignore this value.
    ret = RokuMfg().call("network", {
        component: "mac",
        action: "set",
        iface: netstr,
        data: addrstr
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetIpAddressString
' Get the system's IP address.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String address - The IP address currently associated with the system, or 0.0.0.0 if not connected.
function RokuMfgLegacy_GetIpAddressString() as Object
    ret = RokuMfg().call("network", {
        component: "ip",
        action: "get"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            address: ret.data.ip
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetRouterIpAddressString
' Get the IP address of the router that the system is currently connected to.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' String address - The router's IP address.
function RokuMfgLegacy_GetRouterIpAddressString() as Object
    ret = RokuMfg().call("network", {
        component: "ip",
        action: "get"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess({
            address: ret.data.gateway
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api RenewIpAddress
' Renew wired IP address through DHCP server.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_RenewIpAddress() as Object
    ret = RokuMfg().call("network", {
        component: "ip",
        action: "renew"
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api DoPing
' Ping the specified host.
'
' @args
' String hostchar - The host IP address to ping.
' Integer timeout - The maximum duration that the system will attempt to ping for.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_DoPing(hostchar as String, timeout as Integer) as Object
    ret = RokuMfg().call("network", {
        component: "network",
        action: "ping",
        host: hostchar,
        timeout: timeout
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api LoadWifiDriver
' Load the WiFi manufacturing or production driver.
'
' @args
' Integer mfg - 1 : load manufacturing driver; 0 : load production driver
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_LoadWifiDriver(mfg as Integer) as Object
    if 0 = mfg then
        arg = "prod"
    else
        arg = "mfg"
    end if

    ret = RokuMfg().call("network", {
        component: "wifi",
        action: "loaddriver",
        data: arg
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api CheckWifiDevice
' Returns whether the wifi device is present or not.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer present - 0 : Not Present; 1 : Present
function RokuMfgLegacy_CheckWifiDevice() as Object
    ret = RokuMfg().call("network", {
        component: "wifi",
        action: "device",
        block: true
    })

    if RokuMfgCheckResponse(ret) then
        if true = ret.data.present then
            present = 1
        else
            present = 0
        end if

        return RokuMfgLegacySuccess({
            present: present
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api CheckWifiStatus
' Get the system's WiFi status.
' Additionally, this command performs a WiFi scan, returning all access points found.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer count - How many access points are available, in total.
' String Name[N] - SSID of access point N.
' Integer Signal[N] - Signal strength for access point N.
' Integer RSSI[N] - RSSI for access point N.
' String Encrypt[N] - Security status for access point N, one of: OPEN, SECURE
' Integer Frequency[N] - Frequency in MHz
function RokuMfgLegacy_CheckWifiStatus() as Object
    ret = RokuMfg().call("network", {
        component: "wifi",
        action: "status",
        block: true
    })

    if RokuMfgCheckResponse(ret) then
        data = {
            count: ret.data.count()
        }

        keys = {
            "Name": "name",
            "Signal": "signal",
            "RSSI": "rssi",
            "Encrypt": "encryption",
            "Frequency": "frequency"
        }

        for i=0 to ret.data.count() - 1
            for each key in keys
                data[key + RokuMfgStrCast(i)] = ret.data[i][keys[key]]
            end for
        end for

        return RokuMfgLegacySuccess(data)
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetWifiRssi
' Get the WiFi RSSI of the currently connected AP, for both antennas.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer RSSI1 - Antenna 1's RSSI, a range from 0 to -92dBm.
' Integer RSSI2 - Antenna 2's RSSI, a range from 0 to -92dBm.
function RokuMfgLegacy_GetWifiRssi() as Object
    ret = RokuMfg().call("network", {
        component: "wifi",
        action: "rssi"
    })

    ' The call returns lcase strings, but the legacy interface expects uppercase
    ' This API can return a failed response but still have valid data,
    ' so always convert the keys to ucase
    upper_data = {}
    if RokuMfgIsAA(ret.data) then
        for each key in ret.data
            upper_data[Ucase(key)] = ret.data[key]
        end for
    endif

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess(upper_data)
    end if

    leg_ret = RokuMfgLegacyError(ret.header_.description)
    return RokuMfgMerge(leg_ret, upper_data)
end function

' @api GetWifiStrength
' Get the WiFi signal strength.
'
' @args
' String ssid - The SSID of the access point to retrieve signal strength for.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
' Integer strength - The relative strength of the access point, on a scale of 0-100.
function RokuMfgLegacy_GetWifiStrength(ssid as String) as Object
    ret = RokuMfg().call("network", {
        component: "wifi",
        action: "status",
        block: true
    })

    if RokuMfgCheckResponse(ret) then
        for each entry in ret.data
            if ssid = entry["name"] then
                return RokuMfgLegacySuccess({
                    strength: entry["strength"]
                })
            end if
        end for

        return RokuMfgLegacyError("SSID not found")
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api SetWifiDeviceStateOpen
' Enable or disable the system's WiFi.
'
' @args
' Integer dev_open - 0 : disable WiFi; 1 : enable WiFi
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetWifiDeviceStateOpen(dev_open as Integer) as Object
    ret = RokuMfg().call("network", {
        component: "wifi",
        action: "config",
        enable: RokuMfgBoolCast(dev_open),
        block: true
    })
end function

' @api ConfigWifi
' Enable/Disable the system's WiFi and associate with a given SSID.
' This API blocks the BrightScript app until it completes.
'
' @args
' Integer enable - 0 : disable WiFi; 1 : enable WiFi
' String ssid - SSID for an access point.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_ConfigWifi(enable as Integer, ssid as String) as Object
    ret = RokuMfg().call("network", {
        component: "wifi",
        action: "config",
        enable: RokuMfgBoolCast(enable),
        ssid: ssid,
        block: true
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api DisableWifi
' Disable the system's WiFi regardless
' This API blocks the BrightScript app until it completes.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_DisableWifi() as Object
    ret = RokuMfg().call("network", {
        component: "wifi",
        action: "config",
        enable: false,
        block: true
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api StartConfigWifi
' Enable/disable the system's WiFi and associates with a given SSID.
' This API performs its tasks in the background, instead of blocking like ConfigWifi.
' If a previous call to StartConfigWifi or StartWifiReconnecting is still completing,
' this API will immediately return an error.
'
' @args
' Integer enable - 0 : disable WiFi; 1 : enable WiFi
' String ssid - SSID of access point.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_StartConfigWifi(enable as Integer, ssid as String) as Object
    ret = RokuMfg().call("network", {
        component: "wifi",
        action: "config",
        enable: RokuMfgBoolCast(enable),
        ssid: ssid,
        block: false
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api StartWifiReconnect
' Initiates a WiFi reconnection.
'
' Starts a Wifi reconnection.
' It is recommended that this function is executed 15 to 30 seconds before starting Wifi test(GetWifiRssi()).
' This function starts the wifi reconnection process in the background.
' If a previous call to StartConfigWifi or StartWifiReconnecting is still completing,
' this API will immediately return an error.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_StartWifiReconnect() as Object
    ret = RokuMfg().call("network", {
        component: "wifi",
        action: "reconnect",
        block: false
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api LoadBSAServer
' Load the BSA server for bluetooth manufacturing tests.
'
' @args
' Integer start - 0 : stop BSA server; 1 : start BSA server
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_LoadBSAServer(start as Integer) as Object
    ret = RokuMfg().call("network", {
        component: "bluetooth",
        action: "bsaserver",
        enable: RokuMfgBoolCast(start)
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api StartCheckWifiDevicePresent
' Starts 'CheckWifiDevice'. UpdateWifiDevicePresent is used to get results.
' This is part of the non-blocking version of CheckWifiDevice()
'
' @return
' Integer valid - 0 : not valid; 1 : valid
function RokuMfgLegacy_StartCheckWifiDevicePresent() as Object
    ret = RokuMfg().call("network", {
        component: "wifi",
        action: "device",
        block: true
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api UpdateWifiDevicePresent
' Returns results from StartCheckWifiDevicePresent.
' This is part of the non-blocking version of CheckWifiDevice()
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition
' Integer present - 0 : Device Not Present; 1 : Device Present
function RokuMfgLegacy_UpdateWifiDevicePresent() as Object
    present = RokuMfgState().get("WifiDevicePresent")
    if invalid <> present then
        RokuMfgState().set("WifiDevicePresent", invalid)
        if true = present then
            value = 1
        else
            value = 0
        end if

        return RokuMfgLegacySuccess({
            present: value
        })
    end if

    return RokuMfgLegacyError("device status not available")
end function

' @api StartCheckWifiStatus
' Starts 'CheckWifiStatus'. UpdateWifiStatus is used to get results.
' This is part of the non-blocking version of CheckWifiStatus()
'
' @return
' Integer valid - 0 : not valid; 1 : valid
function RokuMfgLegacy_StartCheckWifiStatus() as Object
    ret = RokuMfg().call("network", {
        component: "wifi",
        action: "status",
        block: false
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api UpdateWifiStatus
' Returns results from StartCheckWifiStatus.
' This is part of the non-blocking version of CheckWifiStatus()
'
' Note: repsonse data is retreived via RokuMfg().checkEvent(). Once the data
' has been returned from this call, it is no longer available until
' StartCheckWifiStatus is executed again.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition
' Integer count - Identifies how many access points in total.
' String Name[N] - SSID of access point N.
' Integer Signal[N] - Signal strength for access point N.
' Integer RSSI[N] - RSSI for access point N.
' String Encrypt[N] - Security status for access point N, one of: OPEN, SECURE
' Integer Frequency[N] - Frequency in MHz
function RokuMfgLegacy_UpdateWifiStatus() as Object
    ap_list = RokuMfgState().get("WifiAPList")
    if invalid <> ap_list then
        RokuMfgState().set("WifiAPList", invalid)
        data = {}
        keys = {
            "Name": "name",
            "Signal": "signal",
            "RSSI": "rssi",
            "Encrypt": "encryption",
            "Frequency": "frequency"
        }

        i = 0
        while i <= ap_list.count() - 1
            if 0 = ap_list[i]["frequency"] then
                ' If the wifi state is unstable, we sometimes get entries that
                ' look valid but have frequency = 0. Prune these entries.
                ap_list.delete(i)
            else
                for each key in keys
                    data[key + RokuMfgStrCast(i)] = ap_list[i][keys[key]]
                end for
                i = i + 1
            end if
        end while

        data.count = ap_list.count()
        return RokuMfgLegacySuccess(data)
    end if

    return RokuMfgLegacyError("wifi status not available")
end function

' @api SetBootWifi
' Sets the state of wifi(enable/disabled) on the next boot cycle.
' Note that the system requires a reboot before the new wifi state will take effect.
'
' @args
' Integer enable - 0 : Wifi disabled; 1 : Wifi enabled
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_SetBootWifi(enable as Integer) as Object
    ret = RokuMfg().call("network", {
        component: "wifi",
        action: "setboot",
        enable: RokuMfgBoolCast(enable)
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api GetBootWifi
' Gets the state of wifi(enable/disabled).
' Returns two wifi states, state of wifi on next boot cycle and current wifi.
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' Integer next_boot_wifi - 0 : disabled; 1 : enabled
' Integer current_wifi - 0 : disabled; 1 : enabled
' String error - Description of error condition.
function RokuMfgLegacy_GetBootWifi() as Object
    ret = RokuMfg().call("network", {
        component: "wifi",
        action: "getboot"
    })

    if RokuMfgCheckResponse(ret) then
        if true = ret.data.next then
            next_boot = 1
        else
            next_boot = 0
        end if

        if true = ret.data.current then
            current = 1
        else
            current = 0
        end if

        return RokuMfgLegacySuccess({
            next_boot_wifi: next_boot,
            current_wifi: current
        })
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api ReadWifiEfuse
' Reads the Wifi Efuse settings
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition
' String settingsXX - Setting for line <XX>, 0-based
' Integer count - number of lines
function RokuMfgLegacy_ReadWifiEfuse() as Object
    ret = RokuMfg().call("network", {
        component: "efuse",
        action: "read"
    })

    if RokuMfgCheckResponse(ret) then
        data = {
            count: ret.data.count()
        }

        for i=0 to ret.data.count() - 1
            data["settings" + RokuMfgStrCast(i)] = ret.data[i]
        end for

        return RokuMfgLegacySuccess(data)
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api WriteWifiEfuse
' Writes the Wifi Efuse settings
'
' WARNING: this should be done sparingly. EFUSE should be written no more than 10 times.
'
' @args
' String map_file - the map file to use
' String mask_file - the mask file to use
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition
function RokuMfgLegacy_WriteWifiEfuse(map_file as String, mask_file as String) as Object
    ret = RokuMfg().call("network", {
        component: "efuse",
        action: "write",
        map: map_file,
        mask: mask_file
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api ActivateBluetooth
' Enable or disable bluetooth.
'
' @args
' Integer activate - 0 : disable bluetooth; 1 : enable bluetooth
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_ActivateBluetooth(activate as Integer) as Object
    ret = RokuMfg().call("network", {
        component: "bluetooth",
        action: "activate",
        enable: RokuMfgBoolCast(activate)
    })

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

