'*********************************************************************
'** (c) 2018 Roku, Inc.  All content herein is protected by U.S.
'** copyright and other applicable intellectual property laws and may
'** not be copied without the express permission of Roku, Inc., which
'** reserves all rights.  Reuse of any of this content for any purpose
'** without the permission of Roku, Inc. is strictly prohibited.
'*********************************************************************
' Roku_MFG_Legacy_I2C.brs
' Wrappers for the legacy "i2c" APIs

' @api I2CRead
' Read data from the I2C bus.
'
' @args
' roAssociativeArray input {
'     Integer port       - The I2C bus to read from.
'     Integer numbytes   - Number of bytes to read from the sink device.
'     Integer devaddress - The device address on the bus to read from.
'     Integer subaddress - (optional) the slave address to access on the sink device.
' }
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' roByteArray buffer - buffer of numbytes read.
' String error - Description of error condition.
function RokuMfgLegacy_I2CRead(input as Object) as Object
    pl = RokuMfgDeepCopy(input)
    pl.action = "read"
    ret = RokuMfg().call("i2c", pl)

    if RokuMfgCheckResponse(ret) then
        ba = createObject("roByteArray")
        ba.fromBase64String(ret.data)
        return RokuMfgLegacySuccess({buffer: ba})
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

' @api I2CWrite
' Write data to the I2C bus.
' Note: if data is not provided, this interface will write the address to the I2C bus.
'
' @args
' roAssociativeArray input {
'     roByteArray data   - (optional) array of data to write. If not provided, will just write devaddress.
'     Integer port       - The I2C bus to write to.
'     Integer devaddress - The device address on the bus to write to.
'     Integer subaddress - (optional) The device subaddress to write to.
' }
'
' @return
' Integer valid - 0 : not valid; 1 : valid
' String error - Description of error condition.
function RokuMfgLegacy_I2CWrite(input as Object) as Object
    pl = {}
    pl.port = input.port
    pl.devaddress = input.devaddress
    if not RokuMfgIsInvalid(input.subaddress) then
        pl.subaddress = input.subaddress
    end if

    if not RokuMfgIsInvalid(input.data) then
        if not RokuMfgHasInterface(input.data, "ifByteArray") then
            return RokuMfgLegacyError("expected input.data as roByteArray")
        else
            pl.data = input.data.toBase64String()
        end if
    end if

    pl.action = "write"
    ret = RokuMfg().call("i2c", pl)

    if RokuMfgCheckResponse(ret) then
        return RokuMfgLegacySuccess()
    end if

    return RokuMfgLegacyError(ret.header_.description)
end function

