' ********** Copyright 2019 Roku Corp. All Rights Reserved. **********

function ShowAudioPlayerScreen(contentList as Object, selectedItem as Object, preload = false as Boolean) as Object
    ' Make sure to disable BB prior to showing audio screen
    buttonBar = m.top.GetScene().buttonBar
    if buttonBar <> invalid and buttonBar.visible = true
        buttonBar.visible = false
    end if

    controller = GetBaseViewController("MediaView")
    controller.Append({
        isDebug: false  'set to true to enable verbose debug prints

        initIndex: 0
        contentList: contentList
        isFirstShuffle: GetAudioShuffleSettings()
        selectedItem: selectedItem
        scene: m.top

        playbackPosition: 0

        SetUpScreen: AudioPlayerScreen_SetUpScreen
        CleanUp: AudioPlayerScreen_CleanUp
        GetOptionKeys: AudioPlayerScreen_GetOptionKeys
        IsPlaylistTooLarge: AudioPlayerScreen_IsPlaylistTooLarge
        InitBusyLabel: AudioPlayerScreen_InitBusyLabel

        ' observers
        OnContentChanged: AudioPlayerScreen_OnContentChanged
        OnButtonSelected: AudioPlayerScreen_OnButtonSelected
        OnStateChanged: AudioPlayerScreen_OnStateChanged
        OnCurrentItemChanged: AudioPlayerScreen_OnCurrentItemChanged
        OnPositionChanged: AudioPlayerScreen_OnPositionChanged
    })
    controller.Observe(controller.view, "content", "OnContentChanged")
    controller.Observe(controller.view, "state", "OnStateChanged")
    controller.Observe(controller.view, "currentItem", "OnCurrentItemChanged")
    controller.Observe(controller.view, "position", "OnPositionChanged")
    controller.Observe(controller.view, "buttonSelected", "OnButtonSelected")
    controller.SetUpScreen(preload)
    ShowNewView(controller.view)
    return controller
end function

sub AudioPlayerScreen_SetUpScreen(preload as Boolean)
    m.view.theme = {
        OverhangTitle: Tr("Media Player")
    }
    m.view.isContentList = true
    m.view.mode = "audio"
    
    ' If playlist is too large, it causes a long delay,
    ' so we should show label as visual feedback
    if m.isPlaylistTooLarge() then m.InitBusyLabel()

    content = CreateObject("roSGNode", "ContentNode")
    config = {
        HandlerConfigMedia: {
            name: "AudioPlayerScreenCH"
            fields: {
                contentList: m.contentList.clone(true)
                selectedItem: m.selectedItem
            }
        }
    }
    content.Update(config, true)
    m.view.buttons = m.GetOptionKeys()
    m.view.content = content
    m.view.repeatAll = GetAudioRepeatSettings()
    m.view.disableScreenSaver = GetScreensaverSetting()
    if preload
        m.view.preloadContent = true
    else
        m.view.control = "play"
    end if
    if m.selectedItem <> invalid and m.selectedItem.hasField("playStart")
        m.view.seek = m.selectedItem.playStart
    end if
end sub

function AudioPlayerScreen_IsPlaylistTooLarge()
    maxPlaylistSize = 100
    currentPlaylistSize = 0
    if m.contentList.GetChild(0) <> invalid and m.contentList.GetChild(0).GetChild(0) <> invalid
        for each row in m.contentList.GetChildren(-1, 0)
            currentPlaylistSize = currentPlaylistSize + row.GetChildCount() 
            if currentPlaylistSize > maxPlaylistSize then return true
        end for
    end if
    return false   
end function

sub AudioPlayerScreen_InitBusyLabel()
    m.busyLabel = CreateObject("roSGNode", "Label")
    m.busyLabel.text = Tr("Retrieving...")
    m.busyLabel.font = "font:MediumBoldSystemFont"
    m.busyLabel.width = 1280
    m.busyLabel.translation = [0, 630]
    m.busyLabel.horizAlign = "center"

    m.view.AppendChild(m.busyLabel)
end sub

function AudioPlayerScreen_GetOptionKeys()
    content = CreateObject("roSGNode", "ContentNode")
    repeatItem = {
        id:"Repeat All"
        title: Tr("Repeat all (off)")
    }
    screensaverItem = {
        id:"Screensaver"
        title: Tr("Screensaver (on)")
    }
    shuffleItem = {
        id:"Shuffle"
        title: Tr("Shuffle (off)")
    }
    start = {
        id:"StartFromBeginning"
        title: Tr("Start from beginning")
    }
    if GetScreensaverSetting()
        ' this seems backward because the setting controls whether the screensaver is disabled
        screensaverItem.title = Tr("Screensaver (off)")
    end if
    if GetAudioRepeatSettings()
        repeatItem.title = Tr("Repeat all (on)")
    end if
    if GetAudioShuffleSettings()
        shuffleItem.title = Tr("Shuffle (on)")
    end if
    content.Update({
        children:[start, repeatItem, shuffleItem, screensaverItem]
    },true)
    return content
end function

'observers
sub AudioPlayerScreen_OnContentChanged(audioScreen as Object, content as Object)
    if content.hasField("jumpToItem")
       if content.jumpToItem > 0 then m.initIndex = content.jumpToItem
        audioScreen.jumpToItem = content.jumpToItem
        content.removeField("jumpToItem")
    else if content.HandlerConfigMedia = invalid and content.getChildCount() > 0
        if m.isFirstShuffle
            audioScreen.shuffle = GetAudioShuffleSettings()
            m.isFirstShuffle = false
        end if
    end if
end sub

sub AudioPlayerScreen_OnButtonSelected(mediaView as Object, itemSelected as Integer)
    if itemSelected >= 0
        content = CreateObject("roSGNode", "ContentNode")
        content = mediaView.buttons.clone(true)
        buttonSelected = mediaView.buttons.getChild(itemSelected)
        if buttonSelected.id = "Repeat All"
            if GetAudioRepeatSettings()
                content.getChild(itemSelected).title = Tr("Repeat all (off)")
            else
                content.getChild(itemSelected).title = Tr("Repeat all (on)")
            end if
            SetAudioRepeatSettings(not GetAudioRepeatSettings())
            m.view.repeatAll = GetAudioRepeatSettings()
        else if buttonSelected.id = "Shuffle"
            if GetAudioShuffleSettings()
                content.getChild(itemSelected).title = Tr("Shuffle (off)")
            else
                content.getChild(itemSelected).title = Tr("Shuffle (on)")
            end if
            SetAudioShuffleSettings(not GetAudioShuffleSettings())
            m.view.shuffle = GetAudioShuffleSettings()
        else if buttonSelected.id = "Screensaver"
            if GetScreensaverSetting()
                content.getChild(itemSelected).title = Tr("Screensaver (on)")
            else
                content.getChild(itemSelected).title = Tr("Screensaver (off)")
            end if
            SetScreensaverSetting(not GetScreensaverSetting())
            m.view.disableScreenSaver = GetScreensaverSetting()
        else if buttonSelected.id = "StartFromBeginning"
            m.view.control = "pause"
            m.view.seek = 0
            m.view.control = "resume"
        end if
        mediaView.buttons = content
    end if
end sub

sub AudioPlayerScreen_OnStateChanged(mediaView as Object, state as String)
    if state = "buffering"
        if m.busyLabel <> invalid and m.busyLabel.visible = true
            m.busyLabel.visible = false
            mediaView.RemoveChild(m.busyLabel)
            m.busyLabel = invalid
        end if
    end if
    if state = "finished"
        ' playback position is within 5 last seconds of playback?
        if m.playbackPosition >= mediaView.duration - 5
            ' yes, clear bookmark by writing zero
            WriteAudioBookmark(m.selectedItem, 0, m.isDebug)
        end if

        ' reset playback position
        m.playbackPosition = 0
    end if
end sub

sub AudioPlayerScreen_OnCurrentItemChanged(mediaView as Object, currentItem as Object)
    currentItem.Update({isTriggered: true}, true)
    if m.selectedItem <> invalid and currentItem <> invalid and currentItem.guid <> m.selectedItem.guid
        ' switched to new piece of content, update cached selected item
        m.selectedItem = currentItem
    end if
end sub

sub AudioPlayerScreen_OnPositionChanged(mediaView as Object, position as Integer)
    ' update playback position
    m.playbackPosition = position

    if m.initIndex <> mediaView.currentIndex
        range = mediaView.currentIndex - m.initIndex
        if range <= 0 then range = 1
        for each item in mediaView.content.GetChildren(range, m.initIndex)
            if not item.hasField("isTriggered")
                WriteAudioBookmark(item, 0, m.isDebug)
                item.Update({isTriggered: true}, true)
            end if
        end for
        m.initIndex = mediaView.currentIndex
    end if

    if m.selectedItem <> invalid
        ' get last bookmarked position for the current piece of content
        lastBookmark = m.selectedItem.playStart
        if lastBookmark = invalid
            lastBookmark = 0
        end if

        ' current position is within 5 last seconds of playback?
        if position >= mediaView.duration - 5
            ' yes, clear bookmark by writing zero
            WriteAudioBookmark(m.selectedItem, 0, m.isDebug)
        else if Abs(position - lastBookmark) >= 10
            ' current position is 10 seconds(or more) greater or less than  last bookmarked one,
            ' need to update bookmark
            WriteAudioBookmark(m.selectedItem, position, m.isDebug)
        end if
    end if
end sub

' will be called when view was closed
sub AudioPlayerScreen_CleanUp()
    m.isFirstShuffle = true
    m.scene.buttonBar.visible = false
    m.contentList = invalid
    m.contentItem = invalid
    m.initIndex = invalid
end sub
