'**  Copyright (c) 2019 Roku Inc. All Rights Reserved.

' forked off from usb.brs

' return object with autorun fields, or invalid if none
function USBUtilReadAutoRunInf(m, volume)
    autoRunInfPath = volume + "/autorun.inf"
    if not m.rofs.Exists(autoRunInfPath)
        return invalid
    endif

    autoRunInf = ReadAsciiFile(autoRunInfPath)
    autoRunInfLines = autoRunInf.Split(Chr(10))

    autoRun = {}

    inAutoRunSection = false
    for each line in autoRunInfLines
        line = line.Trim()
        if line.Instr("[") = 0
            if LCase(line) = "[autorun]"
                inAutoRunSection = true
            else
                inAutoRunSection = false
            endif
        else if line <> ""
            if inAutoRunSection
                kv = line.Split("=")
                if kv.Count() = 2
                    autoRun[ kv[0] ] = kv[1]
                endif
            end if
        end if
    end for

    return autoRun
end function

function BAGetUInt16(ba_, pos_)
    return ba_[pos_] + (ba_[pos_ + 1] << 8)
'    return ba_[pos_] + (ba_[pos_ + 1] * 256)   ' fw 5.6
end function

function BAGetInt32(ba_, pos_)
    return ba_[pos_] + (ba_[pos_ + 1] << 8) + (ba_[pos_ + 2] << 16) + (ba_[pos_ + 3] << 24)
'    return ba_[pos_] + (ba_[pos_ + 1] * 256) + (ba_[pos_ + 2] * 256 * 256) + (ba_[pos_ + 3] * 256 * 256 * 256) ' fw 5.6
end function

'** Read an image record, returning the width, height,
'** dataOffset and dataLength.
'** Only returns records for 32-bit BMP data currently.
'**
function USBUtilReadIcoImageInfo(ba_, numImages_, imageIndex_)
    info_ = {}

    pos_ = 6 + imageIndex_ * 16

    info_.width = ba_[pos_]
    if info_.width = 0
        info_.width = 256
    endif

    info_.height = ba_[pos_ + 1]
    if info_.height = 0
        info_.height = 256
    endif

    info_.numColors = ba_[pos_ + 2]

    if ba_[pos_ + 3] <> 0
        print "Ico file: image reserved is not 0"
        return invalid
    endif

    info_.numColorPlanes = BAGetUInt16(ba_, pos_ + 4)
    if info_.numColorPlanes <> 1
        print "Ico file: image numColorPlanes not recognized"
        return invalid
    endif

    info_.bitsPerPixel = BAGetUInt16(ba_, pos_ + 6)
    if info_.bitsPerPixel <> 32
        print "Ico file: image bitsPerPixel not 32"
        return invalid
    endif

    info_.dataLength = BAGetInt32(ba_, pos_ + 8)
    info_.dataOffset = BAGetInt32(ba_, pos_ + 12)

    bytesPerPixel_ = 4

    minDataLen_ = info_.width * bytesPerPixel_ * info_.height
    if info_.dataLength < minDataLen_
        print "Ico file: image data length "; info_.dataLength.ToStr(); " below min "; minDataLen_.ToStr()
        return invalid
    endif

    maxDataLen_ = (info_.width * bytesPerPixel_ + 31) * info_.height
    if info_.dataLength > maxDataLen_
        print "Ico file: image data length "; info_.dataLength.ToStr(); " above max "; maxDataLen_.ToStr()
        return invalid
    endif

    baLen_ = ba_.Count()

    if (info_.dataOffset < 6 + 16 * numImages_)
        print "Ico file: image data offset invalid (short)"
        return invalid
    endif

    if (info_.dataOffset >= baLen_) or (info_.dataOffset + info_.dataLength > baLen_)
        print "Ico file: image data offset invalid"
        return invalid
    endif

    return info_
end function

function USBUtilGetPngData(ba_, bestImageInfo_, dstPath_)
    pos_ = bestImageInfo_.dataOffset
    len_ = bestImageInfo_.dataLength

    if not ba_.WriteFile(dstPath_, pos_, len_)
        print "Ico file: tmp file read failed"
        return invalid
    endif

    return dstPath_
end function

function USBUtilGetVolumePoster(mm, volume)
    print "USBGetVolumePoster: "; volume

    posterUrl = usbFileExists(volume + "/icon.png")

    if posterUrl = invalid
        posterUrl = usbFileExists(volume + "/icon.jpg")
        if posterUrl = invalid
            posterUrl = USBUtilGetPosterFromIco(mm, volume)
        endif
    endif

    if posterUrl <> invalid
        print "Set posterurl to:"; posterUrl
    end if
    return posterUrl
end function

function USBUtilSavePngFromBM40(ba_, bestImageInfo_, dstPath_)
    pos_ = bestImageInfo_.dataOffset
    width_ = bestImageInfo_.width
    height_ = bestImageInfo_.height

    bmSpec_ = {
        width: width_
        height: height_
        alphaEnable: false
    }
    bm_ = CreateObject("roBitmap", bmSpec_)
    bm_.Clear(&h808080FF)

    bmHeaderSize_ = BAGetInt32(ba_, pos_)
    if bmHeaderSize_ <> 40
        print "Ico file: image hdr is not BITMAPINFOHEADER"
        return invalid
    endif

    pos_ = pos_ + bmHeaderSize_

    for y_ = 0 to height_ - 1
        for x_ = 0 to width_ - 1
            ' assume BGRA
            cb_ = ba_[pos_ + 0]
            cg_ = ba_[pos_ + 1]
            cr_ = ba_[pos_ + 2]
            ca_ = ba_[pos_ + 3]
            rgba_ = (cr_ << 24) + (cg_ << 16) + (cb_ << 8) + (ca_)
'            rgba_ = (cr_ * 256 * 256 * 256) + (cg_ * 256 * 256) + (cb_ * 256) + (ca_) ' fw 5.6
            bm_.DrawRect(x_, height_ - 1 - y_, 1, 1, rgba_)
            pos_ = pos_ + 4
        end for
    end for

    bmPngData_ = bm_.GetPng(0, 0, width_, height_)
    if bmPngData_ = invalid
        print "Ico file: image get PNG failed"
        return invalid
    endif

    if not bmPngData_.WriteFile(dstPath_)
        print "Ico file: file write failed"
        return invalid
    end if

    return dstPath_
end function

function USBUtilGetPosterFromIco(m, volume)
    autoRunInfo = USBUtilReadAutoRunInf(m, volume)
    if autoRunInfo = invalid
        return invalid
    endif

    icoFileRelPath = autoRunInfo.icon
    if icoFileRelPath = invalid
        return invalid
    end if

    print "Ico file: raw path = " ; icoFileRelPath

    ' FIXME: firmware bug in FilesystemPath.cpp boost garbage
    ' causes filenames starting with '.' to be inaccessible
    ' even though they get enumerated.
    if false
        listing = m.rofs.GetDirectoryListing(volume + "/")
        print "=============="
            for each f in listing
                print "Entry: " ; f
            end for
        print "=============="
    endif

    iconFilePath = volume + "/" + icoFileRelPath.Replace("\", "/")
    if not m.rofs.Exists(iconFilePath)
        print "Ico file: file not found: '" ; iconFilePath ; "'"
        return invalid
    endif

    ba_ = CreateObject("roByteArray")
    if not ba_.ReadFile(iconFilePath)
        print "Ico file: file read failed"
        return invalid
    end if

    baLen_ = ba_.Count()
    ' must at least have space for header + 1 image
    if baLen_ < 6 + 16
        print "Ico file: file too short"
        return invalid
    end if

    n = BAGetUInt16(ba_, 0)
    if n <> 0
        print "Ico file: header reserved invalid"
        return invalid
    endif

    n = BAGetUInt16(ba_, 2)
    if n <> 1
        print "Ico file: header type invalid: " ; n.ToStr()
        return invalid
    endif

    numImages_ = BAGetUInt16(ba_, 4)
    if not (numImages_ >= 1 and numImages_ <= 32)
        print "Ico file: header num images invalid"
        return invalid
    endif

    print "Ico file: found "; numImages_.ToStr(); " images"

    bestImageInfo_ = invalid

    for imageIndex_ = 0 to numImages_ - 1
        imageInfo_ = USBUtilReadIcoImageInfo(ba_, numImages_, imageIndex_)
        if imageInfo_ <> invalid
            print "Ico file: image:"
            print imageInfo_
            if (bestImageInfo_ = invalid) or (imageInfo_.width > bestImageInfo_.width)
                bestImageInfo_ = imageInfo_
            endif
        endif
    end for

    if bestImageInfo_ = invalid
        print "Ico file: no supported images matched"
        return invalid
    endif

    print "Ico file: Best image:"
    print bestImageInfo_

    pos_ = bestImageInfo_.dataOffset

    '** note: we need to make the tmp icon file name unique to the media 
    '** device inserted. We can't just use the volume name itself, because  
    '** the Qt UI caches images by path.  So if a previous icon was loaded 
    '** for a volume, it won't get re-loaded in the UI when a new drive is 
    '** inserted on that volume with a different icon.
    '** Hmm, would roImageCanvas PurgeCachedImages help?

    volInfo_ = m.rofs.GetVolumeInfo(volume + "/")
    print "Vol info:"
    print volInfo_
    print "=========="

    volStat_ = m.rofs.Stat(volume + "/")
    print "Vol stat:"
    print volStat_
    print "=========="

    if volInfo_.uuid <> invalid and volInfo_.uuid <> ""
        volumeIdStr_ = volInfo_.uuid
    else if volInfo_.label <> invalid and volInfo_.label <> ""
        '** FIXME: we need a CRC-32 utility in firmware e.g. roByteArray
        volumeId_ = 0
        for each c_ in volInfo_.label.Split("")
            volumeId_ = (volumeId_ << 1) + Asc(c_)
'            volumeId_ = (volumeId_ * 2) + Asc(c_) 'fw 5.6
        end for
        volumeIdStr_ = StrI(volumeId_, 16)
    else
        '** FIXME need to add some management so only a maximum number
        '** of icon files exist, and the old ones get cleaned up.
        volumeId_ = CreateObject("roDateTime").AsSeconds()
        volumeIdStr_ = StrI(volumeId_, 16)
    endif

    tmpPath_ = "tmp:/" + "vol_icon_" + volumeIdStr_ + ".png"

    resultPath_ = invalid

    bmType_ = BAGetInt32(ba_, pos_)
    if bmType_ = &h474e5089
        print "Ico file: image type PNG"
        resultPath_ = USBUtilSavePngData(ba_, bestImageInfo_, tmpPath_)
    else if bmType_ = 40
        print "Ico file: image type BITMAPINFOHEADER"
        resultPath_ = USBUtilSavePngFromBM40(ba_, bestImageInfo_, tmpPath_)
    else
        print "Ico file: image type not supported"
        return invalid
    endif

    if resultPath_ = invalid
        print "Ico file: image save failed"
        return invalid
    endif

    print "Ico file: file write OK: " ; resultPath_
    return resultPath_
end function

