' ********** Copyright 2019 Roku Corp. All Rights Reserved. **********

''''''''''''''''''''''''''''''''''''''''''
'' Config object
'' The config object works with "configData" - shared object in global node
'' Which stores all config fields
''''''''''''''''''''''''''''''''''''''''''
function Config()
    ' instant return if object is created in this scope
    if m.__appConfig <> invalid then return m.__appConfig
    ' init config object
    m.__appConfig = NewConfig()
    gNode = GetGlobalAA().global
    ' save reference to global node to config object for quick access
    m.__appConfig.gNode = gNode
    globalConfigData = gNode.configData
    if globalConfigData = invalid
        ' function called for first time and we should add configData shared object to global
        gNode.AddField("configData", "assocarray", false)
        globalConfigData = {
            fields: {}
            deviceInfo: {}
        }
        gNode.configData = globalConfigData
    end if
    ' save to config object for quick access to config properties from one scope
    m.__appConfig.configData = globalConfigData
    ' return new created config
    return m.__appConfig
end function

function NewConfig()
    this = {
        AddSet: Config_AddSet
        Get: Config_Get
        GetFromGlobal: Config_GetFromGlobal
        IsTrue: Config_IsTrue
        UpdateLocal: Config_UpdateLocal
        SetDeviceInfo: Config_SetDeviceInfo
        GetDeviceInfo: Config_GetDeviceInfo

        __GetSharedObject: Config__GetSharedObject
        __SetSharedObject: Config__SetSharedObject
    }
    return this
end function

sub Config_AddSet(key as String, value as Object)
    gso = m.__GetSharedObject()
    gso.fields.AddReplace(key, value)
    m.configData = gso
    m.__SetSharedObject(gso)
end sub

function Config_Get(key as String) as Object 
    return m.configData.fields[key]
end function

function Config_GetFromGlobal(key as String) as Object
    gso = m.__GetSharedObject()
    return gso.fields[key]
end function

function Config_IsTrue(key as String) as Boolean
    return (m.Get(key) = true)
end function

sub Config_UpdateLocal()
    m.configData = m.__GetSharedObject()
end sub

sub Config_SetDeviceInfo(di as Object)
    if di <> invalid
        gso = m.__GetSharedObject()
        gso.deviceInfo = di
        m.__SetSharedObject(gso)
    end if
end sub

function Config_GetDeviceInfo() as Object
    if  m.configData <> invalid
        return m.configData.deviceInfo
    else
        gso = m.__GetSharedObject()
        return gso.deviceInfo
    end if
end function

function Config__GetSharedObject() as Object
    return m.gNode.configData
end function

sub Config__SetSharedObject(configData as Object)
    m.gNode.Update({ configData: configData })
end sub

''''''''''''''''''''''''''''''''''''''''''
'' Helper functions
''''''''''''''''''''''''''''''''''''''''''
function isAudioOnly() as Boolean
    return Config().IsTrue("audioOnly")
end function
function isUSBOnly() as Boolean
    return (not Config().IsTrue("enableDLNA"))
end function
function isROMOnly() as Boolean
    return Config().IsTrue("romOnly")
end function
function useDLNA() as Boolean
    return Config().IsTrue("enableDLNA")
end function
function useHttpFS() as Boolean
    return Config().IsTrue("enableHTTPFS")
end function
function isGlobalSubtitle() as Boolean
    return Config().IsTrue("isGlobalSubtitle")
end function
function allowPlaylists() as Boolean
    return Config().IsTrue("Playlists")
end function
' function platformHasAudioMetaData() as Boolean
'     return Config().IsTrue("platformHasAudioMetaData")
' end function
function AutorunDisabled() as Boolean
    return Config().IsTrue("AutorunDisabled")
end function
function hasCacheFS() as Boolean
    return Config().IsTrue("hasCacheFS")
end function
function hasSizeEx() as Boolean
    return Config().IsTrue("hasSizeEx")
end function
function isVerbose() as Boolean
    return Config().IsTrue("Verbose")
end function
function hasISODate() as Boolean
    return Config().IsTrue("hasISODate")
end function
function isPro() as Boolean
    return Config().IsTrue("isPro")
end function
function useNWM() as Boolean
    return Config().IsTrue("useNWM")
end function
function dlnaGetAllData() as Boolean
    return Config().IsTrue("dlnaGetAlldata")
end function
function DimensionCheckDisabled() as Boolean
    return Config().IsTrue("DimensionCheckDisabled")
end function
function hasExt3() as Boolean
    return Config().IsTrue("hasExtFS")
end function
function hasLocalBif() as Boolean
    return Config().IsTrue("localBif")
end function
function hasMultipleSRTURLs() as Boolean
    return Config().IsTrue("MultipleSubtitleUrls")
end function

function maxThumbPixels() as Integer
    return Config().Get("maxThumbPixels")
end function
function maxGifSize() as Integer
    return Config().Get("gifMAXsize")
end function
function maxUPNPRequest() as Integer
    return Config().Get("maxUPNPRequest")
end function

''''''''''''''''''''''''''''''''''''''''''
'' Media types
''''''''''''''''''''''''''''''''''''''''''
function DirId() as Integer
    return 0
end function

function AllMediaId() as Integer
    return 0
end function

function VideoID() as Integer
    return 1
end function

function AudioID() as Integer
    return 2
end function

function PictureID() as Integer
    return 3
end function

function MediaID() as Integer
    return 1
end function

function MaxFilterId() as Integer
    return 3
end function

function DLNAAccessType() as integer
    return 0
end function

function USBAccessType() as Integer
    return 1
end function

function HTTPAccessType() as integer
    return 2
end function

function ListFormatID() as Integer
    return 1
end function

''''''''''''''''''''''''''''''''''''''''''
'' Media Error 
''''''''''''''''''''''''''''''''''''''''''
function RFI_NetworkError() as Integer
    return 0
end function

function RFI_HttpError() as Integer
    return -1
end function

function RFI_TimedOut() as Integer
    return -2
end function

function RFI_Unknown() as Integer
    return -3
end function

function RFI_EmptyList() as Integer
    return -4
end function

function RFI_MediaError() as Integer
    return -5
end function

''''''''''''''''''''''''''''''''''''''''''
'' Grid Rows Style
''''''''''''''''''''''''''''''''''''''''''
function DefaultGridRowsStyleID() as Integer
    return 0
end function

function ShortGridRowsStyleID() as Integer
    return 1
end function

function LongGridRowsStyleID() as Integer
    return 2
end function

''''''''''''''''''''''''''''''''''''''''''
'' Option types
''''''''''''''''''''''''''''''''''''''''''
function filteringOptionID() as Integer
    return 0
end function

function sortingOptionID() as Integer
    return 1
end function

''''''''''''''''''''''''''''''''''''''''''
'' Sorting
''''''''''''''''''''''''''''''''''''''''''
function sortingDefaultID() as Integer
    return 0
end function

function sortingAscendingID() as Integer
    return 1
end function

function sortingDescendingID() as Integer
    return 2
end function

function getSorting() as Integer
   return Config().Get("mediaSorting")
end function

function setSorting(sorting as Integer)
    Config().AddSet("mediaSorting", sorting)
    SetSortingSettings(sorting)
end function

function getDefaultSorting() as Integer
    setSort = GetSortingSettings()
    if(setSort <> invalid) then
        return setSort
    end if
    return sortingDefaultID()
end function

function GetLocalizedImage(name as String) as String
    if m.localizer = invalid
        m.localizer = CreateObject("roLocalization")
    end if
    return m.localizer.GetLocalizedAsset("images", name)
end function

''''''''''''''''''''''''''''''''''''''''''
'' Device info
''''''''''''''''''''''''''''''''''''''''''
function hasAutoDetect() as Boolean
    return (Config().GetDeviceInfo().hasAutoDetect = true)
end function
function hasVP9() as Boolean
    return (Config().GetDeviceInfo().hasVP9 = true)
end function
function hasPCM() as Boolean
    return (Config().GetDeviceInfo().hasPCM = true)
end function
function hasFlac() as Boolean
    return (Config().GetDeviceInfo().hasFlac = true)
end function
function hasDTS() as Boolean
    return (Config().GetDeviceInfo().hasDTS = true)
end function
function hasAlac() as Boolean
    return (Config().GetDeviceInfo().hasAlac = true)
end function
function hasOgg() as Boolean
    return (Config().GetDeviceInfo().hasOgg = true)
end function
function hasFLV() as Boolean
    return (Config().GetDeviceInfo().hasFLV = true)
end function
function hasASF() as Boolean
    return (Config().GetDeviceInfo().hasASF = true)
end function
function hasAac() as boolean
    return (Config().GetDeviceInfo().hasAac = true)
end function
function hasAC3() as Boolean
    return (Config().GetDeviceInfo().hasAC3 = true)
end function
function hasEAC3() as Boolean
    return (Config().GetDeviceInfo().hasEAC3 = true)
end function
function hasWMV() as Boolean
    return (Config().GetDeviceInfo().hasWMV = true)
end function
function hasMat() as Boolean
    return (Config().GetDeviceInfo().hasMat = true)
end function
function hasAC4() as Boolean
    return (Config().GetDeviceInfo().hasAC4 = true)
end function
function hasUSB() as Boolean
    return (Config().GetDeviceInfo().hasUSB = true)
end function
function hasMPEG2() as Boolean
    return (Config().GetDeviceInfo().hasMPEG2 = true)
end function
function hasSDCARDhardware() as Boolean
    return (Config().GetDeviceInfo().has_sdcard_hardware = true)
end function
function hasAac51() as Boolean
    return (Config().GetDeviceInfo().hasAac51 = true)
end function
function hasH265() as Boolean
    return (Config().GetDeviceInfo().hasH265 = true)
end function
function hasNativeAC3() as Boolean
    return (Config().GetDeviceInfo().hasNativeAC3 = true)
end function
function hasSPDIF() as Boolean
    return (Config().GetDeviceInfo().hasSPDIF = true)
end function


''''''''''''''''''''''''''''''''''''''''''
'' Filtering
''''''''''''''''''''''''''''''''''''''''''
'folder level
' function getLocalFiltering() as Integer
'     return getMTFilter()
' end function

' function setLocalFiltering(filtering as Integer) 
'    setMTFilter(filtering)
' end function

'global level
function getMTFilter() as Integer
    return Config().Get("mediaTypeFiltering")
end function

function setMTFilter(filter as Integer) 
    ? "setMTFilter";filter
    Config().AddSet("mediaTypeFiltering", filter)
    SetFilteringSettings(filter)
end function

function getMediaTypeFilters() as Object
    return Config().Get("mediaTypeFilters")
end function

function AllMediaFilterEnabled()
    return Config().IsTrue("allMediaFilterEnabled")
end function

function getDefaultFiltering() as Integer
    setFilter = GetFilteringSettings()
    if setFilter = invalid then
        if Config().IsTrue("audioOnly")
            setFilter = audioID()
        else
            ' we check both of these flags because isromonly is false for the usbplayer private channel
            ' and isusbonly might be false if we launch in unconnected mode (USB) but find there is an active network
            if isRomOnly() or isUSBOnly() ' want video as default for USB player (needed for mfg compatibility)
                setFilter = videoID()
            else
                setFilter = allMediaId() ' all media is default for RMP
            endif
        endif
    endif
    return setFilter
end function

function getSafeNetworkStatus() as Boolean
    linkStatus = true
    if Config().IsTrue("linkStatusSupported") then
        linkStatus = CreateObject("roDeviceInfo").GetLinkStatus()
    end if
    return linkStatus
end function

''''''''''''''''''''''''''''''''''''''''''
'' UserAgent related helper functions
''''''''''''''''''''''''''''''''''''''''''
sub applyUserAgent(obj as Object, provider as Object)
    ' Eventually user-agents may be server specific besides platform specific
    ' User Agents are only needed for UPNP
    if provider.UserAgentHeader <> invalid
        UserAgentHeader = computeUserAgent(provider)
        print "USERAGENT=";UserAgentHeader
        if UserAgentheader <> ""
            obj.addheader("user-agent",UserAgentHeader)
        end if
    end if
end sub

function computeUserAgent(provider as Object) as String
   if provider.UserAgentHeader <> ""
       return provider.UserAgentHeader
   else if Config().Get("UserAgentHeader") <> ""
       return Config().Get("UserAgentHeader")
   endif
   return ""
end function

' returns current setting of subtitle mode
function getSubtitleMode() as string
    if Config().IsTrue("hasDynamicSubtitleMode")
        return CreateObject("roDeviceInfo").GetCaptionsMode()
    else
        return Config().Get("subtitleMode")
    endif
end function

' ''''''''''''''''''''''''''''''''''''''''''
' '' config
' ''''''''''''''''''''''''''''''''''''''''''
' function launchVolume()
'     return m.launchVolume
' end function
' function clearLaunchVolume()
'     m.launchVolume = ""
' end function
' function hasCECAutoShutdown()
'     return m.CECAutoshutdown
' end function
' function setSignalValid(valid)
'     if m.CECAutoShutdown
'         m.app.SetSignalValid(valid)
'     endif
' end function
' function allowBobDescription() as boolean
'     return m.allowBobDescription
' end function
' function MaxPhotoScale() as float
'     return m.MaxPhotoScale
' end function
' function hasDolbyDigital() as boolean
'     return m.deviceInfo.hasDolbyDigital
' end function
' function hasNativeEAC3() as boolean
'     return m.deviceInfo.hasNativeEAC3
' end function
' function hasVP9() as boolean
'     return m.deviceInfo.hasVP9
' end function
' function hasVC1() as boolean
'     return m.deviceInfo.hasVC1
' end function
' function is7dot5() as boolean
'     return m.deviceInfo.is7dot5
' end function
' function hasMat() as boolean
'     return m.deviceInfo.hasMat
' end function
' function IsHD() as boolean
'     return m.isHD
' end function
' function enableNWM(enable as boolean)
'     m.useNWM = enable
' end function
' function useCCUI() as boolean
'     return m.ClosedCaptionUI
' end function
' function loopDir() as boolean
'     return m.loopDir
' end function
' function UpExitsGrid() as boolean
'     return m.UpExitsGrid
' end function
' function isVoiceOnly() as boolean
'     return m.VoiceOnly
' end function
' function hasVoiceAdapter() as boolean
'     return m.hasVoiceAdapter
' end function
' function hasFLV() as boolean
'     return m.deviceInfo.hasFLV
' end function

' function allowPlaylists() as boolean
'     return m.Playlists
' end function

' function GetDisplaySize()
'         return m.roDI.GetDisplaySize()
' end function

' function isBreadCrumbSupported() as boolean
'     return m.isBreadCrumbSupported
' end function

' function isUpSupported() as Boolean
'     return m.isUpSupported
' end function
' function isUpdateButtonSupported() as boolean
'     return m.hasUpdateButton
' end function

' ''''''''''''''''''''''''''''''''''''''''''
' '' Current content directory
' ''''''''''''''''''''''''''''''''''''''''''
' function setLastObject(objectID as String) as Void
'     m.lastObjectId = objectID
' end function

' function getLastObject() as String
'     return m.lastObjectId
' end function

' ''''''''''''''''''''''''''''''''''''''''''
' '' Screen configs
' ''''''''''''''''''''''''''''''''''''''''''
' function getGridJumpOffset() as Integer
'     return 10
' end function

' function setSubtitleMode(option as string)
'     m.subtitleMode = option
'     SetSavedSubtitleMode(option)
' end function

' ''''''''''''''''''''''''''''''''''''''''''
' '' Volume configs
' ''''''''''''''''''''''''''''''''''''''''''
' function getActiveVolume() as Object
'    ' print "---------------------------------->>>> getActiveVolume"
'     return m.activeVolume
' end function

' function setActiveVolume(volume as Object) as Boolean
'     'reset
'     if volume = invalid then
'        m.activeVolume = invalid
'        m.volumePtr = invalid
'        return true
'     end if
'     m.activeVolume = volume
'     if volume.accessType = DLNAAccessType()
'         m.volumePtr = ssdpptr()
'     else if volume.accesstype = USBAccessType()
'         m.volumePtr = usbptr()
'     else
'         m.volumePtr = httpfsptr()
'     endif
'     return true
' end function

' function getVolumeHandlePtr() as Object
'     return m.volumePtr
' end function

' ''''''''''''''''''''''''''''''''''''''''''
' '' Queue configs
' ''''''''''''''''''''''''''''''''''''''''''
' function nextID() as Integer
'     return -1
' end function
' function prevID() as Integer
'     return -2
' end function

' function isLinkStatusSupported() as Boolean
'     return m.linkStatusSupported
' end function

' function setNetworkingRequest(req as Boolean) as Void
'       m.networkingRequest = req
'       if req then
'         ErrorTrackerPtr().resetLastError()
'       end if
' end function

' function getNetworkingRequest() as Boolean
'       return m.networkingRequest
' end function

' function setNetworkingRequestURL(url as String) as Void
'     m.targetRequestURL = url
' end function

' function getNetworkingRequestURL() as String
'     if m.targetRequestURL = invalid then
'         return ""
'     end if
'     return m.targetRequestURL
' end function

' function springBoardTempImageID() as String
'     return "tmp:/tmpVideoSpringBoardImage"
' end function

' function isUSB() as Boolean
'  iusb = false
'  activeVolume = getActiveVolume()
'  if activeVolume <> invalid then
'     if activeVolume.accessType = USBAccessType() then
'         iusb = true
'     end if
'  end if
'  return iusb
' end function

' function isTrickControlAdded() as Boolean
'     return m.isTrickControlSupported
' end function

' function setRemovalDetection(detect as Boolean )
'     m.removalDetected = detect
' end function

' function isRemovalDetectionOn() as Boolean 
'     return m.removalDetected
' end function

' function setRemovedVolume(volume as Object )
'     m.removedVolume = volume
' end function

' function getRemovedVolume() as Object 
'     return m.removedVolume
' end function

function no_tr(t as string) as string
    return t
end function
