'///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Sub RunScreenSaver( params As Object )

' DESCRIPTION
'     Fallback default screensaver entry point.
'
'     This screensaver acts as a last-resort screensaver if no other channel store default 
'     has been defined. As a last-resort fallback, this screensaver simply clears the screen 
'     to a selected background color and waits for a button press to tell it to exit.
'
'     In reality, this code should never actually be run. It's just firmware placeholder 
'     code for the default screensaver channel ID. Logic in the FW that launches the active 
'     screensaver at the proper time checks if the FW version of the default screensaver is 
'     being launched, and if it is, the safe-mode screensaver is launched instead.
'
'     The reason we go through this re-direct process is primarily to save FS space and to
'     reduce maintenance headaches by avoiding duplicate code and assets. Since the safe-mode 
'     and default screensavers each have their own channel IDs, they can be launched in 
'     different situations as needed. Having separate channel IDs also allows the two 
'     screensavers to be independently overridden by updated code published to the  channel 
'     store. However, having separate channel IDs requires each screensaver to have its own 
'     code / asset package. This would be fine if the firmware fallback versions of the two 
'     screensavers were fundamentally different, but in fact we want them both to behave 
'     like the safe-mode screensaver because it's pretty much guaranteed to work on any 
'     platform. 
'
'     To implement safe-mode behavior in both screensavers, we could either put copies of 
'     all the safe-mode code and asset files in both packages, or we could reduce the FW 
'     version of the default screensaver to a minimal implementation never intended to be 
'     run, and redirect its requested execution to the safe-mode screensaver. Since file-
'     system space is at a premium, and because maintaining two copies of the same code 
'     and assets is a headache, the redirection approach was selected.
'
' PARAMETERS
'     params - a list of parameters passed by the caller.
'
' RETURNS
'     Void.

    ' Create a single- or double-buffered screen object as desired.
    doubleBuffered = true   
    screen = CreateObject( "roScreen", doubleBuffered )
    
    ' For this simple blanking screensaver, we don't need alpha blending. (Turning alpha
    ' blending off makes for faster draws.)
    '
    screen.SetAlphaEnable( false )
    
    ' Establish a message port so we can sense button presses.
    screenMsgPort = CreateObject( "roMessagePort" )
    screen.SetMessagePort( screenMsgPort )
    
    ' Clear the screen to the background color and display.
    bkgdColor = &h000000FF
    screen.Clear( bkgdColor )
    screen.SwapBuffers()
    
    ' If the screen is double-buffered, clear both buffers for good measure.
    if doubleBuffered
        screen.Clear( bkgdColor )
        screen.SwapBuffers()
    end if
    
    ' Finally wait for a button press to trigger screensaver exit.
    while true

        msg = screenMsgPort.WaitMessage(0)
        
        ' If we got a button press, we're done.
        if Type(msg) = "roUniversalControlEvent"
            if msg.IsPress()
                exit while
            end if
        end if

    end while 'true
 
End Sub ' RunScreenSaver( params As Object )
