Library "v30/bslCore.brs"

function main() as Void
    urls = [
        { type: "normal",   url: "https://tvupdate.roku.com/cgi-bin/update-factory" }
        { type: "staging",  url: "https://tvupdate.staging.roku.com/cgi-bin/update-factory" }
        { type: "QA",       url: "https://tvupdate.qa.roku.com/cgi-bin/update-factory" }
    ]

    ' Create objects
    port = CreateObject("roMessagePort")
    m.screen = CreateObject("roScreen", true, 1280, 720)
    m.screen.SetMessagePort(port)
    update = CreateObject("roFirmwareUpdate")
    update.SetMessagePort(port)

    reg = CreateObject("roFontRegistry")
    m.msgs = []
    m.msg_line = 0
    m.font = reg.GetDefaultFont(30, false, false)
    codes = bslUniversalControlEventCodes()

    devInfo = CreateObject("roDeviceInfo")
    brand = LCase(devInfo.GetModelDetails().VendorUSBName)

    tv = CreateObject("roTvManufacturing")
    esn = tv.getSerialNumber()

    ' Wait for instructions from user.
    Message("     === Firmware Update ===")
    Message("")
    Message("Press LEFT to change or OK to start update.")
    countdown = 10
    timer = CreateObject("roTimespan")
    server = 0
    while true
        remain = countdown - timer.TotalSeconds()
        banner = "Starting "+urls[server].type+" update in "+remain.ToStr()+" seconds."
        Message(banner, false)
        msg = wait(100, port)
        if type(msg) = "roUniversalControlEvent" then btn = msg.GetInt() else btn = 0
        if btn = codes.BUTTON_LEFT_RELEASED then
            server = server + 1
            if server >= urls.count() server = 0
            timer.Mark()
        elseif btn = codes.BUTTON_SELECT_RELEASED or remain <= 0 then
            exit while
        end if
    end while

    ' Initiate update and get a request code for the update package.
    Message("Starting update ...")
    err = update.Start()
    if err <> 0 then 
        ErrorMessage(err, "Start error")
        WaitForButton(port)
        return
    end if

    ' Extract info from the update package.
    id       = update.GetManifestID()
    name     = update.GetUpdateName()
    req_code = update.GetRequestCode()

    Message("Update #" + id.ToStr() + " : " + name)
    Message("Request code " + req_code)
    Message("Getting authorization code ...", false)

    ' Get response code from Roku server.
    server_url = urls[server].url
    xfer = CreateObject("roUrlTransfer")
    xfer.SetCertificatesFile("common:/certs/ca-bundle.crt")
    request_url = server_url + "?brand=" + brand + "&code=" + req_code
    if 1 = esn.valid then
        request_url = request_url + "__" + esn.serial
    end if
    xfer.SetURL(request_url)
    response = xfer.GetToString()

    ' Response should be "20x RESPONSE_CODE"
    if response.Mid(0,1) <> "2" then
        ' First digit not 2 is an error
        Message("Error: server says:")
        Message(response)
        WaitForButton(port)
        return
    elseif response.Mid(1,1) <> "0" then
        ' Second digit not 0 means update is not factory-compatible.
        Message("Error: this is not a manufacturing update")
        WaitForButton(port)
        return
    end if

    ' Finish update using the response code.
    resp_code = response.Mid(4)
    Message("Authorization code " + resp_code)
    err = update.Finish(resp_code)
    if err <> 0 then 
        ErrorMessage(err, "Finish error")
        WaitForButton(port)
        return
    end if

    Message("Installing update ...")
    while true
        msg = port.GetMessage()
        if type(msg) = "roFirmwareUpdateEvent" then
            progress = msg.GetInfo().progress
            if progress <> invalid then
                pct = Int(progress / 10)
                Message("progress " + pct.ToStr() + "%", false)
            end if
        end if
        ' Status=1 means update is still in progress
        if update.GetStatus() <> 1 then exit while
        if msg = invalid then Draw()
    end while

    err = update.GetStatus()
    ErrorMessage(err, "Update finished")
    if err = 0 then
        tv = CreateObject("roTvManufacturing")
        if tv <> invalid then
            Message("Rebooting system now ...")
            Sleep(1000)
            tv.RebootSystem()
        end if
    end if
    WaitForButton(port)

end function

function ErrorMessage(err as Integer, msg as String) as Void
    if err =  0 then
        desc = "Success"
    elseif err =  1 then 
        desc = "Update in progress"
    elseif err = -1 then 
        desc = "File not found"
    elseif err = -2 then 
        desc = "Update is not compatible"
    elseif err = -3 then 
        desc = "Cannot downgrade"
    elseif err = -9 then 
        desc = "Update failed"
    else                 
        desc = "Error " + err.ToStr()
    end if
    Message(msg + " : " + desc)
end function

function Message(msg as String, advance = true as Boolean) as Void
    m.msgs[m.msg_line] = msg
    if advance then m.msg_line = m.msg_line + 1
    Draw()
end function

function Draw() as Void
    m.screen.Clear(&h301030FF)
    h = m.font.GetOneLineHeight()
    for i = 0 to m.msgs.count()-1
        m.screen.DrawText(m.msgs[i], 80, 50 + i*h, &hF0F0F0FF, m.font)
    end for
    m.screen.SwapBuffers()
end function

function WaitForButton(port as Object) as Void
    Message(" ")
    Message("-- Press any button --")
    while true ' drain port
        if port.GetMessage() = invalid then exit while
    end while
    while true
        msg = wait(0,port)
        if type(msg) = "roUniversalControlEvent" then return
    end while
end function

