#!/usr/bin/env python

# Copyright (C) 2013-2014 Opera TV AS.  All rights reserved.
#
# This file is part of the Opera Devices SDK.
# It includes Opera TV proprietary information and distribution is prohibited
# without Opera TV's prior, explicit and written consent.

import os
from ast import literal_eval
from shutil import rmtree
import argparse
import fnmatch

if __name__ == '__main__':
  parser = argparse.ArgumentParser("Pack or unpack data files.")
  parser.add_argument("-g", "--grit", metavar="grit_path", default=os.path.join(os.path.dirname(__file__), "../grit"))
  parser.add_argument("-u", "--unpack", metavar="data_file", help="data file to unpack")
  parser.add_argument("-q", "--quiet", action='store_true', help="quiet")
  parser.add_argument("source_folder", nargs="?", help="path to source folder")
  parser.add_argument("dest_pak", nargs="?", help="output .pak file name. Default value is source_folder's name")
  parser.add_argument("-e", "--exclude", nargs="?", metavar="exclude_pattern", help="Comma-separated glob pattern list to exclude files matching it when packing")
  parser.add_argument("-a", "--extra-folder", nargs="?", metavar="extra_folder", help="Additonal folder to also include in the .pak file")

  args = parser.parse_args()

  os.sys.path.append(args.grit)

from grit.format.data_pack import DataPack
from grit import util

def unpack(file_name):
  if not os.path.isfile(file_name):
    print "Unable to locate data file: " + file_name
    return
  data = DataPack.ReadDataPack(file_name)
  # remove old folder if it already exists:
  folder = os.path.split(file_name)[1][:-4]
  if os.path.exists(folder):
    rmtree(folder)
  # iterate over all data resources in the data file:
  for (resource_id, text) in data.resources.iteritems():
    if resource_id == 0:
      paths = literal_eval(text)
      continue
    directory = os.path.join(folder, os.path.dirname(paths[resource_id-1]))
    if not os.path.exists(directory):
      os.makedirs(directory)
    file_path = os.path.join(folder, paths[resource_id-1])
    if os.path.exists(file_path):
      os.sys.exit("The data file seems to be misconfigured and contains "
          "two files with the same path.")
    with open(file_path, 'wb') as output_file:
      output_file.write(text)

def pack(folder, output_file = None, exclude = None, quiet = False, extra_folder = None):
  if not os.path.isdir(folder):
    os.sys.exit("Unable to pack single files, please provide a folder to pack "
        "or a file ending with .pak to unpack.")
  if exclude is not None:
    excludes = exclude.split(',')
  else:
    excludes = []

  data = []
  data.append([]) # this will be the file index
  def add_file(data, folder, excludes, root, files):
    for exclude in excludes:
      files = [f for f in files if not fnmatch.fnmatch(os.path.join(root, f), exclude)]
    for f in files:
      file_path = os.path.abspath(os.path.join(root, f))
      pack_path = file_path[len(os.path.abspath(folder))+1:]
      data[0].append(pack_path)
      data.append(util.ReadFile(file_path, util.BINARY))
  for root, _, files in os.walk(folder):
    add_file(data, folder, excludes, root, files)
  if extra_folder is not None:
    for root, _, files in os.walk(extra_folder):
      add_file(data, extra_folder, excludes, root, files)
  data[0] = str(data[0]).replace("'", "\"")
  if output_file == None:
    output_file = os.path.basename(os.path.abspath(folder)) + ".pak"
  DataPack.WriteDataPack(dict(enumerate(data)), output_file, 0)
  if not quiet:
    print "Saved {} files from {} to {}".format(len(data)-1, folder, output_file)

def main(args):
  if args.unpack:
    unpack(args.unpack)
  else:
    pack(args.source_folder, args.dest_pak, args.exclude, args.quiet, args.extra_folder)

if __name__ == '__main__':
  main(args)

