/* (c) 2011 Netflix, Inc. Do not copy or use without prior written permission from Netflix, Inc. */

function ttmlParseTime(time) {
    var result = 0;
    var caps = (/([0-9]+):([0-9][0-9]):([0-9][0-9]):([0-9][0-9])/).exec(time);
    if(caps) {
        result += parseInt(caps[1]) * 60 * 60 * 1000;
        result += parseInt(caps[2]) * 60 * 1000;
        result += parseInt(caps[3]) * 1000;
        result += parseInt(caps[4]);
    } else {
        caps = (/([0-9]+):([0-9][0-9]):([0-9][0-9]).([0-9][0-9][0-9])/).exec(time);
        if(caps) {
            result += parseInt(caps[1]) * 60 * 60 * 1000;
            result += parseInt(caps[2]) * 60 * 1000;
            result += parseInt(caps[3]) * 1000;
            result += parseInt(caps[4]);
        } else {
            caps = (/([0-9]+).([0-9]+)s/).exec(time);
            if(caps) {
                result += parseInt(caps[1]) * 1000;
                result += parseInt(caps[2]);
            } else {
                caps = (/([0-9]+)t/).exec(time);
                if(caps) {
                    result = parseInt(caps[1]) / 10000;
                }
            }
        }
    }
    return result;
}

function ttmlLoad(input, scene, processEvents)
{
    var xml;
    var showDebug = nrdp.js_options.debug;
    nrdp.gibbon.load({url: input, secure: true}, function(response) {
        if(response.statusCode != 200) {
            nrdp.log.error("Unable to load: " + JSON.stringify(response));
            return;
        }

        var familyOverride = {
            "ar_Arial": "Arabic_Naskh",
            "fa-ir_Arial": "Arabic_Nastliq"
        };

        function makeRegion() {
            return {
                position: "center",
                displayAlign: "before"
            };
        }

        var styling = {
            default_region: "__default__",
            default_div_style: undefined,
            styles: {},
            regions: { "__default__": makeRegion() }
        };
        styling.regions[styling.default_region].id = styling.default_region;
        styling.default_style = {
            "id": "__default__",
            "opacity": 1.0,
            "displayAlign": "before",
            "direction": "ltr",
            "position": undefined,
            "extent": undefined,
            "origin": undefined,
            "unicodeBidi": "normal",
            "backgroundColor": "transparent",
            "lineHeight": undefined,
            "textDecoration": "none",
            "textOutline": "none",
            "textShadow": "none",
            "showBackground": "always",
            "textAlign": "center",
            "fontFamily": "Arial",
            "fontWeight": "normal",
            "fontShear": "0%",
            "fontSize": "32px",
            "color": "yellow",
            "textEmphasis": undefined,
            "ruby": undefined,
            "rubyPosition": undefined,
            "rubyAlign": undefined,
            "rubyReserve": undefined,
            "textCombine": undefined
        };

        function copyStyle(style, attr) {
            for(var key in attr) {
                var value = attr[key];
                if(key == "writingMode") {
                    key = "direction";
                    if (value == "lrtb" || value == "lr")
                        value = "ltr";
                    else if (value == "rltb" || value == "rl")
                        value = "rtl";
                    else if (value == "tbrl" || value == "tb")
                        value = "ttb";
                    else if (value == "tblr")
                        value = "lttb";
                    else
                        nrdp.log.error("Unhandled writing mode: " + value);
                } else if(key == "fontStyle") {
                    /*if(value == "italic") {
                        key = "fontShear";
                        value = "16%";
                    } else */if(value == "bold") {
                        key = "fontWeight";
                        value = "bold";
                    }
                }
                //nrdp.log.error("SetStyle: " + key + " " + value);
                if(styling.default_style.hasOwnProperty(key))
                    style[key] = value;
                // else
                //     nrdp.log.error("Unhandled style copy: " + key);
            }
        }

        function makeStyle(attr) {
            var result = {};
            if(attr)
                copyStyle(result, attr);
            return result;
        }

        function resolveAttr(attr, styles) {
            var value;
            for(var i = styles.length-1; !value && i >= 0; --i)
                value = styles[i][attr];
            if(typeof value == "string")
                value = value.trim();
            return value;
        }
        function resolveMetric(metric, container, direction) {
            if(!direction)
                direction = "width";
            if(metric !== undefined) {
                if(metric.substr(metric.length-2) == "vh") {
                    return (container.height / 100) * parseFloat(metric.substr(0, metric.length-2));
                } else if(metric.substr(metric.length-2) == "vw") {
                    return (container.width / 100) * parseFloat(metric.substr(0, metric.length-2));
                } else if(metric.substr(metric.length-1) == "%") {
                    return (container[direction] / 100) * parseFloat(metric.substr(0, metric.length-1));
                } else if(metric.substr(metric.length-2) == "em") {
                    return parseFloat(metric.substr(0, metric.length-2)) * resolveMetric(styling.default_style.fontSize, container);
                } else if(metric.substr(metric.length-2) == "px") {
                    return parseFloat(metric.substr(0, metric.length-2));
                }
            }
            return undefined;
        }
        function escapeXML(value) {
            if (typeof (value) == "string") {
                return value.replace(/&/g, '&amp;').
                    replace(/</g, '&lt;').
                    replace(/>/g, '&gt;').
                    replace(/"/g, '&quot;').
                    replace(/'/g, '&apos;');
            }

            return value;
        }
        function resolveFontSize(size, container) {
            if (!size)
                return;

            var result = "";
            sizes = size.split(" ");
            for (var i = 0; i < sizes.length; i++) {
                if (sizes[i] && sizes[i].substr(sizes[i].length-1) == "%") {
                    value = (resolveMetric(styling.default_style.fontSize, container) / 100) * parseFloat(sizes[i].substr(0, sizes[i].length-1));
                } else {
                    value = resolveMetric(sizes[i], container);
                }

                if (result != "")
                    result += " ";

                result += parseInt(value, 10);
            }
            return result;
        }
        function setValue(output, attr, value, styles, region, bounds) {
            //nrdp.log.info("SetValue: " + attr + " " + JSON.stringify(value));
            if(value) {
                if(attr == "textEmphasis") {
                    var args = value.split(" ");
                    var argument = 0;

                    var hasFilled = false;
                    var filled = true;
                    if (args[0] == "open") {
                        hasFilled = true;
                        filled = false;
                    } else if (args[0] == "filled") {
                        hasFilled = true;
                        filled = true;
                    }

                    var hasStyle = false;
                    var style;
                    if (hasFilled)
                        style = "circle";
                    else
                        style = "auto";

                    if (hasFilled) ++argument;

                    if (args[argument] == "circle") {
                        hasStyle = true;
                        style = "circle";
                    } else if (args[argument] == "dot") {
                        hasStyle = true;
                        style = "dot";
                    } else if (args[argument] == "sesame") {
                        hasStyle = true;
                        style = "sesame";
                    }

                    if (filled) {
                        style = "filled " + style;
                    } else {
                        style = "open " + style;
                    }

                    if (hasStyle || (!hasStyle && hasFilled)) {
                        ++argument;
                    }

                    var hasPosition = false;
                    var position = "auto";
                    var lastArgument = args.length - 1;

                    if (args[lastArgument] == "auto") {
                        hasPosition = true;
                        position = "auto";
                    } else if (args[lastArgument] == "before") {
                        hasPosition = true;
                        position = "right";
                    } else if (args[lastArgument] == "after") {
                        hasPosition = true;
                        position = "left";
                    } else if (args[lastArgument] == "outside") {
                        hasPosition = true;
                        position = "outside";
                    }

                    var hasColor = false;
                    var color = "";

                    if (argument < args.length - 1 && (lastArgument != argument || !hasPosition)) {
                        color = args[argument];
                    }

                    if(!output.span)
                        output.span = {};

                    if (hasStyle)
                        output.span.emphasis_style = style;

                    if (hasPosition)
                        output.span.emphasis_position = position;

                    if (hasColor)
                        output.span.emphasis_color = color;

                } else if(attr == "ruby") {
                    if (value == "container") {
                        if (!output.span)
                            output.span = {};
                        output.span.ruby = "container";
                    } else if (value == "text") {
                        if (!output.ruby) {
                            output.ruby = {};
                            output.ruby.position = "before";
                            output.ruby.align = "space_around";
                        }
                    } else if (value != "base") {
                        nrdp.log.info("Unhandled ruby value: '" + value + "'");
                    }

                } else if(attr == "rubyPosition") {
                    if(!output.ruby)
                        output.ruby = {};
                    output.ruby.position = value;

                } else if(attr == "rubyAlign") {
                    if(!output.ruby)
                        output.ruby = {};

                    if (value == "auto")
                        value = "space_around";
                    else if (value == "spaceAround")
                        value = "space_around";
                    else if (value == "spaceBetween")
                        value = "space_between";
                    else if (value == "withBase")
                        value = "center";
                    output.ruby.align = value;

                } else if(attr == "textCombine") {
                    if (!output.span)
                        output.span = {};

                    if (value == "none") {
                        output.span.combine = false;
                    } else if (value == "all") {
                        output.span.combine = true;
                    } else if (value.indexOf("digits") != -1 || !isNaN(value)) {
                        nrdp.log.info("The 'digits' textCombine value is not supported");
                    }

                } else if(attr == "textAlign") {
                    if(value == "start")
                        value = "left";
                    else if(value == "end")
                        value = "right";
                    else if(value == "center")
                        value = "center-horizontal";
                    if(output.text.directStyle[attr] && output.text.directStyle[attr] != value) {
                        nrdp.log.info(attr + " set multiple times: " + value + " vs " + output.text.directStyle[attr]);
                    } else {
                        if(output.text.displayProperties.displayAlign == "after")
                            value = "bottom " + value;
                        else
                            value = "top " + value;
                        output.text.directStyle[attr] = value;
                        output.text.align = value; //outer style
                    }
                } else if(attr == "textShadow") {
                    if(output.text.directStyle[attr] && output.text.directStyle[attr] != value) {
                        nrdp.log.info(attr + " set multiple times: " + value + " vs " + output.text.directStyle[attr]);
                    } else {
                        output.text.directStyle[attr] = value;
                        if(value == "none") {
                            output.text.shadow = undefined;
                        } else {
                            var args = value.split(" ");
                            var offset = resolveMetric(args[0], bounds);
                            if(args.length > 1)
                                offset += resolveMetric(args[1], bounds);
                            var color = output.text.color;
                            if(args.length > 2)
                                color = resolveMetric(args[2], bounds);
                            output.text.shadow = { offsetX: offset, offsetY: offset, outlineColor: color };
                        }
                    }
                } else if(attr == "textOutline") {
                    if(output.text.directStyle[attr] && output.text.directStyle[attr] != value) {
                        nrdp.log.info(attr + " set multiple times: " + value + " vs " + output.text.directStyle[attr]);
                    } else {
                        output.text.directStyle[attr] = value;
                        if(value == "none") {
                            output.text.edgeEffect = undefined;
                        } else {
                            var args = value.split(" ");
                            var width = 1;
                            if(args.length >= 2) {
                                width = resolveMetric(args[1], bounds);
                                if(args.length > 2)
                                    width += resolveMetric(args[2], bounds);
                            }
                            output.text.edgeEffect = { type: "outline", width: width, outlineColor: args[0] };
                        }
                    }
                } else if(attr == "textDecoration") {
                    if(output.text.directStyle[attr] && output.text.directStyle[attr] != value) {
                        if(!output.span)
                            output.span = {};
                        if(value == "underline")
                            output.span.underline = true;
                        else if(value == "none")
                            output.span.underline = false;
                    } else {
                        if(value == "underline") {
                            output.text.directStyle.underline = true;
                            output.text.underline = true;
                        } else if(value == "none") {
                            output.text.directStyle.underline = false;
                            output.text.underline = false;
                        }
                    }
                } else if(attr == "direction") {
                    if(resolveAttr("unicodeBidi", styles) == "bidiOverride") {
                        output.tags.push('bdo direction="' + value + '"');
                    } else if (value == "ttb") {
                        output.text.directStyle[attr] = value;
                    } else  {
                        //if (output.text.directStyle[attr] && output.text.direcStyle[attr] != value) {
                            if (!output.span)
                                output.span = {};
                            output.span.direction = value;
                        //}
                    }
                } else if(attr == "color") {
                    if(output.text.directStyle[attr] && output.text.directStyle[attr] != value) {
                        if(!output.span)
                            output.span = {};
                        output.span.color = value;
                    } else {
                        output.text.directStyle[attr] = value;
                        output.text.color = value;
                    }
                } else if(attr == "backgroundColor") {
                    if(value == "transparent")
                        value = "#00000000";
                    if(output.text.directStyle[attr] && output.text.directStyle[attr] != value) {
                        if(!output.span)
                            output.span = {};
                        output.span.bgcolor = value;
                    } else {
                        output.text.directStyle[attr] = value;
                        output.text.backgroundColor = value;
                    }
                } else if(attr == "fontFamily") {
                    var family = familyOverride[scene.text.locale + "_" + value];
                    if(!family)
                        family = value;
                    if(output.text.directStyle[attr] && output.text.directStyle[attr] != value) {
                        if(!output.span)
                            output.span = {};
                        output.span.font_family = family;
                    } else {
                        output.text.directStyle[attr] = value;
                        output.text.family = family;
                    }
                } else if(attr == "fontWeight") {
                    if(output.text.directStyle[attr] && output.text.directStyle[attr] != value) {
                        if(!output.span)
                            output.span = {};
                        output.span.bold = value == "bold" ? "true" : "false";
                    } else {
                        output.text.directStyle[attr] = value;
                        output.text.bold = value == "bold";
                    }
                } else if(attr == "fontStyle") {
                    if(output.text.directStyle[attr] && output.text.directStyle[attr] != value) {
                        if(!output.span)
                            output.span = {};
                        output.span.italic = value == "italic" ? "true" : "false";
                        if (value == "oblique")
                            output.span.shear = "16%";
                    } else {
                        output.text.directStyle[attr] = value;
                        output.text.italic = value == "italic";
                        if (value == "oblique")
                            output.text.shear = 16;
                    }
                } else if(attr == "fontShear") {
                    if(output.text.directStyle[attr] && output.text.directStyle[attr] != value) {
                        if(!output.span)
                            output.span = {};
                        output.span.shear = value;
                    } else {
                        output.text.directStyle[attr] = parseFloat(value);
                        output.text.shear = parseFloat(value);
                    }
                } else if(attr == "fontSize") {
                    value = resolveFontSize(value, bounds);
                    if(output.text.directStyle[attr] && output.text.directStyle[attr] != value) {
                        if(!output.span)
                            output.span = {};
                        output.span.size = value;
                    } else {
                        values = value.split(" ");
                        if (values.length > 1) {
                            value = { horizontal: parseInt(values[0]), vertical: parseInt(values[1])};
                        } else {
                            value = parseInt(values[0]);
                        }
                        output.text.directStyle[attr] = value;
                        output.text.size = value;
                    }
                } else if(attr == "lineHeight") {
                    value = resolveMetric(value, bounds, "height");
                    if(output.text.directStyle[attr] && output.text.directStyle[attr] != value) {
                        nrdp.log.error(attr + " set multiple times: " + value + " vs " + output.text.directStyle[attr]);
                    } else {
                        output.text.directStyle[attr] = value;
                        output.text.lineHeight = value; //outer style
                    }
                } else if(attr == "position" || attr == "opacity" || attr == "origin" || attr == "extent" || attr == "displayAlign" || attr == "unicodeBidi") { //nothing, handled elsewhere
                } else if(attr == "showBackground") { //not sure what I should do here
                } else {
                    nrdp.log.error("Unknown attribute: " + attr);
                    value = resolveMetric(value.split(" ")[0], bounds);
                    if(!output.span)
                        output.span = {};
                    output.span[attr] = value;
                }
            }
        }
        function makeSpan(text, span, styles, region) {
            nrdp.log.info("**** Span Handle: " + JSON.stringify(span));
            var tag;
            var bounds = { width: nrdp.gibbon.scene.width, height: nrdp.gibbon.scene.height };
            // if(region.extent) {
            //     var extent = region.extent.split(" ");
            //     bounds.width = resolveMetric(extent[0], bounds);
            //     bounds.height = resolveMetric(extent[1], bounds);
            // }

            var output = { text: text, tags: [] };
            text.direction = region.direction;
            for(var property in styling.default_style) {
                if(property != "id")
                    setValue(output, property, resolveAttr(property, styles), styles, region, bounds);
            }

            if(output.ruby && typeof(span) == "string") {
                //nrdp.log.info("Ruby Create: " + JSON.stringify(output.ruby));
                output.text.contents += "<ruby ";
                for(var r in output.ruby)
                    output.text.contents += r + '="' + escapeXML(output.ruby[r]) + '" ';
                output.text.contents += ">";
            }

            if(output.span) {
                //nrdp.log.info("Span Create: " + JSON.stringify(output.span));
                output.text.contents += "<span ";
                for(var s in output.span)
                    output.text.contents += s + '="' + escapeXML(output.span[s]) + '" ';
                output.text.contents += ">";
            }

            var tags = [];
            for(tag = 0; tag < output.tags.length; ++tag) {
                var t = output.tags[tag];
                //nrdp.log.info(tag + ": Tag Create: " +  t);
                output.text.contents += "<" + t + ">";
                tags.push(t.split(' ')[0]);
            }
            if(span) {
                if(span instanceof Object)
                    makeGibbonText(output.text, span, styles, region);
                else
                    output.text.contents += escapeXML(span);
            }

            for(tag = 0; tag < tags.length; ++tag) {
                //nrdp.log.info(tag + ": Tag(" +  tags[tag] + " Done!");
                output.text.contents += "</" + tags[tag] + ">";
            }

            if(output.span) {
                //nrdp.log.info("Span Done!");
                output.text.contents += "</span>";
            }

            if(output.ruby && typeof(span) == "string") {
                output.text.contents += "</ruby>";
                output.ruby = undefined;
            }
        }
        function stylePush(style, styles) {
            if(!style)
                return 0;

            if(typeof style === "string") {
                style = styling.styles[style];
                if(!style)
                    return 0;
            }
            var pushed = 1;
            if(style.style)
                pushed += stylePush(style.style, styles);
            //nrdp.log.info("Style Push: " + JSON.stringify(style));
            styles.push(style);
            return pushed;
        }
        function makeGibbonText(text, span, styles, region) {
            if(span instanceof Object) {
                var attr = span["$attributes"];
                var children = span["$children"];
                for(var i = 0; children && i < children.length; ++i) {
                    var style_pop = 0;
                    var child = children[i];
                    if(child instanceof Object) {
                        var name = child["$name"];
                        if(name == "span") {
                            var attributes = child["$attributes"];
                            if(attributes) {
                                var style_name = attributes.style;
                                if(style_name)
                                    style_pop += stylePush(style_name, styles);
                                style_pop += stylePush(makeStyle(attributes), styles);
                            }
                            makeSpan(text, child, styles, region);
                        } else if(name == "br") {
                            text.contents += "<" + child["$name"] + "/>";
                        } else {
                            nrdp.log.error("Cannot process: " + name);
                        }
                    } else {
                        //text.contents += child;
                        makeSpan(text, child, styles, region);
                    }
                    while(style_pop) {
                        --style_pop;
                        styles.pop();
                        //nrdp.log.info("Style Done!");
                    }
                }
            } else {
                text.contents += escapeXML(span);
            }
        }

        var subtitles = [];
        function parseTTML(path, xml) {
            if(path.length)
                path += ".";
            path += xml["$name"];

            var attr = xml["$attributes"];
            if(path == "tt.head.styling.style") {
                var new_style = styling.styles[attr.id];
                if(!new_style)
                    new_style = styling.styles[attr.id] = makeStyle();
                copyStyle(new_style, attr);
                return;
            } else if(path == "tt.head.styling.initial") {
                copyStyle(styling.default_style, attr);
                return;
            } else if(path == "tt.head.layout.region") {
                var new_region = styling.regions[attr.id];
                if(!new_region)
                    new_region = styling.regions[attr.id] = makeRegion();
                copyStyle(new_region, attr);
                return;
            } else if(path == "tt.body.div.p") {
                var region_id = attr["region"];
                if(!region_id)
                    region_id = styling.default_region;
                var region = styling.regions[region_id];
                var styles = [];
                var id = "subtitle_" + subtitles.length;
                stylePush(styling.default_style, styles);
                stylePush("__body__", styles); //body
                if(region)
                    stylePush(region, styles);
                if(styling.default_div_style)
                    stylePush(styling.default_div_style, styles);
                if(attr) {
                    stylePush(makeStyle(attr), styles); //inline
                    if(attr.style)
                        stylePush(attr.style, styles);
                    if(attr.id)
                        id = attr.id;
                }

                var text = { directStyle: {}, contents: "", padding: { horizontal: undefined, vertical: 0 }, displayProperties: {}  };
                function initTextValue(property) {
                    var value = resolveAttr(property, styles);
                    if(!value && region)
                        value = region[property];
                    text.displayProperties[property] = value;
                }
                initTextValue("opacity");
                initTextValue("displayAlign");
                initTextValue("position");
                initTextValue("extent");
                initTextValue("origin");
                makeGibbonText(text, xml, styles, region);

                if(text.contents.length) {
                    if (!text.padding.horizontal) {
                        text.padding.horizontal = Math.floor((((text.size * 72.0) / 96) / 6) * 1.5);
                    }

                    var subtitle = {id: id, region: region, p: xml, text: text, begin: ttmlParseTime(xml["$attributes"].begin), end: ttmlParseTime(xml["$attributes"].end)};
                    //nrdp.log.info("Subtitle: " + JSON.stringify(subtitle));
                    subtitles.push(subtitle);
                }
                return;
            } else if(path == "tt.body.div") {
                if(attr) {
                    if(attr.style)
                        styling.default_div_style = attr.style;
                    if(attr.region)
                        styling.default_region = attr.region;
                }
            } else if(path == "tt") {
                if(attr && attr.lang && attr.lang.length)
                    scene.text.locale = attr.lang;
            } else if(path == "tt.body") {
                if(attr) {
                    var body_style = {};
                    copyStyle(body_style, attr);
                    styling.styles["__body__"] = body_style;
                    if(attr.region)
                        styling.default_region = attr.region;
                }
            }

            var children = xml["$children"];
            for(var i = 0; children && i < children.length; ++i)
                parseTTML(path, children[i]);
        }
        parseTTML("", nrdp_platform.parseXML(response.data, true)); //avoid __fixXML;

        function makeRegionWidget(subtitle) {
            var widget;
            if(!widget) {
                widget = nrdp.gibbon.makeWidget({parent: scene, name: subtitle.region.id, x: 0, y: 0 });
                if(showDebug)
                    widget.backgroundColor = "#FF0000";
                if(subtitle.text.displayProperties.extent) {
                    var extents = subtitle.text.displayProperties.extent.split(" ");
                    widget.width = resolveMetric(extents[0], scene);
                    widget.height = resolveMetric(extents[1], scene, "height");
                }

                var padding = { left:0, right:0, top:0, bottom:0 };
                var positions = subtitle.text.displayProperties.position.split(" ");
                var position_offset = 0;

                var h_align = positions[position_offset++];
                if(position_offset < positions.length) {
                    var h_edge = resolveMetric(positions[position_offset], scene);
                    if(h_edge !== undefined) {
                        padding[h_align] = h_edge;
                        ++position_offset;
                    }
                }

                var v_align = "center";
                if(position_offset < positions.length) {
                    v_align = positions[position_offset++];
                    if(position_offset < positions.length) {
                        var v_edge = resolveMetric(positions[position_offset], scene, "height");
                        if(v_edge !== undefined) {
                            padding[v_align] = v_edge;
                            ++position_offset;
                        }
                    }
                }

                var displayAlign = subtitle.text.displayProperties.displayAlign;
                if(displayAlign == "before")
                    displayAlign = "top";
                if(displayAlign == "after")
                    displayAlign = "bottom";

                if(subtitle.text.displayProperties.origin && subtitle.text.displayProperties.origin != "auto") {
                    var origins = subtitle.text.displayProperties.origin.split(" ");
                    widget.x += resolveMetric(origins[0], scene);
                    widget.y += origins.length >= 2 ? resolveMetric(origins[1], scene, "height") : widget.x;
                } else {
                    if(h_align == "right")
                        widget.x = scene.width - widget.width;
                    else if(h_align == "center")
                        widget.x = (scene.width-widget.width) / 2;
                    if(displayAlign == "bottom")
                        widget.y = scene.height - widget.height;
                    else if(displayAlign == "center")
                        widget.y = (scene.height-widget.height) / 2;
                    widget.x -= padding.left + padding.right;
                    widget.y -= padding.top + padding.bottom;
                }

                widget.layout = { layout: "v", align: displayAlign };

            }
            return widget;
        }

        function makeSubtitleWidget(subtitle) {
            //nrdp.log.info("Showing: " + JSON.stringify(subtitle));
            var widget = subtitle.widget;
            if(!subtitle.widget) {
                widget = subtitle.widget = nrdp.gibbon.makeWidget({sendRenderProperties: true, name: subtitle.id});
                widget.container = widget;
                widget.opacity = subtitle.text.displayProperties.opacity;
                widget.subtitle = subtitle;
                widget.text = subtitle.text;
                widget.text.richText = true;
                widget.text.wrap = true;
                if(subtitle.region) {
                    var region = makeRegionWidget(subtitle);
                    widget.container = region;
                    widget.parent = region;
                    if(showDebug)
                        widget.parent.backgroundColor = { a: 127, r: 0, g: 0, b: 0 };
                }
            }
            return widget;
        }

        var events = {
            setEventVisible: function setEventVisible(event, v) {
                var that = this;
                for(var i = 0; i < event.subtitles.length; ++i) {
                    var subtitle = event.subtitles[i];
                    var widget = makeSubtitleWidget(subtitle);
                    nrdp.log.info((v ? " + Show" : " - Hide") + ": " + subtitle.id);
                    widget.container.visible = v;
                    if(widget.text.direction == "ltr" || widget.text.direction == "rtl")
                        widget.text.info(function(textInfo) { that.processSubtitleTextInfo(subtitle, textInfo); });
                    else
                        widget.text.padding = 0;
                }
            },
            processSubtitleTextInfo: function processSubtitleTextInfo(subtitle, textInfo) {
                if(subtitle.widget.text.padding == -1) {
                    var padding = 0;
                    for(var i = 0; i < textInfo.layout.lines.length; ++i) {
                        var p = Math.floor((((textInfo.layout.lines[i].bounds.height * 72.0)/96)/6)*1.5);
                        if(p > padding)
                            padding = p;
                    }
                    subtitle.widget.text.padding = { horizontal: padding, vertical: 0 };
                    subtitle.widget.width += padding * 2;
                }
            },
            showEvent: function showEvent(event) { this.setEventVisible(event, true); },
            hideEvent: function hideEvent(event) { this.setEventVisible(event, false); },
            events: []
        };
        var time = subtitles[0].begin;
        while(true) {
            var next_time = undefined;
            var event = { id: "event" + events.events.length, begin: time, subtitles: [] };
            for(var i = 0; i < subtitles.length; ++i) {
                var subtitle = subtitles[i];
                if(subtitle.begin <= time) {
                    if(subtitle.end > time) {
                        if(event.end === undefined || subtitle.end < event.end)
                            event.end = subtitle.end;
                        event.subtitles.push(subtitle);
                    }
                } else if(next_time === undefined || subtitle.begin < next_time) {
                    next_time = subtitle.begin;
                    if(next_time < event.end)
                        event.end = next_time;
                }
            }
            if(!event.subtitles.length)
                break;
            if(event.subtitles.length == 1 && event.subtitles[0].id)
                event.id = "event_" + event.subtitles[0].id;
            nrdp.log.info("Event(" + event.id + "): " + (event.begin/1000) + "s->" + (event.end/1000) + "s: " + event.subtitles.length);
            events.events.push(event);
            time = next_time === undefined ? event.end : next_time;
        }
        processEvents(events);
    });
}
