-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local Viewport = plugins.views:lazyRequire('views.infiniteList.view.Viewport')
local Signal = plugins.lua:lazyRequire('signals.Signal')

local POSITION_START = 'start'
local POSITION_END = 'end'

local ACTION_ADD = 'add'
local ACTION_REMOVE = 'remove'

local ViewportViewsManager = class(function(
		self, viewPool, viewport, managedView)
	if viewPool == nil then
		error('The viewPool argument is required')
	elseif viewport == nil then
		error('The viewport argument is required')
	elseif managedView == nil then
		error('The managedView argument is required')
	end

	self._viewPool = viewPool
	self._viewport = viewport
	self._managedView = managedView
	self._containerNode = self._managedView:getRootNode()

	self._haveDoneFirstPass = false
	self._startIndex = nil
	self._endIndex = nil
	self._viewportViews = {}
	self._viewsWaitingForData = {}

	self.viewAddedToViewport = Signal.new()
	self.viewRemovedFromViewport = Signal.new()
end)

function ViewportViewsManager:getViewport()
	return self._viewport
end

function ViewportViewsManager:getViewPool()
	return self._viewPool
end

function ViewportViewsManager:addViewWaitingForData(view)
	if view then
		self._viewsWaitingForData[view:getUuid()] = true
	end
end

function ViewportViewsManager:removeViewWaitingForData(view)
	if view then
		self._viewsWaitingForData[view:getUuid()] = nil
	end
end

function ViewportViewsManager:isViewWaitingForData(view)
	if view then
		return (self._viewsWaitingForData[view:getUuid()] == true)
	end
	return false
end

function ViewportViewsManager:isViewAvailable(viewIndex)
	local view = self:getView(viewIndex)

	if view and (not self:isViewWaitingForData(view)) then
		return true
	end

	return false
end

function ViewportViewsManager:getView(viewIndex)
	for i=1, #self._viewportViews do
		if self._viewportViews[i].indexWithinDataSet == viewIndex then
			return self._viewportViews[i]
		end
	end

	return nil
end

function ViewportViewsManager:getHighlightedView()
	return self:getView(self:getViewport():getHighlightedIndex())
end

function ViewportViewsManager:getIndexOfFirstView()
	return self._viewportViews[1].indexWithinDataSet
end

function ViewportViewsManager:getIndexOfLastView()
	return self._viewportViews[#self._viewportViews].indexWithinDataSet
end

function ViewportViewsManager:calculateTargetViewPositionsForNextUpdateCycle()
	local delta = self:_getViewportDelta()

	if delta ~= 0 then
		if delta > 0 then
			self:_flagViewsToRemove(POSITION_START, delta)
			self:_flagViewsToAdd(POSITION_END, delta)
		elseif delta < 0 then
			self:_flagViewsToRemove(POSITION_END, -delta)
			self:_flagViewsToAdd(POSITION_START, -delta)
		end

		self:_updateStartAndEndIndices()
	end
end

function ViewportViewsManager:_getViewportDelta()
	-- If we haven't done the first render yet then the delta is the movement
	-- required to encompass the full size of the viewport, so that the whole
	-- thing gets rendered.
	if self._haveDoneFirstPass == false then
		return self._viewport:getSize()
	else
		return self._viewport:getStartIndex() - self._startIndex
	end
end

function ViewportViewsManager:_flagViewsToRemove(removeFrom, numToRemove)
	local indexToRemoveFrom

	if removeFrom == POSITION_START then
		indexToRemoveFrom = 1
	elseif removeFrom == POSITION_END then
		indexToRemoveFrom = self:getNumVisibleViews() - numToRemove + 1
	end

	local numRemoved = 0

	while numRemoved < numToRemove do
		local view = self._viewportViews[indexToRemoveFrom]

		if view ~= nil then
			context:getSceneNodeOpsQueue():queueOp(
				self._containerNode:getSceneNode(),
				"removeChild",
				view:getRootNode():getSceneNode())

			self._viewPool:returnView(view)
			table.remove(self._viewportViews, indexToRemoveFrom)

			-- Note: add 1 to indexToRemoveFrom because the first element in the
			-- childElements array is reserved to the root (ContainerDirective)
			local indexInChildren = indexToRemoveFrom+1

			-- Shift all children left
			for i=indexInChildren+1, self._managedView:getNumChildElements() do
				local nextChild = self._managedView:getChildElement(i)
				self._managedView:setChildElement(i-1, nextChild)
			end
			self._managedView:setChildElement(
						self._managedView:getNumChildElements(), nil)

			context:getNodesRegistry():remove(view:getRootNode():getSceneNode():getId())
			self:removeViewWaitingForData(view)

			self.viewRemovedFromViewport:dispatch(view)
		end

		numRemoved = numRemoved + 1
	end
end

function ViewportViewsManager:_flagViewsToAdd(addTo, numToAdd)
	local indexToAddAt

	if addTo == POSITION_START then
		indexToAddAt = 1
	elseif addTo == POSITION_END then
		indexToAddAt = self:getNumVisibleViews() + 1
	end

	local numAdded = 0
	while numAdded < numToAdd do
		local view = self._viewPool:acquireView()
		table.insert(self._viewportViews, indexToAddAt, view)

		-- Note: add 1 to indexToAddAt because the first element in the
		-- childElements array is reserved to the root (ContainerDirective)
		local indexInChildren = indexToAddAt + 1

		-- Shift all children right
		for i=indexInChildren, self._managedView:getNumChildElements() do
			local prevChild = self._managedView:getChildElement(i)
			self._managedView:setChildElement(i+1, prevChild)
		end
		self._managedView:setChildElement(indexInChildren, view)

		context:getNodesRegistry():add(view:getRootNode():getSceneNode():getId(), view)

		-- Work out the index of the view in the over all data set.
		view.indexWithinDataSet = self._viewport:getStartIndex() + indexToAddAt - 1

		self.viewAddedToViewport:dispatch(view)

		indexToAddAt = indexToAddAt + 1
		numAdded = numAdded + 1
	end
end

function ViewportViewsManager:_updateStartAndEndIndices()
	self._startIndex = self._viewport:getStartIndex()
	self._endIndex = self._viewport:getEndIndex()

	self._haveDoneFirstPass = true
end

function ViewportViewsManager:provideDataToViews(chunk, dataBuffer)
	if self:_chunkFallsWithinViewportRange(chunk) then
		local chunkStartIndex = chunk:getStartIndex()
		local chunkEndIndex = chunk:getEndIndex()
		local chunkToViewportOffset = self._startIndex - chunkStartIndex

		for index = chunkStartIndex, chunkEndIndex do
			local indexWithinChunk = index - chunkStartIndex + 1
			local thisDatum = chunk:getData()[indexWithinChunk]

			local indexWithinViewport = indexWithinChunk - chunkToViewportOffset
			local thisView = self:getViewAtIndex(indexWithinViewport)

			-- Break if we've reached the end of the views array, i.e. the chunk
			-- data runs past the end of the currently visible views.
			if indexWithinViewport > #self._viewportViews then
				break
			-- Check that we've entered into the range of currently visible views
			-- before attempting to access the view, i.e. in the case where the
			-- start index of the chunk is before the start of the visible views
			-- array.
			elseif indexWithinViewport > 0 then
				self:removeViewWaitingForData(thisView)
				thisView:updateModel(thisDatum)
			end
		end
	else
		self:_chunkForViewsInWaitingState(chunk, dataBuffer)
	end
end

function ViewportViewsManager:_chunkForViewsInWaitingState(chunk, dataBuffer)
	for i = 1, #self._viewportViews do
		local view = self:getViewAtIndex(i)
		if view.relativeIndexWithinDataSet ~= nil then
			if view.relativeIndexWithinDataSet >= chunk:getStartIndex()
					and view.relativeIndexWithinDataSet <= chunk:getEndIndex() then
				self:removeViewWaitingForData(view)
				local thisDatum = dataBuffer:getItemByIndex(view.relativeIndexWithinDataSet)
				view:updateModel(thisDatum)
			end
		end
	end
end

function ViewportViewsManager:_chunkFallsWithinViewportRange(chunk)
	local chunkStart = chunk:getStartIndex()
	local chunkEnd = chunk:getEndIndex()
	return chunkStart and chunkStart <= self._endIndex
		and chunkEnd and chunkEnd >= self._startIndex
end

local FORWARDS = Viewport.DIRECTION_FORWARDS
local BACKWARDS = Viewport.DIRECTION_BACKWARDS

function ViewportViewsManager:updateViewPositionsWithinParentContainer()
	for index = 1, #self._viewportViews do
		local view = self._viewportViews[index]

		context:getSceneNodeOpsQueue():queueOp(
			self._containerNode:getSceneNode(),
			"addChild",
			view:getRootNode():getSceneNode())
	end
end

function ViewportViewsManager:getViewAtIndex(index)
	return self._viewportViews[index]
end

function ViewportViewsManager:getNumVisibleViews()
	return #self._viewportViews
end

function ViewportViewsManager:getVisibleViews()
	return self._viewportViews
end

function ViewportViewsManager:dispose()
	for i, view in ipairs(self._viewportViews) do
		self._viewPool:returnView(view)

		context:getSceneNodeOpsQueue():queueOp(
			self._containerNode:getSceneNode(),
			"removeChild",
			view:getRootNode():getSceneNode())
	end

	self.viewAddedToViewport:dispose()
	self.viewRemovedFromViewport:dispose()
end

return ViewportViewsManager
