-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local ViewPool = class(function(self, size, viewDefinition, mediationGroup)
	if type(size) ~= 'number' then
		error('The size argument is required, and must be a number')
	elseif viewDefinition == nil or viewDefinition.isViewDefinition ~= true then
		error('The viewDefinition argument is required, and must be a View class')
	end

	self._size = 0
	self._viewDefinition = viewDefinition
	self._views = {}
	self._mediationGroup = mediationGroup

	self:resize(size)
end)

function ViewPool:resize(newSize)
	local delta = newSize - self._size
	
	if delta > 0 then
		self:_allocateViews(delta)
	elseif delta < 0 then
		self:_deallocateViews(-delta)
	end
	
	self._size = newSize
end

function ViewPool:_allocateViews(count)
	for i = 1, count do
		table.insert(self._views, self:_createView())
	end
end

function ViewPool:_deallocateViews(count)
	for i = 1, count do
		if #self._views > 0 then
			local view = table.remove(self._views)
			view:dispose()
		else
			break
		end
	end
end

function ViewPool:_createView()
	local view = context:getViewFactory():createViewWithDefinition(
		self._viewDefinition,
		self._mediationGroup)

	view:getChildElement(1):setParent(view)
	
	return view
end

function ViewPool:getSize()
	return self._size
end

function ViewPool:getNumAvailableViews()
	return #self._views
end

function ViewPool:acquireView()
	local numViewsInPool = #self._views

	if numViewsInPool > 0 then
		local view = table.remove(self._views, 1)
		return view
	end
end

function ViewPool:returnView(view)
	table.insert(self._views, view)
end

function ViewPool:dispose()
	self:_deallocateViews(#self._views)
end

return ViewPool
