-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local Signal = plugins.views:lazyRequire('signals.Signal')
local DataChunk = plugins.views:lazyRequire('views.infiniteList.data.DataChunk')
local IndexTranslator = plugins.views:lazyRequire('views.IndexTranslator')
local RetainedObjectUtils = plugins.views:lazyRequire('utils.RetainedObjectUtils')

local DataBuffer = class(function(self, chunkSize)
	self._chunkSize = chunkSize
	self._chunks = {}
	self._waitingChunks = {}
	self._startIndex = nil
	self._endIndex = nil
	self._totalDataSetSize = nil

	self.chunkDataRequired = Signal.new()
	self.chunkDataReceived = Signal.new()
	self.chunkDataRemoved = Signal.new()
end)

function DataBuffer:getChunkSize()
	return self._chunkSize
end

function DataBuffer:setIndices(startIndex, endIndex)
	assert(
		(type(startIndex) == "number") and (type(endIndex) == "number"),
		"Start and End indices must be numbers"
	)

	assert(
		startIndex <= endIndex,
		"Start and End indices would give the buffer a negative size"
	)

	self._startIndex = startIndex
	self._endIndex = endIndex

	self:_requestNewChunks()
	self:_removeOldChunks()
end

function DataBuffer:getStartIndex()
	return self._startIndex
end

function DataBuffer:getEndIndex()
	return self._endIndex
end

function DataBuffer:getItemByIndex(itemIndex)
	if self:_dataBufferIsFull() then
		itemIndex = self:_wrapIndexWithSize(itemIndex, self:_getEndIndexOfLastChunk())
	end
	local chunk = self:_getChunkForItemIndex(itemIndex)
	local chunkData
	if chunk then
		chunkData = chunk:getData()
	end
	return chunkData and chunkData[itemIndex - chunk:getStartIndex() + 1]
end

function DataBuffer:_dataBufferIsFull()
	if self._totalDataSetSize ~= nil then
		local lowestStartRequestIndex = self._totalDataSetSize -- guaranteed to be bigger
		local highestEndRequestIndex = 1 -- guaranteed to be smaller
		for i = 1, #self._chunks do
			local chunk = self._chunks[i]
			if chunk:getStartRequestIndex() < lowestStartRequestIndex then
				lowestStartRequestIndex = chunk:getStartRequestIndex()
			end
			if highestEndRequestIndex < chunk:getEndRequestIndex() then
				highestEndRequestIndex = chunk:getEndRequestIndex()
			end
		end
		return 1 == lowestStartRequestIndex and
				self._totalDataSetSize == highestEndRequestIndex
	else
		return false
	end
end

function DataBuffer:_getChunkForItemIndex(itemIndex)
	for i = 1, #self._chunks do
		local chunk = self._chunks[i]
		if itemIndex >= chunk:getStartIndex() and itemIndex <= chunk:getEndIndex()
		then
			return chunk
		end
	end
	return nil
end

function DataBuffer:getCurrentSize()
	return self._endIndex - (self._startIndex - 1)
end

function DataBuffer:getTotalDataSetSize()
	return self._totalDataSetSize
end

function DataBuffer:setTotalDataSetSize(size)
	self._totalDataSetSize = size
end

function DataBuffer:addChunkData(chunk)
	self:_checkChunkIsCorrectSize(chunk)
	if not self:_insertChunkIfOnBoundary(chunk) then
		if not self:_chunkIsAlreadyInDataBuffer(chunk) then
			if self:_chunkSitsWithinIndices(chunk) then
				table.insert(self._waitingChunks, chunk)
			else
				log.error("Chunk outside indices")
			end
		end
	else
		self.chunkDataReceived:dispatch(chunk)
	end
	self:_requestNewChunks()
	self:_removeOldChunks()
end

function DataBuffer:_chunkIsAlreadyInDataBuffer(chunk)
	return self:_indexIsWithinBufferRequestRange(chunk:getStartRequestIndex()) or
			self:_indexIsWithinBufferRequestRange(chunk:getEndRequestIndex())
end

function DataBuffer:getNumQueuedChunks()
	return #self._waitingChunks
end

function DataBuffer:_requestNewChunks()
	if #self._chunks == 0 then
		self:_requestFirstChunk()
	else
		local endChunkIndex = self:_getEndRequestIndexOfLastChunk()
		local startChunkIndex = self:_getStartRequestIndexOfFirstChunk()
		if self._startIndex <= self:_getStartIndexOfFirstChunk() - 1
		then
			local numberOfChunksNeeded = self:_numChunksNeededBetween(
				self._startIndex, self:_getStartIndexOfFirstChunk() - 1)
			self:_requestChunksDown(numberOfChunksNeeded, startChunkIndex - 1)
		elseif self._endIndex >= self:_getEndIndexOfLastChunk() + 1
		then
			local numberOfChunksNeeded = self:_numChunksNeededBetween(
					self:_getEndIndexOfLastChunk() + 1, self._endIndex)
			self:_requestChunksUp(numberOfChunksNeeded, endChunkIndex + 1)
		end
	end
end

function DataBuffer:_indexIsWithinBufferRequestRange(index)
	local indexInBufferRange
	local endChunkIndex = self:_getEndRequestIndexOfLastChunk()
	local startChunkIndex = self:_getStartRequestIndexOfFirstChunk()
	if self:_isLastChunkEndIndexWrapped() then
		indexInBufferRange = index >= startChunkIndex or index <= endChunkIndex
	else
		indexInBufferRange = index <= endChunkIndex and index >= startChunkIndex
	end
	return indexInBufferRange
end

function DataBuffer:_isLastChunkEndIndexWrapped()
	return self:_getEndRequestIndexOfLastChunk() <
			self:_getStartRequestIndexOfFirstChunk()
end

function DataBuffer:_requestFirstChunk()
	-- here we fetch the first chunk at _startIndex and then fetch the rest going up
	local startChunkIndex = self:_getLowerChunkBoundaryForIndex(self._startIndex)
	local numberOfChunksNeeded = self:_numChunksNeededBetween(
			startChunkIndex, self._endIndex)
	self:_requestChunksUp(numberOfChunksNeeded, startChunkIndex)
end

function DataBuffer:_isEndIndexWrapped()
	return self._endIndex < self._startIndex
end

function DataBuffer:_numChunksNeededBetween(startIndex, endIndex)
	startIndex = self:_wrapIndex(startIndex)
	endIndex = self:_wrapIndex(endIndex)
	local numElementsRequired
	if startIndex <= endIndex then
		numElementsRequired = endIndex + 1 - startIndex
	else
		numElementsRequired = (self._totalDataSetSize + 1 - startIndex) +
				(endIndex + 1 - 1)
	end
	return math.ceil(numElementsRequired / self._chunkSize)
end

function DataBuffer:_getLowerChunkBoundaryForIndex(index)
	return (math.floor((index - 1) / self._chunkSize) * self._chunkSize) + 1
end

function DataBuffer:_requestChunksDown(numChunks, index)
	local endIndex = index
	local startIndex = self:_getLowerChunkBoundaryForIndex(self:_wrapIndex(index))
	self:_requestChunksImpl(startIndex, endIndex, numChunks, -1)
end

function DataBuffer:_requestChunksUp(numChunks, index)
	local startIndex = index
	local endIndex = startIndex + (self._chunkSize - 1)
	self:_requestChunksImpl(startIndex, endIndex, numChunks, 1)
end

function DataBuffer:_requestChunksImpl(startIndex, endIndex, numChunks, direction)
	for numTimes = 1, numChunks do
		startIndex = self:_wrapIndex(startIndex)
		endIndex = self:_wrapIndex(endIndex)
		if endIndex < startIndex and self._totalDataSetSize then -- wrapping
			endIndex = self._totalDataSetSize
		end
		local newChunk = DataChunk.new(startIndex, endIndex)
		if not self:_chunkIsAlreadyInDataBuffer(newChunk) then
			self.chunkDataRequired.dispatch(newChunk)
		end
		startIndex = startIndex + direction * self._chunkSize
		endIndex = endIndex + direction * self._chunkSize
	end
end

-- Checks for any chunks which now fall outside the start or end indices of the
-- buffer itself, and removes them.
function DataBuffer:_removeOldChunks()
	local chunksRemoved = {}
	if self._startIndex ~= nil and self._endIndex ~= nil then
		-- iterate backwards because we remove items
		for chunkIndex = #self._chunks, 1, -1 do
			local chunk = self._chunks[chunkIndex]
			if ((chunk:getEndIndex() < self._startIndex) -- doesn't overlap
					or (chunk:getStartIndex() > self._endIndex)) and
				not self:_dataBufferIsFull()
			then
				table.insert(chunksRemoved, self._chunks[chunkIndex])
				self:_removeChunk(chunkIndex)
			end
		end
	end

	if #chunksRemoved > 0 then
		self.chunkDataRemoved.dispatch(chunksRemoved)
	end
end

function DataBuffer:_removeChunk(chunkIndex)
	table.remove(self._chunks, chunkIndex)
end

function DataBuffer:_getFirstChunk()
	return self._chunks[1]
end

function DataBuffer:_getLastChunk()
	return self._chunks[#self._chunks]
end

function DataBuffer:_getStartIndexOfFirstChunk()
	local firstChunk = self:_getFirstChunk()

	if firstChunk ~= nil then
		return firstChunk:getStartIndex()
	else
		return 0
	end
end

function DataBuffer:_getStartRequestIndexOfFirstChunk()
	local firstChunk = self:_getFirstChunk()
	if firstChunk ~= nil then
		return firstChunk:getStartRequestIndex()
	else
		return 0
	end
end

function DataBuffer:_getEndRequestIndexOfLastChunk()
	local lastChunk = self:_getLastChunk()
	if lastChunk ~= nil then
		return lastChunk:getEndRequestIndex()
	else
		return 0
	end
end

function DataBuffer:_getEndIndexOfLastChunk()
	local lastChunk = self:_getLastChunk()
	if lastChunk ~= nil then
		return lastChunk:getEndIndex()
	else
		return 0
	end
end

function DataBuffer:_insertChunkIfOnBoundary(chunk)
	local hasInsertedChunk = false
	if #self._chunks == 0 then
		hasInsertedChunk = self:_insertFirstChunk(chunk)
	end
	if not self:_chunkIsAlreadyInDataBuffer(chunk) then
		if self:_isChunkOnEndBoundary(chunk) then -- insert at end first
			self:_insertChunkAtEnd(chunk)
			hasInsertedChunk = true
		elseif self:_isChunkOnStartBoundary(chunk) then
			self:_insertChunkAtStart(chunk)
			hasInsertedChunk = true
		end
	end
	return hasInsertedChunk
end

function DataBuffer:_isChunkOnStartBoundary(chunk)
	return chunk:getEndRequestIndex() == self:_wrapIndex(
			self:_getStartRequestIndexOfFirstChunk() - 1) 
end

function DataBuffer:_isChunkOnEndBoundary(chunk)
	return chunk:getStartRequestIndex() == self:_wrapIndex(
			self:_getEndRequestIndexOfLastChunk() + 1) 
end

function DataBuffer:_insertFirstChunk(chunk)
	local hasInsertedChunk = false
	if not self:_chunkSitsWithinIndices(chunk) then -- check its in buffer
		hasInsertedChunk = false
	else
		chunk:setStartIndex(chunk:getStartRequestIndex())
		chunk:setEndIndex(chunk:getStartRequestIndex() + chunk:getSize() - 1)
		table.insert(self._chunks, 1, chunk)
		hasInsertedChunk = true
	end
	return hasInsertedChunk
end

function DataBuffer:_chunkSitsWithinIndices(chunk)
	if self:_isEndIndexWrapped() then -- wrapped
		return not (chunk and chunk:getStartRequestIndex() > self._endIndex and
						chunk:getEndRequestIndex() < self._startIndex)
	else --not wrapped
		return not (chunk and chunk:getEndRequestIndex() < self._startIndex or
						chunk:getStartRequestIndex() > self._endIndex)
	end
	
end

function DataBuffer:_insertChunkAtStart(chunk)
	local endIndex = self:_getStartIndexOfFirstChunk() - 1
	chunk:setEndIndex(endIndex)
	chunk:setStartIndex(endIndex - (chunk:getSize() - 1))
	table.insert(self._chunks, 1, chunk)
	self:_insertWaitingChunks()
end

function DataBuffer:_insertChunkAtEnd(chunk)
	local startIndex = self:_getEndIndexOfLastChunk() + 1
	chunk:setStartIndex(startIndex)
	chunk:setEndIndex(startIndex + (chunk:getSize() - 1) )
	table.insert(self._chunks, chunk)
	self:_insertWaitingChunks()
end

function DataBuffer:_insertWaitingChunks()
	local chunksDoneWaiting = {}
	for chunkIndex = 1, #self._waitingChunks do
		local chunk = self._waitingChunks[chunkIndex]
		if self:_insertChunkIfOnBoundary(chunk) then
			table.insert(chunksDoneWaiting, chunk)
			self.chunkDataReceived:dispatch(chunk)
		end
	end
	for doneWaitingIndex = #chunksDoneWaiting, 1, -1 do
		table.remove(self._waitingChunks, doneWaitingIndex)
	end
end

function DataBuffer:_wrapIndex(index)
	return self:_wrapIndexWithSize(index, self._totalDataSetSize)
end

function DataBuffer:_wrapIndexWithSize(index, size)
	return IndexTranslator.getRelativeIndexWithinDataSet(index, size)
end

function DataBuffer:_checkChunkIsCorrectSize(chunk)
	assert(
		chunk:getSize() <= self._chunkSize,
		"Incorrect chunk size: " .. chunk:getSize()
	)
end

function DataBuffer:dispose()
	self.chunkDataRequired:dispose()
	self.chunkDataReceived:dispose()
	self.chunkDataRemoved:dispose()
end

return DataBuffer
