-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local ViewPropertiesRegistry = plugins.views:lazyRequire("core.definitions.ViewPropertiesRegistry")
local ElementView = ViewClass('ElementView')
local TextFieldView = class(ElementView, function(self, handle)
	ElementView._ctor(self, handle)

	self._labelText = ""
end)

TextFieldView.typeName = 'TextFieldView'
TextFieldView.CONTAINER_NODE_INDEX = 1
TextFieldView.BACKGROUND_NODE_INDEX = 2
TextFieldView.PLACEHOLDER_NODE_INDEX = 3
TextFieldView.ACTUAL_TEXT_NODE_INDEX = 4
TextFieldView.CURSOR_TEXT_NODE_INDEX = 5

TextFieldView.STYLE_TAGS_OPS_INDEX = 'style'
TextFieldView.ACTUAL_TEXT_OPS_INDEX = 'label'
TextFieldView.TEXTFIELD_NAME_OPS_INDEX = 'name'
TextFieldView.PLACEHOLDER_TEXT_OPS_INDEX = 'placeholder'

TextFieldView.propertyDefinitions = ViewPropertiesRegistry.mergePropertyDefinitions(
	ElementView.propertyDefinitions,
	{
		cursorText =
		{
			default = "_",
			onChange = function(textView, oldValue, newValue)
				textView:_setCursorText()
			end
		},
		maskText =
		{
			default = nil,
			onChange = function(textView, oldValue, newValue)
				textView:_setLabelViewText()
			end
		}
	}
)

TextFieldView.elements =
{
	{
		className = 'ContainerNodeDirective',
		type = ElementView.TYPE_DIRECTIVE,
		parent = nil,
		staticProps =
		{
			tags =
			{
			 	'text-field'
			}
		}
	},
	{
		name = 'text-background',
		className = 'ImageNodeDirective',
		type = ElementView.TYPE_DIRECTIVE,
		parent = 1,
		staticProps =
		{
			tags =
			{
				'text-background'
			}
		}
	},
	{
		name = 'text-placeholder',
		className = 'TextNodeDirective',
		type = ElementView.TYPE_DIRECTIVE,
		parent = 1,
		staticProps =
		{
			tags =
			{
				'text-generic',
				'text-placeholder'
			}
		}
	},
	{
		name = 'text-label',
		className = 'TextNodeDirective',
		type = ElementView.TYPE_DIRECTIVE,
		parent = 1,
		staticProps =
		{
			tags =
			{
				'text-generic',
				'text-label'
			}
		}
	},
	{
		name = 'text-cursor',
		className = 'TextNodeDirective',
		type = ElementView.TYPE_DIRECTIVE,
		parent = 1,
		staticProps =
		{
			tags =
			{
				'text-generic',
				'text-cursor'
			}
		}
	}
}

function TextFieldView:getChildSceneNode(id)
	local childElement = self:getChildElementOrDie(id)
	return childElement:getRootNode():getSceneNode()
end

function TextFieldView:_initElements()
	self:_setCursorText()
	self:getChildSceneNode(TextFieldView.PLACEHOLDER_NODE_INDEX):setText('')
	self:getChildSceneNode(TextFieldView.ACTUAL_TEXT_NODE_INDEX):setText('')
end

function TextFieldView:_setPlaceholderVisibility(placeholderVisible)
	local textVisible = not placeholderVisible

	local placeholderNode = self:getChildSceneNode(TextFieldView.PLACEHOLDER_NODE_INDEX)
	local textNode = self:getChildSceneNode(TextFieldView.ACTUAL_TEXT_NODE_INDEX)
	local cursorNode = self:getChildSceneNode(TextFieldView.CURSOR_TEXT_NODE_INDEX)
	placeholderNode:setVisible(placeholderVisible)
	textNode:setVisible(textVisible)
	cursorNode:setVisible(textVisible)
end

function TextFieldView:_setCursorText()
	local cursorText = self:getProperty('cursorText')
	self:getChildSceneNode(TextFieldView.CURSOR_TEXT_NODE_INDEX):setText(cursorText)
end

function TextFieldView:_setLabelViewText()
	local maskTextValue = self:getProperty('maskText')
	local viewText

	if (maskTextValue == nil) then
		viewText = self._labelText
	elseif (maskTextValue == "") then
		viewText = ""
	else
		viewText = string.rep(maskTextValue, #self._labelText)
	end

	self:getChildSceneNode(TextFieldView.ACTUAL_TEXT_NODE_INDEX):setText(viewText)
end

function TextFieldView:createResources()
	ElementView.createResources(self)
	self:_initElements()
end

function TextFieldView:update(ops)
	ElementView.update(self, ops)

	local placeholderTextOps = ops:getLocal(TextFieldView.PLACEHOLDER_TEXT_OPS_INDEX)
	local textFieldNameOps = ops:getLocal(TextFieldView.TEXTFIELD_NAME_OPS_INDEX)
	local textFieldStyleOps = ops:getLocal(TextFieldView.STYLE_TAGS_OPS_INDEX)
	local viewTextOps = ops:getLocal(TextFieldView.ACTUAL_TEXT_OPS_INDEX)

	if placeholderTextOps ~= nil then
		local placeholderText = placeholderTextOps.value
		local placeholderNode = self:getChildSceneNode(TextFieldView.PLACEHOLDER_NODE_INDEX)
		placeholderNode:setText(placeholderText)
	end

	if textFieldNameOps ~= nil then
		local container = self:getChildSceneNode(TextFieldView.CONTAINER_NODE_INDEX)
		container:addTag(textFieldNameOps.value)
		container:setName(textFieldNameOps.value)
	end

	if textFieldStyleOps ~= nil then
		local container = self:getChildSceneNode(TextFieldView.CONTAINER_NODE_INDEX)
		container:addTag(textFieldStyleOps.value)
	end

	if viewTextOps ~= nil then
		self._labelText = viewTextOps.value
		self:_setLabelViewText()
	end

	local isPlaceholderVisible = (string.len(self._labelText) == 0)
	self:_setPlaceholderVisibility(isPlaceholderVisible)
end

return TextFieldView
