-- Copyright 2013 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local IView = plugins.views:lazyRequire('core.types.IView')
local ElementView = ViewClass('ElementView')
local RoutingView = class(ElementView)

RoutingView.typeName = 'RoutingView'

RoutingView.routing = {}

function RoutingView:preflight(ops)
	ElementView.preflight(self, ops)

	OpConversion.opsToAggregates(ops, self.routing)

	self:_eachAggregate(ops, function(index, aggregate)
		local childView = self:getChildElementOrDie(index)
		self:_setViewProperties(childView, aggregate._localOps)
		childView:preflight(aggregate)
	end)

	return true
end

function RoutingView:update(ops)
	ElementView.update(self, ops)

	self:_applyProps(ops)
	self:_applyAggregates(ops)

	return true
end

function RoutingView:_applyProps(ops)
	-- Not all Ops containers come from the ModelDiffer (originating in JS).
	-- Internaly, some view types such as CarouselView pass OpsContainers internally.
	-- CarouselView updates describe the currently active index to its children view
	-- an OpsContainer.
	-- We check that the OpsContaiener has an ObjectID before setting _retainedObjectId
	-- so that we don't lose the pairing with the JS Object when an internal
	-- OpsContainer is set.
	if ops:getObjectId() ~= nil then
		self._retainedObjectId = ops:getObjectId()
	end

	local props = OpConversion.opsToProps(ops, self.routing)

	for index, props in pairs(props) do
		local element = self.elements[index]

		-- We only push props to directives. Everything else must go through
		-- an aggregate OpsContainer to a child view.
		if element.type == IView.TYPE_DIRECTIVE then
			self:getChildElement(index):setProperties(props)
		else
			-- TODO (pcowburn) Removed whilst the yAnimate property is still
			-- required in AppDredd. Corresponding test is marked as pending.
			--self:_error('Properties can only be pushed to directives', element)
		end
	end
end

function RoutingView:_applyAggregates(ops)
	self:_eachAggregate(ops, function(index, aggregate)
		local childView = self:getChildElementOrDie(index)
		self:_setViewProperties(childView, aggregate._localOps)
		childView:update(aggregate)
	end)
end

function RoutingView:_eachAggregate(ops, callback)
	local aggregateIndices = ops:getAggregateIndices()

	for i = 1, #aggregateIndices do
		-- Aggregate indices refer to the element index the aggregate goes to.
		local index = aggregateIndices[i]
		local element = self.elements[index]

		if element == nil then
			self:_error("Element was nil at index: " .. index)
		end

		local type = element.type

		-- Only views and view-fragments can have aggregates pushed to them.
		if (type == IView.TYPE_VIEW) or (type == IView.TYPE_VIEW_FRAGMENT) then
			callback(index,	ops:getAggregate(index))
		else
			self:_error(
				'Aggregates can only be pushed to views or view fragments',
				element)
		end
	end
end

function RoutingView:getRetainedObjectId()
	return self._retainedObjectId
end

return RoutingView
